#ifdef WIN32        
#pragma warning(disable:4786)        
#endif

#include <iostream>
#include <fstream>
#include <sstream>
#include <iomanip>
#include <cmath>
#include <algorithm>
#include <vector>
#include <set>
#include <utility>
#include <cassert>
using namespace std;

#include "f2c.h"

#include "sortedEvents.h"
#include "threeSome.h"
#include "rockElem.h"
#include "shape.h"
#include "apex.h"


const double    Apex::PI =                 acos(-1.0);
const double    Apex::EPSILON =            1.0E-10;
const double    Apex::INF_NEG_NUM =        -1.0E21;
const double    Apex::INF_POS_NUM =        1.0E21;
const double    Apex::NEG_ALMOST_ZERO =    -1.0E-12;
const double    Apex::POS_ALMOST_ZERO =    1.0E-12;
const double    Apex::SMALL_NUM =          1.0E-7;
const int       Apex::MAX_ITR =            5000;

/////////////////////////////////////////////////////////////////////////////
// Constructors
/////////////////////////////////////////////////////////////////////////////
CornerApex::CornerApex(double initConAng, Shape* parent) : Apex(parent), m_initConAng(initConAng), 
    m_trappedInside(false, 0.0) 
{
    m_pinnedCalled = false;
    m_initPinningPc = 0.0;
}

/////////////////////////////////////////////////////////////////////////////
// Copy constructors
/////////////////////////////////////////////////////////////////////////////
Apex::Apex(const Apex& apexCp, Shape* parent) : m_parentShape(parent)
{
    m_pinnedApexDist = apexCp.m_pinnedApexDist;
    m_hingingPcRange = apexCp.m_hingingPcRange;
    m_pinned = apexCp.m_pinned;
    m_exists = apexCp.m_exists;
}

CornerApex::CornerApex(const CornerApex& cornerCp, Shape* parent) : Apex(cornerCp, parent), 
    m_initConAng(cornerCp.m_initConAng)
{
    m_trappedInside = cornerCp.m_trappedInside;
    m_initApexDist = cornerCp.m_initApexDist;
    m_initPinningPc = cornerCp.m_initPinningPc;
    m_pinnedCalled = cornerCp.m_pinnedCalled;
}

LayerApex::LayerApex(CornerApex* innerApex, const LayerApex& layerCp, Shape* parent) : Apex(layerCp, parent), 
    m_innerApex(innerApex)
{
    m_collapsePc = layerCp.m_collapsePc; 
    m_trappedOutside = layerCp.m_trappedOutside; 
    m_stable = layerCp.m_stable;
    m_isInCollapseVec = false;
    m_isInReformVec = false;
}

///////////////////////////////////////////////////////////////////////////////////
// Corner Apex Functions
//////////////////////////////////////////////////////////////////////////////////
bool CornerApex::createFilm(double pc, double conAng, double maxSpontConAng, double halfAng, double interfaceTen)
{
    m_exists = m_initConAng < PI/2.0 - halfAng;
    return m_exists;
}

void CornerApex::pinnCorner(double pc, double conAngRec, double conAngAdv, double halfAng, 
                            double interfaceTen, bool didInjOil)
{
    if(!m_exists) return;

    if(m_trappedInside.first) pc = m_trappedInside.second;
    
    if(!m_pinnedCalled || 
        pc > m_initPinningPc || 
        (m_pinned && m_pinnedApexDist == m_initApexDist)) 
    {
        conAngRec = m_initConAng; 
    }
        
    double prevConAng = didInjOil ? conAngRec: conAngAdv;
    
    if(!m_pinned)
    {
        m_pinned = true;
        m_pinnedApexDist = (interfaceTen/pc)*cos(prevConAng+halfAng)/sin(halfAng);
        assure(m_pinnedApexDist > 0.0, "pinnToLarge");
    }
    
    double cosTermAdv = conAngAdv+halfAng > PI ? PI: conAngAdv+halfAng;
    double cosTermRec = conAngRec+halfAng > PI ? PI: conAngRec+halfAng;

    m_hingingPcRange.first = interfaceTen*cos(cosTermAdv)/(m_pinnedApexDist*sin(halfAng));
    m_hingingPcRange.second = interfaceTen*cos(cosTermRec)/(m_pinnedApexDist*sin(halfAng));
    
    if(!m_pinnedCalled || pc > m_initPinningPc) 
    {
        m_initApexDist = m_pinnedApexDist;
        m_initPinningPc = pc;
        m_pinnedCalled = true;
    }
}

bool CornerApex::currentState(double& conAng, double& apexDist, double pc, double halfAng, double interfaceTen,
                              double conAngAdv, bool overRideTrapping)
{
    if(!m_exists) return false;

    if(m_trappedInside.first && !overRideTrapping) pc = m_trappedInside.second;

    if(!m_pinnedCalled || pc > m_initPinningPc+SMALL_NUM) conAng = m_initConAng;    
    double cosTerm = conAng+halfAng > PI ? PI: conAng+halfAng;

    if(!m_pinned) 
    {
        double bi((interfaceTen/pc)*cos(cosTerm)/sin(halfAng));
        if(m_pinnedCalled && (bi < m_initApexDist || conAng > PI/2.0 - halfAng))
        {                                                                               
            m_pinned = true;                            // Interface is over virgin                                       
            if(conAng > PI/2.0 - halfAng && bi < m_initApexDist && pc < 0.0)
            {                                                                                           // Need to get a valid config state when receding con ang
                m_initApexDist = (interfaceTen/(pc-SMALL_NUM))*cos(cosTerm)/sin(halfAng);               // is greater than (PI/2.0-halfAng) => hinging angle needs to
                m_initPinningPc = (interfaceTen/m_initApexDist)*cos(m_initConAng+halfAng)/sin(halfAng); // flip between 0 and receding angle
                assure(m_initApexDist > 0.0 && m_initPinningPc > 0.0, "a");
            }
            
            m_pinnedApexDist = m_initApexDist;  
            m_hingingPcRange.second = m_initPinningPc;  
            m_hingingPcRange.first = interfaceTen*cos(cosTerm)/(m_pinnedApexDist*sin(halfAng));
                        
            assure(m_hingingPcRange.first <= pc, "b"); 
        }
        else
        {
            apexDist = apexDistance(pc, conAng, halfAng, interfaceTen);
            return true;
        }
    }
                                                                    
    if(pc < m_hingingPcRange.first-SMALL_NUM || pc > m_hingingPcRange.second+SMALL_NUM)                           
    {                                                                                   
        m_pinned = false;                                               // Make sure that we're not getting unpinned
        apexDist = apexDistance(pc, conAng, halfAng, interfaceTen);     // just because of rounding errors.
        return true;
    }

    if(pc > m_hingingPcRange.second)        // More ways to prevent rounding errors making our life hell
        pc = m_hingingPcRange.second;
    else if(pc < m_hingingPcRange.first) 
        pc = m_hingingPcRange.first;
  
    conAng = hingingConAng(pc, conAng, halfAng, interfaceTen);;
    apexDist = m_pinnedApexDist;
    return true;
}

///////////////////////////////////////////////////////////////////////////////////
// Layer Apex Functions
//////////////////////////////////////////////////////////////////////////////////
bool LayerApex::createFilm(double pc, double conAng, double maxSpontConAng, double halfAng, double interfaceTen)
{
    m_exists = (conAng > PI/2.0 + halfAng && conAng > maxSpontConAng);
    m_gravityCorrection = m_parentShape->gravityCorrection();
    assure(m_advConAng == conAng, "c");

    if(!m_exists) return false;
    
    if(m_innerApex->exists())
        m_collapsePc = collapsePc_noTrap(pc, conAng, halfAng, interfaceTen);
    else
        m_collapsePc = INF_NEG_NUM;

    m_stable = pc > m_collapsePc;
    
    if(!m_stable) m_stateChangePc = m_collapsePc;

    return m_exists;
}

void LayerApex::pinnLayer(double pc, double conAngRec, double conAngAdv, double halfAng, 
                          double interfaceTen, bool didInjOil)
{
    if(!m_exists) return;

    double prevConAng = didInjOil ? conAngRec: conAngAdv;
    double currConAng = didInjOil ? conAngAdv: conAngRec;

    if(!m_pinned)
    {
        m_pinned = true;
        if(m_stable && m_trappedOutside.first)
            m_pinnedApexDist = (interfaceTen/m_trappedOutside.second)*cos(prevConAng-halfAng)/sin(halfAng);
        else if(m_stable && !m_trappedOutside.first)
            m_pinnedApexDist = (interfaceTen/pc)*cos(prevConAng-halfAng)/sin(halfAng);
        else
            m_pinnedApexDist = (interfaceTen/m_stateChangePc)*cos(prevConAng-halfAng)/sin(halfAng);

        if(m_pinnedApexDist < m_innerApex->pinnedApexDist()) 
        {
            m_stable = false;       // When untrapping water during water inj, the oil might no longer be
            return;                 // able to achieve a stable config => We need to collapse this layer  
        }
    }

    m_hingingPcRange.first = interfaceTen*cos(conAngAdv-halfAng)/(m_pinnedApexDist*sin(halfAng));
    m_hingingPcRange.second = interfaceTen*cos(conAngRec-halfAng)/(m_pinnedApexDist*sin(halfAng));

    m_collapsePc = calcCollapsePc(pc, currConAng, halfAng, interfaceTen, !didInjOil);
}

void CornerApex::pinnBeforeCollapse(double pc, double conAngRec, double conAngAdv, double halfAng, double interfaceTen)
{
    if(m_pinned && (pc < m_hingingPcRange.first || pc > m_hingingPcRange.second))
    {
        m_pinnedApexDist = (interfaceTen/pc)*cos(conAngAdv+halfAng)/sin(halfAng);

        double cosTermAdv = conAngAdv+halfAng > PI ? PI: conAngAdv+halfAng;
        double cosTermRec = conAngRec+halfAng > PI ? PI: conAngRec+halfAng;

        m_hingingPcRange.first = interfaceTen*cos(cosTermAdv)/(m_pinnedApexDist*sin(halfAng));
        m_hingingPcRange.second = interfaceTen*cos(cosTermRec)/(m_pinnedApexDist*sin(halfAng));
    }
}

void LayerApex::pinnBeforeCollapse(double pc, double conAngRec, double conAngAdv, double halfAng, double interfaceTen)
{
    if(m_pinned && (pc < m_hingingPcRange.first || pc > m_hingingPcRange.second))
    {
        m_pinnedApexDist = (interfaceTen/pc)*cos(conAngAdv-halfAng)/sin(halfAng);
        m_hingingPcRange.first = interfaceTen*cos(conAngAdv-halfAng)/(m_pinnedApexDist*sin(halfAng));
        m_hingingPcRange.second = interfaceTen*cos(conAngRec-halfAng)/(m_pinnedApexDist*sin(halfAng));
    }
}

bool LayerApex::currentState(double& conAng, double& apexDist, double pc, double halfAng, double interfaceTen, 
                             double conAngAdv, bool overRideTrapping)
{
    if(!m_stable) return false;

    if(m_trappedOutside.first && !overRideTrapping) pc = m_trappedOutside.second;

    apexDist = apexDistance(pc, conAng, halfAng, interfaceTen);
    assure(apexDist > 0.0, "d");

    if(!m_pinned)
    {
        return true;
    }
    else if(pc < m_hingingPcRange.first-SMALL_NUM || pc > m_hingingPcRange.second+SMALL_NUM)
    {
        m_pinned = false;
        return true;
    }
    else
    {
	    conAng = hingingConAng(pc, conAng, halfAng, interfaceTen);
        return true;
    }
}


