#ifndef APEX_H
#define APEX_H

class Shape;

//////////////////////// BASE CLASS /////////////////////////////
class Apex
{
public:

    Apex(Shape* parent) : m_pinned(false), m_exists(false), m_parentShape(parent) {}
    Apex(const Apex& apexCp, Shape* parent);
    virtual ~Apex() {}
    	
    virtual bool createFilm(double pc, double conAng, double maxSpontConAng, double halfAng, double interfaceTen) = 0;    
    virtual bool currentState(double& conAng, double& b, double pc, double halfAng, double ten, double conAngAdv, bool overRideTrp) = 0;
    virtual void pinnBeforeCollapse(double pc, double conAngRec, double conAngAdv, double halfAng, double interfaceTen) = 0;
    bool exists() const {return m_exists;}
    double pinnedApexDist() const {return m_pinnedApexDist;}
    inline double minHingingPc() const;
    inline double maxHingingPc() const;
    inline bool pinned(double pc) const;
    bool pinned() const {return m_pinned;}
    void checkPinning(double pc);
	
protected:

    inline void assure(bool cond, const string errorMsg = "Shit") const;

    static const double             PI;
    static const double             INF_NEG_NUM;
    static const double             INF_POS_NUM;
    static const double             EPSILON;
    static const double             SMALL_NUM;
    static const double             NEG_ALMOST_ZERO;
    static const double             POS_ALMOST_ZERO;
    static const int                MAX_ITR; 

    const Shape*                    m_parentShape;
    double                          m_pinnedApexDist;
    std::pair<double, double>       m_hingingPcRange;
    bool                            m_pinned;
    bool                            m_exists;
};

////////////////////////// CORNER FILM //////////////////////////////////////
class CornerApex : public Apex
{
public:

    CornerApex(double initConAng, Shape* parent);
    CornerApex(const CornerApex& cornerCp, Shape* parent);
    virtual ~CornerApex() {}

    void pinnCorner(double pc, double recAng, double advAng, double halfAng, double intTen, bool didInjOil);
    inline void removeCorner();
    inline double apexDistance(double pc, double conA, double halfA, double ten) const;
    inline double hingingConAng(double pc, double conA, double halfA, double ten, bool accurat = false) const;
    inline void updateTrapping(const pair< int, double >&, double, double, double, double, double, bool);    
    virtual bool createFilm(double pc, double conAng, double maxSpontConAng, double halfAng, double interfaceTen);
    virtual bool currentState(double& conAng, double& b, double pc, double halfAng, double ten, double conAngAdv, bool overRideTrp);
    virtual void pinnBeforeCollapse(double pc, double conAngRec, double conAngAdv, double halfAng, double interfaceTen);
    const pair<bool, double>& trappedInside() const {return m_trappedInside;}
    bool pinnedInInitState() const {return (m_pinned && m_initApexDist == m_pinnedApexDist);}
    inline void dump(double pc) const;
    void setInitConAng(double conAng) {m_initConAng = conAng;}

private:

    double                          m_initConAng;
    pair<bool, double>              m_trappedInside;
    double                          m_initApexDist;
    double                          m_initPinningPc;
    bool                            m_pinnedCalled;
};

/////////////////////// LAYER FILM /////////////////////////////////////////
class LayerApex : public Apex
{
public:

    LayerApex(CornerApex* innerApex, Shape* parent) : Apex(parent), m_innerApex(innerApex), m_gravityCorrection(0.0),
        m_trappedOutside(false, 0.0), m_advConAng(0.0), m_stable(false), m_stateChangePc(0.0) {}
    LayerApex(CornerApex* apex, const LayerApex& layerCp, Shape* parent);
    virtual ~LayerApex() {} 

    virtual bool createFilm(double pc, double conAng, double maxSpontConAng, double halfAng, double interfaceTen);
    virtual bool currentState(double& conAng, double& b, double pc, double halfAng, double ten, double conAngAdv, bool overRideTrp);
    virtual void pinnBeforeCollapse(double pc, double conAngRec, double conAngAdv, double halfAng, double interfaceTen);

    void pinnLayer(double pc, double conAngRec, double conAngAdv, double halfAng, double intTen, bool didInjOil);
    
    inline double apexDistance(double pc, double conA, double halfA, double ten, bool itr = false) const;
    inline double hingingConAng(double pc, double conA, double halfA, double ten, bool accurat = false) const;
    inline void updateTrapping(const pair< int, double >&, double, double, double, double, double, bool);
    inline double collapsePc_noTrap(double pc, double conAng, double halfAng, double interfaceTen) const;
    const pair<bool, double>& trappedOutside() const {return m_trappedOutside;}
    bool stable() const {return m_stable;}
    inline void stable(bool isIt);
    double gravCorrectedCollapsePc() const {return m_collapsePc+m_gravityCorrection;}
    double collapsePc() const {return m_collapsePc;}
    inline bool stableAtPrs(double pc) const;
    inline bool forcedSnapOff(double prs) const;
    inline void removeLayer();
    void isInCollapseVec(bool isIt) {m_isInCollapseVec = isIt;}
    void isInReformVec(bool isIt) {m_isInReformVec = isIt;}
    bool isInCollapseVec() const {return m_isInCollapseVec;}
    bool isInReformVec() const {return m_isInReformVec;}
    void advConAng(double conAng) {m_advConAng = conAng;}
    inline bool pinnedInLastCycle(double minPcLastCycle) const {return (m_stable && m_pinned && m_hingingPcRange.first == minPcLastCycle);}
      
private:

    inline double collapsePc_inTrap(double outPc, double inPc, double conAng, double halfAng, double ten) const;
    inline double collapsePc_outTrap(double outPc, double inPc, double conAng, double halfAng, double ten) const;
    inline double calcCollapsePc(double pc, double conAng, double halfAng, double interfaceTen, bool injOil) const;
    inline double errorMsg() const;

    CornerApex*                     m_innerApex;
    pair<bool, double>              m_trappedOutside;
    double                          m_advConAng;
    bool                            m_stable;
    double                          m_stateChangePc;
    double                          m_gravityCorrection;
    double                          m_collapsePc;
    bool                            m_isInCollapseVec;
    bool                            m_isInReformVec;
};

/////////////////////////////////////////////////////////////////////////////////////
//////////////////////////  Apex Inline Functions  //////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////

inline void CornerApex::dump(double pc) const
{
    cout << m_pinned << "  " << pc << "  " << m_hingingPcRange.first << "  " << m_hingingPcRange.second;
}

inline bool Apex::pinned(double pc) const
{
    return m_pinned && pc > m_hingingPcRange.first-SMALL_NUM && pc < m_hingingPcRange.second+SMALL_NUM;
}

inline double Apex::minHingingPc() const
{
    return m_hingingPcRange.first;
}

inline double Apex::maxHingingPc() const
{
    return m_hingingPcRange.second;
}

/////////////////////////////////////////////////////////////////////////////////////
///////////////////////  CornerApex Inline Functions  ///////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////
inline double CornerApex::apexDistance(double pc, double conAng, double halfAng, double interfaceTen) const
{
    if(m_pinned && pc > m_hingingPcRange.first-SMALL_NUM && pc < m_hingingPcRange.second+SMALL_NUM) 
        return m_pinnedApexDist;

    double apexDist = (interfaceTen/pc)*cos(conAng+halfAng)/sin(halfAng);
    assure(apexDist > 0.0, "e");
    assure(conAng+halfAng <= PI, "f");
    if(apexDist < 0.0) return POS_ALMOST_ZERO;
    return apexDist;
}

inline double CornerApex::hingingConAng(double pc, double conAng, double halfAng, double interfaceTen, bool accurat) const
{
    double delta = accurat ? 0.0: SMALL_NUM;
    
    if(m_pinned && pc > m_hingingPcRange.first-delta && pc < m_hingingPcRange.second+delta)
    {
        double part(pc*m_pinnedApexDist*sin(halfAng)/interfaceTen);
        part = min(part, 1.0);
        part = max(part, -1.0);
        double hingAng(acos(part)-halfAng);
        assure(hingAng >= -SMALL_NUM && hingAng <= PI+SMALL_NUM, "g");
        double ang(min(max(hingAng, 0.0), PI));
        return ang;
    }

    assure(conAng >= 0.0 && conAng <= PI, "h");
    return conAng;
}

inline void CornerApex::updateTrapping(const pair< int, double >& trpInside, double pc, double conAngRec, 
                                       double conAngAdv, double halfAng, double interfaceTen, bool injOil)
{
    if(!m_exists) return;

    if(trpInside.first > -1 && !m_trappedInside.first)
    {
        m_trappedInside.first = true;
        m_trappedInside.second = trpInside.second;
    }
    else if(trpInside.first == -1)
    {
        m_trappedInside.first = false;                                          // If untrapping at a higher pressure
    }
}

inline void CornerApex::removeCorner()
{
    m_exists = false;
    m_pinned = false;
    m_trappedInside.first = false;
}

/////////////////////////////////////////////////////////////////////////////////////
///////////////////////  LayerApex Inline Functions  ////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////
inline void LayerApex::removeLayer()
{
    m_exists = false;
    m_stable = false;
    m_pinned = false;
    m_trappedOutside.first = false;
}

inline double LayerApex::apexDistance(double pc, double conAng, double halfAng, double interfaceTen, bool itrRoutine) const
{
	if(m_pinned && pc > m_hingingPcRange.first-SMALL_NUM && pc < m_hingingPcRange.second+SMALL_NUM) 
        return m_pinnedApexDist;
	
    double apexDist = (interfaceTen/pc)*cos(conAng-halfAng)/sin(halfAng);
    if(itrRoutine && apexDist <= 0.0) return 1.0E-15;
    assure(apexDist > 0.0, "i");
    return apexDist;
}

inline void LayerApex::updateTrapping(const pair< int, double >& trpOutside, double pc, double conAngRec, double conAngAdv, 
                                      double halfAng, double interfaceTen, bool injOil)
{
    if(!m_exists) return;

    double conAng = injOil ? conAngRec: conAngAdv;

    if(trpOutside.first > -1 && !m_trappedOutside.first)    // Going from untrapped to trapped
    {
        m_trappedOutside.first = true;
        m_trappedOutside.second = trpOutside.second;
    }
    else if(trpOutside.first == -1)                         // Becomming untrapped
    {
        m_trappedOutside.first = false;
    }
    
    m_collapsePc = calcCollapsePc(pc, conAng, halfAng, interfaceTen, injOil);
}

///////////////////////////////////////////////////////////////////
// This really isn't completly correct. Since we don't know when 
// the oil will become coalesced it becomes sligthly difficult.
//////////////////////////////////////////////////////////////////
inline bool LayerApex::stableAtPrs(double pc) const 
{
    if(!m_trappedOutside.first) 
        return m_exists && pc > m_collapsePc;
    else
        return m_exists && m_stable;
}

inline void LayerApex::stable(bool isIt)
{
    m_stable = isIt;
    m_stateChangePc = m_collapsePc;
}

inline double LayerApex::hingingConAng(double pc, double conAng, double halfAng, double interfaceTen, bool accurat) const
{
    double delta = accurat ? 0.0: SMALL_NUM;
    if(m_pinned && pc > m_hingingPcRange.first-delta && pc < m_hingingPcRange.second+delta) 
    {
        double hingAng(acos(pc*m_pinnedApexDist*sin(halfAng)/interfaceTen)+halfAng);
        hingAng = max(min(PI, hingAng), 0.0);
        assure(hingAng >= 0.0 && hingAng <= PI, "j");
        return hingAng;
    }

    assure(conAng >= 0.0 && conAng <= PI, "k");
    return conAng;
}

inline double LayerApex::collapsePc_inTrap(double outPc, double inPc, double conAng, double halfAng, 
                                           double interfaceTen) const
{
    double innerConAng = m_innerApex->hingingConAng(inPc, conAng, halfAng, interfaceTen);
    assure(innerConAng != conAng, "o");
    double ki = (interfaceTen/inPc)*(cos(innerConAng)/sin(halfAng)-1.0);

    if(m_pinned)
    {
        double slipPrs = m_hingingPcRange.first;
        double ko = (interfaceTen/slipPrs)*(cos(m_advConAng)/sin(halfAng)+1.0);
        if(ko - ki < NEG_ALMOST_ZERO)
        {
            double oldRad(interfaceTen/(slipPrs+SMALL_NUM));    // Don't get caught by rounding errors
            for(int itr = 0; itr < MAX_ITR; ++itr)
            {
                double outConAng = acos(m_pinnedApexDist*sin(halfAng)/oldRad)+halfAng;
                double outConAngDr = m_pinnedApexDist*sin(halfAng)/(sin(outConAng-halfAng)*oldRad*oldRad);
                double func = oldRad*(cos(outConAng)/sin(halfAng)+1.0)-ki;
                double funcDr = (cos(outConAng)-oldRad*outConAngDr*sin(outConAng))/sin(halfAng) + 1.0;
                double newRad = oldRad - func/funcDr;
                if(fabs((newRad-oldRad)/newRad) < EPSILON) 
                {
                    double pc(interfaceTen/newRad);
                    assure(fabs(ki-newRad*(cos(outConAng)/sin(halfAng)+1.0)) < EPSILON, "inTrap");
                    return pc;
                }
                oldRad = newRad;
            }
            return errorMsg();
        }
    }

    double collPc = interfaceTen*(cos(m_advConAng)+sin(halfAng))/(ki*sin(halfAng));
    assure(collPc < 0.0, "p");
    return collPc;
}

inline double LayerApex::errorMsg() const
{
    cerr << endl
        << "============================================"   << endl
        << "Error: Failed to converge to a solution for"    << endl
        << "layer collapsing pressure."                     << endl
        << "============================================"   << endl;
    exit(-1);
    return 0.0;
}

inline double LayerApex::calcCollapsePc(double pc, double conAng, double halfAng, double interfaceTen, 
                                        bool injOil) const
{
    double collPrs;

    if(!m_innerApex->exists())
    {
        assure(m_exists, "quacki di quack");
        collPrs = INF_NEG_NUM;                      // No water in corner => never to collapse
    }
    else if(injOil && m_stable)                      // Oil Inj, stable layer => no update needed
        collPrs = INF_NEG_NUM;
    else if(!injOil && !m_stable)               // Wat Inj, collapsed layer => no update needed
        collPrs = INF_POS_NUM;
    else if(!m_innerApex->trappedInside().first && !m_trappedOutside.first) // No trapping 
        collPrs = collapsePc_noTrap(pc, conAng, halfAng, interfaceTen);
    else if(m_innerApex->trappedInside().first && !m_trappedOutside.first)  // Inside trapped
        collPrs = collapsePc_inTrap(pc, m_innerApex->trappedInside().second, conAng, halfAng, interfaceTen);
    else if(!m_innerApex->trappedInside().first && m_trappedOutside.first)  // Outside trapped
        collPrs = collapsePc_outTrap(m_trappedOutside.second, pc, conAng, halfAng, interfaceTen);
    else if(injOil)                 // Oil Inj, both trapped => never reform
        collPrs = INF_POS_NUM;
    else                            // Wat Inj, both trapped => never collapse
       collPrs = INF_NEG_NUM;

    return collPrs;
}

inline bool LayerApex::forcedSnapOff(double prs) const
{
    return prs > m_hingingPcRange.second && prs > 0.0;
}

inline void Apex::assure(bool cond, const string errorMsg) const
{
    return;     // Comment for checking
    if(!cond)
    {
        cerr << "Apex::" << errorMsg << endl;
        //exit(-1);
    }
}

inline double LayerApex::collapsePc_outTrap(double outPc, double inPc, double conAng, double halfAng, 
                                            double interfaceTen) const
{
    double outerConAng = m_pinned ? acos(m_pinnedApexDist*sin(halfAng)*outPc/interfaceTen)+halfAng: conAng;
    double ko = (interfaceTen/outPc)*(cos(outerConAng)/sin(halfAng)+1.0);
    double innerSlipPrs = m_innerApex->minHingingPc();
    double maxConAng = m_advConAng+halfAng > PI ? PI-halfAng: m_advConAng;
    double ki = (interfaceTen/innerSlipPrs)*(cos(maxConAng)/sin(halfAng)-1.0);

    if(ko - ki < NEG_ALMOST_ZERO)
    {
        double oldRad(interfaceTen/(innerSlipPrs+SMALL_NUM));
        double bi = m_innerApex->pinnedApexDist();
        for(int itr = 0; itr < MAX_ITR; ++itr)
        {
            assure((m_innerApex->pinnedApexDist()*sin(halfAng)/oldRad) >= -1.0 && 
                (m_innerApex->pinnedApexDist()*sin(halfAng)/oldRad) <= 1.0, "r");

            double inConAng = acos(bi*sin(halfAng)/oldRad)-halfAng;
            double inConAngDr = bi*sin(halfAng)/(sin(inConAng+halfAng)*oldRad*oldRad);
            double func = oldRad*(cos(inConAng)/sin(halfAng)-1.0)-ko;
            double funcDr = (cos(inConAng)-oldRad*inConAngDr*sin(inConAng))/sin(halfAng) - 1.0;

            double newRad = oldRad - func/funcDr;
            if(fabs((newRad-oldRad)/newRad) < EPSILON)
            {
                double pc(interfaceTen/newRad); 
                assure(fabs(ko-(interfaceTen/pc)*(cos(inConAng)/sin(halfAng)-1.0)) < EPSILON, "outTrap");
                return pc;
            }
            oldRad = newRad;
        }
        return errorMsg();
    }

    return innerSlipPrs;
}

inline double LayerApex::collapsePc_noTrap(double pc, double conAng, double halfAng, double interfaceTen) const
{  
    if(m_pinned)
    {
        double slipPrs = m_hingingPcRange.first;
        assure(slipPrs < 0.0, "l");
        double innerHingConAng = m_innerApex->hingingConAng(slipPrs, conAng, halfAng, interfaceTen);
        double ki = (interfaceTen/slipPrs)*(cos(innerHingConAng)/sin(halfAng)-1.0);
        double ko = (interfaceTen/slipPrs)*(cos(m_advConAng)/sin(halfAng)+1.0);
        
        if(ko - ki < NEG_ALMOST_ZERO)                
        {  
            double oldRad(interfaceTen/(slipPrs+SMALL_NUM));
            double bi = m_innerApex->pinnedApexDist();
            double bo = m_pinnedApexDist;
            for(int itr = 0; itr < MAX_ITR; ++itr)
            {
                double inConAng = acos(bi*sin(halfAng)/oldRad)-halfAng;
                double outConAng = acos(bo*sin(halfAng)/oldRad)+halfAng;
                double inConAngDr = bi*sin(halfAng)/(sin(inConAng+halfAng)*oldRad*oldRad);
                double outConAngDr = bo*sin(halfAng)/(sin(outConAng-halfAng)*oldRad*oldRad);
                
                double func = (cos(inConAng)-cos(outConAng))/sin(halfAng)-2.0;
                double funcDr = (outConAngDr*sin(outConAng)-inConAngDr*sin(inConAng))/sin(halfAng);
                double newRad = oldRad - func/funcDr;
                if(fabs((newRad-oldRad)/newRad) < EPSILON) 
                {
                    assure(fabs(newRad*(cos(innerHingConAng)/sin(halfAng)-1.0) - 
                        newRad*(cos(m_advConAng)/sin(halfAng)+1.0)) < EPSILON, "noTrap");
                    assure(interfaceTen/newRad < 0.0, "m");
                    return interfaceTen/newRad;
                }
                oldRad = newRad;
            }
            return errorMsg();
        }
    }
    
    double innerConAng = acos(2.0*sin(halfAng)+cos(m_advConAng));
    double innerApexDist = m_innerApex->pinnedApexDist();  
    double collPc = interfaceTen*cos(innerConAng+halfAng)/(innerApexDist*sin(halfAng));
    assure(collPc < 0.0, "n");
    
    return collPc;
}


#endif
