#ifndef COMMONDATA_H
#define COMMONDATA_H

class RockElem;
class InputData;

#include "fluid.h"

enum TrappingCriteria {escapeToInlet = 0, escapeToOutlet, escapeToEither, escapeToBoth};
enum FluidBlob {filmBlob = 0, bulkBlob};

class CommonData
{
public:

    CommonData(vector< double >&, const string&, double, InputData& input, double, double, double);
    CommonData(const CommonData& data); 
    ~CommonData(){}

    void finalizeCopyConstruct(const CommonData& data, const vector< RockElem* >& elems);

    double radiiWeights(int i) const {return m_radiiWeights[i];}
    const string& poreFillAlg() const {return m_poreFillAlg;}
   
    int numCircles() const {return m_numCircles;}
    int numTriangles() const {return m_numTriangles;}
    int numSquares() const {return m_numSquares;}

    int drainageCycle() const {return m_drainageCycle;}
    int imbibitionCycle() const {return m_imbibitionCycle;}
    void incrementDrainCycle() {++m_drainageCycle;}
    void incrementImbCycle() {++m_imbibitionCycle;}

    void addCircle() {++m_numCircles;}
    void addTriangle() {++m_numTriangles;}
    void addSquare() {++m_numSquares;}
    void drainagePhase(bool isIt) {m_drainagePhase = isIt;}
    bool drainagePhase() const {return m_drainagePhase;}

    double maxDatumPc() const {return m_maxCappPress;}
    double maxDatumPcLastCycle() const {return m_maxPcLastCycle;}
    inline void maxDatumPc(double pc);
    double minDatumPc() const {return m_minCappPress;}
    double minDatumPcLastCycle() const {return m_minPcLastCycle;}
    inline void minDatumPc(double pc);
    void datumCappPress(double pc) {m_cappPress = pc;}
    double datumCappPress() const {return m_cappPress;}
    double gravConstX() const {return m_gravConstX;}
    double gravConstY() const {return m_gravConstY;}
    double gravConstZ() const {return m_gravConstZ;}

    double circWatCondMultFact() const {return m_circWatCondMultFact;}
    
    void injectant(const Fluid* injFluid) {m_injectant = injFluid;}
    const Fluid* injectant() const {return m_injectant;}

    size_t newOilTrappingIndex() const {return m_trappedRegionsOil.size();}
    size_t newWatTrappingIndex() const {return m_trappedRegionsWat.size();}
    inline size_t numTrappedOilRegions() const;
    inline size_t numTrappedOilElems() const;
    inline size_t numTrappedWatRegions() const;
    inline size_t numTrappedWatElems() const;
    inline void removeTrappedOilElem(int idx, RockElem* elem);
    inline void removeTrappeWatElem(int idx, pair<RockElem*, FluidBlob> elem);
    inline void addTrappeWatElem(int idx, pair<RockElem*, FluidBlob> elem);

    void addTrappedRegionOil(const vector< RockElem* >& elems) {m_trappedRegionsOil.push_back(elems);}
    void addTrappedRegionWat(const vector< pair<RockElem*, FluidBlob> >& elems) {m_trappedRegionsWat.push_back(elems);}
    void removeTrappedRegionOil(int region) {m_trappedRegionsOil[region].clear();}
    void removeTrappedRegionWat(int region) {m_trappedRegionsWat[region].clear();}
    
    const vector<RockElem*>& trappedRegionsOil(int region) const {return m_trappedRegionsOil[region];}
    const vector< pair<RockElem*,FluidBlob> >& trappedRegionsWat(int region) const {return m_trappedRegionsWat[region];}

    int numPores() const {return m_numPores;}
    int numThroats() const {return m_numThroats;}
    void setNumElem(int numP, int numT) {m_numPores = numP; m_numThroats = numT;}

    ofstream                        dbgOut;

private:

    void readDBGangles(const string& fileName);               

    const Fluid*                                        m_injectant;
    double										        m_maxCappPress;
    double                                              m_maxPcLastCycle;
	double										        m_minCappPress;
    double                                              m_minPcLastCycle;
    double                                              m_cappPress;
    double										        m_circWatCondMultFact;
    const double                                        m_gravConstX;
    const double                                        m_gravConstY;
    const double                                        m_gravConstZ;
    int									                m_numPores;
    int									                m_numThroats;
    bool                                                m_drainagePhase;
    int											        m_numSquares;
    int											        m_numTriangles;
    int											        m_numCircles;
    int                                                 m_drainageCycle;
    int                                                 m_imbibitionCycle;
    vector< double >							        m_radiiWeights;
    string										        m_poreFillAlg;
    vector< vector<RockElem*> >					        m_trappedRegionsOil;
    vector< vector< pair<RockElem*,FluidBlob> > >	    m_trappedRegionsWat;
};

inline size_t CommonData::numTrappedOilElems() const
{
    size_t numTrapped(0);
    for(size_t i = 0; i < m_trappedRegionsOil.size(); ++i)
        numTrapped += m_trappedRegionsOil[i].size();
    
    return numTrapped;
}

inline size_t CommonData::numTrappedWatElems() const
{
    size_t numTrapped(0); 
    for(size_t i = 0; i < m_trappedRegionsWat.size(); ++i)
        numTrapped += m_trappedRegionsWat[i].size();
    
    return numTrapped;
}

inline size_t CommonData::numTrappedOilRegions() const 
{
    size_t numTrpRegions(0);
    for(size_t i = 0; i < m_trappedRegionsOil.size(); ++i)
    {
        if(!m_trappedRegionsOil[i].empty()) 
            ++numTrpRegions;
    }
    return numTrpRegions;
}

inline size_t CommonData::numTrappedWatRegions() const 
{
    size_t numTrpRegions(0);
    for(size_t i = 0; i < m_trappedRegionsWat.size(); ++i)
    {
        if(!m_trappedRegionsWat[i].empty()) 
            ++numTrpRegions;
    }
    return numTrpRegions;
}

inline void CommonData::maxDatumPc(double pc) 
{
    m_maxCappPress = max(pc, m_maxCappPress);
    m_maxPcLastCycle = pc;
}

inline void CommonData::minDatumPc(double pc) 
{
    m_minCappPress = min(pc, m_minCappPress);
    m_minPcLastCycle = pc;
}

inline void CommonData::removeTrappedOilElem(int idx, RockElem* elem)
{
    m_trappedRegionsOil[idx].erase(
        remove(m_trappedRegionsOil[idx].begin(),m_trappedRegionsOil[idx].end(),elem),
        m_trappedRegionsOil[idx].end());
}

inline void CommonData::removeTrappeWatElem(int idx, pair<RockElem*, FluidBlob> elem)
{
    m_trappedRegionsWat[idx].erase(
        remove(m_trappedRegionsWat[idx].begin(),m_trappedRegionsWat[idx].end(),elem),
        m_trappedRegionsWat[idx].end());
}

inline void CommonData::addTrappeWatElem(int idx, pair<RockElem*, FluidBlob> elem)
{
    m_trappedRegionsWat[idx].push_back(elem);
}


#endif

