#ifndef INPUTDATA_H
#define INPUTDATA_H

typedef struct
{
    int         index;
    double      x;
    double      y;
    double      z;
    int         connNum;
    double      volume;
    double      radius;
    double      shapeFact;
    double      clayVol;
} PoreStruct;

typedef struct
{
    int     index;
    int     poreOne;
    int     poreTwo;
    double  radius;
    double  shapeFact;
    double  lenPoreOne;
    double  lenPoreTwo;
    double  lenThroat;
    double  lenTot;
    double  volume;
    double  clayVol;
} ThroatStruct;

class InputData
{
public:

    InputData(){}
    InputData(const string&);

    void addData(const string& keyword, const string& data); 
    void title(string& baseFileName);
    void randSeed(int& seedNum);
    bool satTarget(double& sw, double& pc, double& dSw, double& dPc, bool& kr, bool& I);
    void prsBdrs(bool& usePrsBdr, bool& reportPrsBdr, int& numPlanes);
    void solverTune(double& eps, int& scaleFact, int& slvrOutput, bool& verbose, double& condCutOff);
    void poreFillWgt(vector< double >& weights);
    void poreFillAlg(string& algorithm);
    void relPermDef(string& flowRef, bool& strictTrpCond);
    void relPermCompression(bool& useComp, double& krThres, double& deltaSw, bool& wettPhase, bool& nonWettPhase);
    void satConvergence(int& minNumFillings, double& initStepSize, double& cutBack, double& maxIncrFact, 
        bool& stable);
    void trapping(bool& fromEntry, bool& fromExit, bool& drainEnds, double& gMult);
    void fillingList(bool& drainage, bool& imbibition, bool& location);
    void output(bool& propOut, bool& swOut);
    void resFormat(bool& matlabFormat, bool& excelFormat);
    void fluid(double& intFaceTen, double& watVisc, double& oilVisc, double& watRes, double& oilRes, 
        double& watDens, double& oilDens);
    void clayEdit(double& clayEditFact);
    void aCloseShave(double& shaveSetting);
    void apexPrs(bool& doApexAnalysis);
    void gravityConst(double& gravX, double& gravY, double& gravZ);
    void calcBox(double& inletBdr, double& outletBdr);
    void prsDiff(double& inletPrs, double& outletPrs, bool& useGravInKr);
    void network(int& numPores, int& numThroats, double& xDim, double& yDim, double& zDim);
    void poreData(int idx, double& x, double& y, double& z, int& n, vector< int >& throats, vector< int >& pores, 
        double& c, double& vcl, double& e, double& g);
    void throatData(int idx, int& p1, int& p2, double& v, double& vcl, double& r, double& g, double& lp1, 
        double& lp2, double& lt, double& lTot);
    void fracWetting(bool& applyFracWet, double& whatFrac, bool& volBased, double& min, 
        double& max, double& delta, double& eta, string& fracModel, int& clustDiam, bool& oilInWat);
    void initConAng(double& min, double& max, double& delta, double& eta);
    void equilConAng(int& wettClass, double& min, double& max, double& delta, double& eta, string& mod, double& sep);
    void writeNetwork(bool& writeNet, bool& binary, string& netName);
    void solverDebug(bool& watMat, bool& oilMat, bool& resMat, bool& watVel, bool& oilVel, bool& resVel, 
        string& fileName, bool& matlab, bool& initOnly);
    void modifyRadDist(int& throatModel, int& poreModel, string& throatOpt, string& poreOpt, 
        bool& maintainLtoR, bool& toFile, int& numPts);
    void modifyPoro(double& netPoroTrgt, double& clayPoroTrgt);
    void modifyConnNum(double& targetConnNum, string& model);
    void sourceNode(int& sourceNode);
    void matBal(bool& reportMatBal);
    void modifyGDist(int& throatModel, int& poreModel, string& throatOpt, string& poreOpt, 
        bool& toFile, int& numPts);
    void modifyModelSize(double& scaleFactor);
    void finishedLoadingNetwork();
    inline void echoKeywords(ostream& out) const;
    void poreLocation(int idx, double& xPos) const;
    
private:

    typedef multimap< int, ThreeSome<int, int, double> >::value_type MultiConnValType;
    typedef multimap< int, ThreeSome< int, int, double > >::iterator MapItr;

    void removeComments(string& data) const; 
    bool terminatorFound(string& data) const; 
    void getRadDist(istream& data, int& model, string& options) const;
    void getOutletData(MapItr itr, int numNetsInFront, int& throatIdx, int& thatIdx) const;
    void getInletData(MapItr itr, int numNetsInFront, int& throatIdx, int& thatIdx) const;
    int findClosestPore(const vector< ThreeSome< PoreStruct*, int*, int* > > pores, double xPos, 
        double yPos, double zPos, double& totalLen) const;        
    inline void errorMsg(const string& keyword) const;
    inline void missingDataErr(const string& keyword) const;
    inline bool getData(istringstream& data, const string& keyword); 
    inline void errorInDataCheck(istringstream& data, const string& keyword) const;
    inline bool outletThroat(int throatIdx) const;
    inline bool inletThroat(int throatIdx) const;
    inline void appendPoreData(int thisPoreIdx, int throatIdx, int thatPoreIdx);
    void loadPoreData();
    void loadThroatData();
    void findBoundaryPores();
    void findClosestPoreForPBC(const vector< ThreeSome< double, double, double > >& pos, 
        vector< int >& idx, vector< double >& p2BLen);

    static const int                                        DUMMY_INDEX;           
 
    ifstream                                                m_poreConn;
    ifstream                                                m_poreProp;
    ifstream                                                m_throatConn;
    ifstream                                                m_throatProp;
    vector< pair< string, string > >                        m_parsedData;
    string                                                  m_baseFileName;
    int                                                     m_connectionsRemoved;
    int                                                     m_origNumPores;
    int                                                     m_origNumThroats;
    int                                                     m_numNetInSeries;
    int                                                     m_workingSatEntry;
    int                                                     m_numInletThroats;
    bool                                                    m_binaryFiles;
    bool                                                    m_useAvrXOverThroatLen;
    bool                                                    m_useAvrPbcThroatLen;
    bool                                                    m_addPeriodicBC;
    vector< ThreeSome< PoreStruct*, int*, int* > >          m_poreData;
    vector< ThroatStruct* >                                 m_throatData;
    vector< ThreeSome< PoreStruct*, int*, int* > >          m_inletPores;
    vector< ThreeSome< PoreStruct*, int*, int* > >          m_outletPores;
    multimap< int, ThreeSome< int, int, double > >          m_outletConnections;
    multimap< int, ThreeSome< int, int, double > >          m_inletConnections;
    set< pair< int, int > >                                 m_xyPbcConn;
    set< pair< int, int > >                                 m_xzPbcConn;
    vector< ThreeSome< int, int, double > >                 m_pbcData;    
    vector< int >                                           m_throatHash;
    vector< int >                                           m_reverseThroatHash;
    double                                                  m_origXDim;
    double                                                  m_origYDim;
    double                                                  m_origZDim;
    double                                                  m_averageThroatLength;
    double                                                  m_averagePoreHalfLength;
    double                                                  m_networkSeparation;
};

inline bool InputData::outletThroat(int throatIdx) const
{
    ThroatStruct *throat = m_throatData[throatIdx-1];
    return (throat->poreOne == 0 || throat->poreTwo == 0);
}

inline bool InputData::inletThroat(int throatIdx) const
{
    ThroatStruct *throat = m_throatData[throatIdx-1];
    return (throat->poreOne == -1 || throat->poreTwo == -1);
}

//////////////////////////////////////////////////////////////////////
// Returns error message is error occurs during reading of data string
//////////////////////////////////////////////////////////////////////
inline void InputData::errorMsg(const string& keyword) const
{
    cerr << endl
        << "================================"   << endl
        << "Error while reading input file. "   << endl
        << "Keyword: " << keyword               << endl
        << "================================"   << endl;
    exit(-1);
}

//////////////////////////////////////////////////////////
// Returns error message when required keyword is missing
//////////////////////////////////////////////////////////
inline void InputData::missingDataErr(const string& keyword) const
{
    cerr << endl
        << "=============================================="   << endl
        << "Error: " << keyword << " is a required keyword."  << endl
        << "=============================================="   << endl;
    exit(-1);
}

/////////////////////////////////////////////////////////////////////
// Echos the keywords and associated data to the given output stream
/////////////////////////////////////////////////////////////////////
inline void InputData::echoKeywords(ostream& out) const 
{
    for(unsigned i = 0; i < m_parsedData.size(); ++i)
    {
        out << m_parsedData[i].first    << endl
            << m_parsedData[i].second   << endl
            << endl;
    }
}

//////////////////////////////////////////////////////////////////////
// Retrieves data sting based on supplied keyword, and connects a 
// string stream to it. Data is removed from storage after having been
// retrived
//////////////////////////////////////////////////////////////////////
inline bool InputData::getData(istringstream& data, const string& keyword) 
{
    for(unsigned i = 0; i < m_parsedData.size(); ++i) 
    {
        if(m_parsedData[i].first == keyword) 
        {
            data.str(m_parsedData[i].second);
            return true;
        }
    }
    return false;
}

inline void InputData::errorInDataCheck(istringstream& data, const string& keyword) const
{
    char leftOvers;
    data >> leftOvers;
    if(data)
    {
        cerr << endl
            << "===================================================================================="   << endl
            << "Error: Too much data read for keyword: " << keyword                                     << endl
            << "Data read: " << data.str()                                                              << endl
            << "===================================================================================="   << endl;
        exit(-1);
    }
}

inline void InputData::appendPoreData(int thisPoreIdx, int throatIdx, int thatPoreIdx)
{
    int oldConnNum = m_poreData[thisPoreIdx-1].first()->connNum;
    int *newThroatConnList = new int[oldConnNum+1];
    int *newPoreConnList = new int[oldConnNum+1];
    for(int i = 0; i < oldConnNum; ++i)
    {
        newPoreConnList[i] = m_poreData[thisPoreIdx-1].second()[i];
        newThroatConnList[i] = m_poreData[thisPoreIdx-1].third()[i];
    }
    newPoreConnList[oldConnNum] = thatPoreIdx;
    newThroatConnList[oldConnNum] = throatIdx;
    m_poreData[thisPoreIdx-1].first()->connNum++;
    delete[] m_poreData[thisPoreIdx-1].second();
    delete[] m_poreData[thisPoreIdx-1].third();
    m_poreData[thisPoreIdx-1].second(newPoreConnList);
    m_poreData[thisPoreIdx-1].third(newThroatConnList);
}

#endif

