#ifdef WIN32        
#pragma warning(disable:4786)        
#endif

#include <iostream>
#include <fstream>
#include <sstream>
#include <iomanip>
#include <cmath>
#include <cstdlib>
#include <cassert>
#include <ctime>
#include <set>
#include <vector>
#include <algorithm>
#include <functional>
#include <map>
using namespace std;

#include "f2c.h"
#include "sortedEvents.h"
#include "threeSome.h"
#include "inputData.h"
#include "node.h"
#include "fluid.h"
#include "apex.h"
#include "shape.h"
#include "compareFuncs.h"
#include "rockElem.h"
#include "solver.h"


#include "netsim.h"

const double    Netsim::MAX_FLOW_ERR = 0.02;
const int       Netsim::DUMMY_IDX = -99;

/////////////////////////
// Netsim constructor
/////////////////////////
Netsim::Netsim(unsigned seed) : m_randSeed(seed)
{
    srand(m_randSeed);

    m_inletSolverPrs = 1.0;
	m_outletSolverPrs = 0.0;
    m_minCappPress = 0.0;
    m_maxCappPress = 0.0;
    m_satWater = 1.0; 
    m_cappPress = 0.0;
    m_clayAdjust = 0.0;
    m_netVolume = 0.0; 
    m_clayVolume = 0.0; 
    m_relPermWater = 1.0; 
    m_relPermOil = 0.0; 
    m_resistivityIdx = 1.0; 
    m_maxNonZeros = 0; 
    m_maxOilFlowErr = 0.0; 
    m_maxWatFlowErr = 0.0; 
    m_maxResIdxErr = 0.0; 
    m_numIsolatedElems = 0;
    m_modelTwoSepAng = 0.0;
    m_wettingClass = 1;
    m_totNumFillings = 0;
    m_wettingFraction = 0.0;
    m_shavingFraction = 1.0;
    m_currentPc = 0.0;
    m_calcRelPerm = true;
    m_calcResIdx = true;
    m_matlabFormat = false;
    m_excelFormat = false;
    m_stableFilling = true;
    m_injAtEntryRes = false;
    m_injAtExitRes = false;
    m_reportMaterialBal = false;
    m_includeGravityInRelPerm = false;
    m_solver = NULL;
    m_solverBoxStart = 0.0;
    m_boundPress = 0.0;
    m_amottOilIdx = 0.0;
    m_amottWaterIdx = 1.0;
    m_solverBoxEnd = 1.0;
    m_singlePhaseDprs = 1.0;
    m_singlePhaseDvolt = 1.0;
    m_numPressurePlanes = 0;
    m_oilFlowRate = 0.0;
    m_watFlowRate = 0.0;
    m_current = 0.0;
    m_medianLenToRadRatio = 0.0;
    m_writeWatMatrix = false;
    m_writeOilMatrix = false;
    m_writeResMatrix = false;
    m_writeWatVelocity = false;
    m_apexPrsReported = false;
    m_writeOilVelocity = false;
    m_writeResVelocity = false;
    m_writeSlvMatrixAsMatlab = false;
    m_amottDataDrainage.resize(3);
    m_amottDataImbibition.resize(3);
    m_deltaPo = 0.0;
    m_deltaPw = 0.0;
    m_relPermDef = "single";
}

////////////////////////////////////////////////////////////////////////////////////////
// Copy constructor takes a copy of another netsim instance. This might be useful in 
// say simulations where one would use multiple instances of the network model in 
// different blocks. 
////////////////////////////////////////////////////////////////////////////////////////
Netsim::Netsim(const Netsim& netsim) : m_randSeed(netsim.m_randSeed)
{
    m_oil = new Oil(*static_cast< Oil* >(netsim.m_oil));
    m_water = new Water(*static_cast< Water* >(netsim.m_water));
    m_commonData = new CommonData(*netsim.m_commonData);
    
    m_numPores = netsim.m_numPores;
    m_numThroats = netsim.m_numThroats;
    m_injAtEntryRes = netsim.m_injAtEntryRes;
    m_injAtExitRes = netsim.m_injAtExitRes;
    m_angDistScheme = netsim.m_angDistScheme;
    
    int numElem(m_numPores + m_numThroats + 2);
    m_rockLattice.resize(numElem);
    m_rockLattice[0] = new EndPore(m_commonData, m_oil, m_water, *static_cast< EndPore* >(netsim.m_rockLattice[0]));
    m_rockLattice[m_numPores+1] = new EndPore(m_commonData, m_oil, m_water, *static_cast< EndPore* >(netsim.m_rockLattice[m_numPores+1]));
        
    for(int i = 1; i <= m_numPores; ++i)
        m_rockLattice[i] = new Pore(m_commonData, m_oil, m_water, *static_cast< Pore* >(netsim.m_rockLattice[i]));

    for(int j = m_numPores+2; j < numElem; ++j)
        m_rockLattice[j] = new Throat(m_commonData, m_oil, m_water, *static_cast< Throat* >(netsim.m_rockLattice[j]));

    for(int k = 0; k < numElem; ++k)        // RockElements can't be copy constructed directly since they
    {                                       // contain pointers to neighbouring elements. First create all
        vector< RockElem* > connections;    // elemnts, then set up inter connecting pointers
        for(int conn = 0; conn < netsim.m_rockLattice[k]->connectionNum(); ++conn)
        {
            int index = netsim.m_rockLattice[k]->connection(conn)->latticeIndex();
            connections.push_back(m_rockLattice[index]);
        }

        m_rockLattice[k]->finalizeCopyConstruct(connections);
    }

    for(size_t inEl = 0; inEl < netsim.m_krInletBoundary.size(); ++inEl)     
    {
        int index =  netsim.m_krInletBoundary[inEl]->latticeIndex();
        m_krInletBoundary.push_back(m_rockLattice[index]);
    }

    for(size_t outEl = 0; outEl < netsim.m_krOutletBoundary.size(); ++outEl)
    {
        int index =  netsim.m_krOutletBoundary[outEl]->latticeIndex();
        m_krOutletBoundary.push_back(m_rockLattice[index]);
    }

    m_numPressurePlanes = netsim.m_numPressurePlanes;
    m_pressurePlanesLoc = netsim.m_pressurePlanesLoc;
    m_pressurePlanes.resize(m_numPressurePlanes);
    for(int pl = 0; pl < m_numPressurePlanes; ++pl)
    {
        size_t numElems = netsim.m_pressurePlanes[pl].size();
        for(size_t el = 0; el < numElems; ++el)
        {
            int index = netsim.m_pressurePlanes[pl][el]->latticeIndex();
            m_pressurePlanes[pl].push_back(m_rockLattice[index]);
        }
    }

    for(size_t dEl = 0; dEl < m_drainageEvents.size(); ++dEl)
    {
        int index =  netsim.m_drainageEvents.at(dEl)->latticeIndex();
        m_drainageEvents.quickInsert(m_rockLattice[index]);
    }

    for(size_t iEl = 0; iEl < m_imbibitionEvents.size(); ++iEl)
    {
        int index =  netsim.m_imbibitionEvents.at(iEl)->latticeIndex();
        m_imbibitionEvents.quickInsert(m_rockLattice[index]);
    }

    for(size_t cEl = 0; cEl < m_layerCollapseEvents.size(); ++cEl)
    {
        int index =  netsim.m_layerCollapseEvents.at(cEl).first->parent()->latticeIndex();
        Polygon* polyShape = dynamic_cast< Polygon* >(m_rockLattice[index]->shape());
        assert(polyShape);
        pair<Polygon*, int> elem(polyShape, netsim.m_layerCollapseEvents.at(cEl).second);
        m_layerCollapseEvents.quickInsert(elem);
    }

    for(size_t rEl = 0; rEl < m_layerReformEvents.size(); ++rEl)
    {
        int index =  netsim.m_layerReformEvents.at(rEl).first->parent()->latticeIndex();
        Polygon* polyShape = dynamic_cast< Polygon* >(m_rockLattice[index]->shape());
        assert(polyShape);
        pair<Polygon*, int> elem(polyShape, netsim.m_layerReformEvents.at(rEl).second);
        m_layerReformEvents.quickInsert(elem);
    }


    m_maxNonZeros = netsim.m_maxNonZeros;   // Memory for solver is not assigned until we actually need it. Hence
    if(netsim.m_solver != NULL)              // we need to check wheter is has been constructred or not.
    {
        m_solver = new Solver(m_rockLattice, m_krInletBoundary, m_krOutletBoundary, m_numPores+1, m_maxNonZeros, "nothing", true);
    }
    else
        m_solver = NULL;

    m_commonData->finalizeCopyConstruct(*netsim.m_commonData, m_rockLattice);   // Needed to finalize all pointers to trapped elems
            
    m_waterSatHistory = netsim.m_waterSatHistory;
    m_cpuTimeTotal = netsim.m_cpuTimeTotal;
    m_cpuTimeKrw = netsim.m_cpuTimeKrw;
    m_cpuTimeKro = netsim.m_cpuTimeKro;
    m_cpuTimeTrapping = netsim.m_cpuTimeTrapping;
    m_prtOut = netsim.m_prtOut;                     // The various output streams that have been assigned are just
    m_drainResultsOut = netsim.m_drainResultsOut;   // directly copied. Perhaps a bit messy (especially prt output)
    m_imbResultsOut = netsim.m_imbResultsOut;
    m_results = netsim.m_results;
    
    m_numIsolatedElems = netsim.m_numIsolatedElems;
    m_minNumFillings = netsim.m_minNumFillings;
    m_totNumFillings = netsim.m_totNumFillings;
    
    m_apexPrsReported = netsim.m_apexPrsReported;
    m_currentPc = netsim.m_currentPc;
    m_clayAdjust = netsim.m_clayAdjust;
    m_maxOilFlowErr = netsim.m_maxOilFlowErr;
    m_maxWatFlowErr = netsim.m_maxWatFlowErr;
    m_maxResIdxErr = netsim.m_maxResIdxErr;
    m_singlePhaseWaterQ = netsim.m_singlePhaseWaterQ;
    m_singlePhaseOilQ = netsim.m_singlePhaseOilQ;
    m_singlePhaseDprs = netsim.m_singlePhaseDprs;
    m_singlePhaseDvolt = netsim.m_singlePhaseDvolt;
    m_singlePhaseCurrent = netsim.m_singlePhaseCurrent;
    m_formationFactor = netsim.m_formationFactor;
    m_absPermeability = netsim.m_absPermeability;
    m_satBoxStart = netsim.m_satBoxStart;
    m_shavingFraction = netsim.m_shavingFraction;
    m_satBoxEnd = netsim.m_satBoxEnd;
    m_solverBoxStart = netsim.m_solverBoxStart;
    m_solverBoxEnd = netsim.m_solverBoxEnd;
    m_modelTwoSepAng = netsim.m_modelTwoSepAng;
    m_reportMaterialBal = netsim.m_reportMaterialBal;
    m_netVolume = netsim.m_netVolume;
    m_clayVolume = netsim.m_clayVolume;
    m_includeGravityInRelPerm = netsim.m_includeGravityInRelPerm;
    m_rockVolume = netsim.m_rockVolume;
    m_maxCappPress = netsim.m_maxCappPress;
    m_satWater = netsim.m_satWater;
    m_amottWaterIdx = netsim.m_amottWaterIdx;
    m_amottOilIdx = netsim.m_amottOilIdx;
    m_relPermWater = netsim.m_relPermWater;
    m_relPermOil = netsim.m_relPermOil;
    m_resistivityIdx = netsim.m_resistivityIdx;
    m_cappPress = netsim.m_cappPress;
    m_xSize = netsim.m_xSize; 
    m_ySize = netsim.m_ySize; 
    m_zSize = netsim.m_zSize;
    m_wettingClass = netsim.m_wettingClass;
    m_initStepSize = netsim.m_initStepSize;
    m_extrapCutBack = netsim.m_extrapCutBack;
    m_maxFillIncrease = netsim.m_maxFillIncrease;
    m_cpuTimeResIdx = netsim.m_cpuTimeResIdx;
	m_inletSolverPrs = netsim.m_inletSolverPrs;
	m_outletSolverPrs = netsim.m_outletSolverPrs;
    m_wettingFraction = netsim.m_wettingFraction;
    
    m_calcRelPerm = netsim.m_calcRelPerm;
    m_calcResIdx = netsim.m_calcResIdx;
    m_trappingCriteria = netsim.m_trappingCriteria;
    m_oilFlowRate = netsim.m_oilFlowRate;
    m_watFlowRate = netsim.m_watFlowRate;
    m_current = netsim.m_current;
    m_matlabFormat = netsim.m_matlabFormat;
    m_excelFormat = netsim.m_excelFormat;
    m_stableFilling = netsim.m_stableFilling;
    m_medianLenToRadRatio = netsim.m_medianLenToRadRatio;
    m_relPermDef = netsim.m_relPermDef;
    m_writeWatMatrix = false;
    m_writeOilMatrix = false;
    m_writeResMatrix = false;
    m_writeWatVelocity = false;
    m_writeOilVelocity = false;
    m_writeResVelocity = false;
    m_writeSlvMatrixAsMatlab = false;
    m_createDrainList = false;
    m_createImbList = false;
}

////////////////////////
// Destructor
///////////////////////
Netsim::~Netsim()
{
    delete m_oil;
    delete m_water;
    delete m_commonData;

    for(size_t i = 0; i < m_rockLattice.size(); ++i)
        delete m_rockLattice[i];
}

//////////////////////////////////////////////////////////////////////////////////////////////////////
// The network is created and initialized by reading in the connection files that also contains 
// all other relevant information about the pores/throats. Pointers to all elements are contained
// in a vector. Element 0 and (n_pores + 1) are inlet/outlet. Throats follows after the pores.
// Since all rock elements contain pointers to connecting elements rather than vector indicies,
// initialization has to be in correct order: throats, pores, in/outlet and finally finishing the
// throats. A single phase solve is also conducted to have flowrates to scale relperms against.
//////////////////////////////////////////////////////////////////////////////////////////////////////
void Netsim::init(InputData& input)
{
    double interfacTen, watVisc, oilVisc, watResist, oilResist, circWatCondMultFact, condCutOff(0.0); 
    double minEqConAng(0.0), maxEqConAng(0.0), watDens(1000.0), oilDens(1000.0);
    double gravX(0.0), gravY(0.0), gravZ(-9.81), wettDelta, wettEta, eps;
    int scaleFact, slvrOutput;
    vector< double > fillWeights;
    vector< string > solverOptions;
    string poreFillAlgoritm, modPoroOptions;
    bool createLocData(false), writeMatInitOnly, drainSinglets(true), verboseSlvr, strictTrpCond(true); 

    input.calcBox(m_satBoxStart, m_satBoxEnd);
    input.network(m_numPores, m_numThroats, m_xSize, m_ySize, m_zSize);            
    input.fluid(interfacTen, watVisc, oilVisc, watResist, oilResist, watDens, oilDens);
    input.satConvergence(m_minNumFillings, m_initStepSize, m_extrapCutBack, m_maxFillIncrease, m_stableFilling);
    input.poreFillWgt(fillWeights);
    input.solverTune(eps, scaleFact, slvrOutput, verboseSlvr, condCutOff);
    input.clayEdit(m_clayAdjust);
    input.resFormat(m_matlabFormat, m_excelFormat);
    input.poreFillAlg(poreFillAlgoritm);
    input.prsBdrs(m_useAvrPrsAsBdr, m_prtPressureProfile, m_numPressurePlanes);
    input.matBal(m_reportMaterialBal);
    input.trapping(m_injAtEntryRes, m_injAtExitRes, drainSinglets, circWatCondMultFact);
    input.apexPrs(m_apexPrsReported);
    input.gravityConst(gravX, gravY, gravZ);
    input.relPermDef(m_relPermDef, strictTrpCond);
    input.aCloseShave(m_shavingFraction);
                   
    if(m_injAtEntryRes && m_injAtExitRes)
    {
        m_trappingCriteria = escapeToEither;
    }
    else if(strictTrpCond)
    {
        m_trappingCriteria = escapeToBoth;
    }
    else if(m_injAtEntryRes)
    {
        m_trappingCriteria = escapeToOutlet;
    }
    else
    {
        m_trappingCriteria = escapeToInlet;
    }

    input.solverDebug(m_writeWatMatrix, m_writeOilMatrix, m_writeResMatrix, m_writeWatVelocity, m_writeOilVelocity, 
		m_writeResVelocity, m_matrixFileName, m_writeSlvMatrixAsMatlab, writeMatInitOnly);
    input.fillingList(m_createDrainList, m_createImbList, createLocData);
	input.prsDiff(m_inletSolverPrs, m_outletSolverPrs, m_includeGravityInRelPerm);
    m_deltaPw = m_deltaPo = m_inletSolverPrs-m_outletSolverPrs;
    input.equilConAng(m_wettingClass, minEqConAng, maxEqConAng, wettDelta, wettEta, m_angDistScheme, m_modelTwoSepAng);
    input.sourceNode(m_sourceNode);
    if(m_sourceNode != 0)
    {
        if(m_sourceNode > m_numPores)
        {
            cerr << "=============================================="                << endl
                << "Source pore (" << m_sourceNode << ") needs to be"               << endl
                << "less than the total number of pores (" << m_numPores << ")."    << endl
                << "=============================================="                 << endl;
            exit(-1);
        }
        m_trappingCriteria = escapeToBoth;
        m_injAtEntryRes = false;
        m_injAtExitRes = false;
    }

    Solver::initSolver(eps, scaleFact, slvrOutput, verboseSlvr, m_includeGravityInRelPerm);
    RockElem::useGravInKr(m_includeGravityInRelPerm);
    RockElem::conductanceCutOff(condCutOff);

    m_commonData = new CommonData(fillWeights, poreFillAlgoritm, circWatCondMultFact, input, gravX, gravY, gravZ);
    m_water = new Water(watVisc, interfacTen, watResist, watDens);
    m_oil = new Oil(oilVisc, interfacTen, oilResist, oilDens);

    if(!m_useAvrPrsAsBdr)                           // If we don't average pressures to obtain                                  
    {                                               // rel perm (ie we're moving the boundary)        
        m_solverBoxStart = m_satBoxStart;           // then the lattice we pass to the solver 
        m_solverBoxEnd = m_satBoxEnd;               // will be the same as that we compute sat across
    }

    int numSingletsRemoved = initNetwork(input, drainSinglets);
    m_commonData->setNumElem(m_numPores, m_numThroats);
    modifyNetwork(input);

    vector< RockElem* > pores2Set(m_rockLattice.begin(), m_rockLattice.begin()+2+m_numPores);
    vector< RockElem* > throats2Set(m_rockLattice.begin()+2+m_numPores, m_rockLattice.end());
    setContactAngles(pores2Set, throats2Set, minEqConAng, maxEqConAng, wettDelta, wettEta);
    checkStateOfInitConAng();
    initFlow(input, writeMatInitOnly, createLocData, drainSinglets);

    ostringstream out;
	out << endl << endl
        <<  "Number of pores:                           " << m_numPores                                     << endl
        <<  "Number of throats:                         " << m_numThroats                                   << endl
        <<  "Average connection number:                 " << (double)(m_maxNonZeros-m_numPores)/m_numPores  << endl
        <<  "Number of connections to inlet:            " << m_rockLattice[0]->connectionNum()              << endl
        <<  "Number of connections to outlet:           " << m_rockLattice[m_numPores+1]->connectionNum()   << endl
        <<  "Number of physically isolated elements:    " << m_numIsolatedElems                             << endl
        <<  "Number of singlets removed:                " << numSingletsRemoved                             << endl
        <<  "Number of triangular shaped elements:      " << m_commonData->numTriangles()                   << endl
        <<  "Number of square shaped elements:          " << m_commonData->numSquares()                     << endl
        <<  "Number of circular shaped elements:        " << m_commonData->numCircles()                     << endl
        <<  "Median throat length to radius ratio:      " << m_medianLenToRadRatio                          << endl        
        <<  "Net porosity:                              " << m_netVolume / m_rockVolume                     << endl
        <<  "Clay bound porosity:                       " << m_clayVolume / m_rockVolume                    << endl
        <<  "Absolute permeability (mD)                 " << m_absPermeability                              << endl
        <<  "Absolute permeability (m2)                 " << m_absPermeability * 9.869233E-16               << endl
        <<  "Formation factor:                          " << m_formationFactor                              << endl
        << endl;

    writePrtData(out);
}

////////////////////////////////////////////////////////////////////////////////////
// Creates all the pores and throats
////////////////////////////////////////////////////////////////////////////////////
int Netsim::initNetwork(InputData& input, bool drainSinglets)
{
    vector< pair< int, double > > poreHash(m_numPores);
    vector< int > throatHash(m_numThroats, DUMMY_IDX);
    int newNumPores = setupPoreHashing(input, poreHash);
        
    m_rockLattice.resize(newNumPores + 2);                
    vector< pair<int, int> > connectingPores(m_numThroats);
    vector< RockElem* > throatsToInlet;                                 // All throats connected to the in/outlet are
    vector< RockElem* > throatsToOutlet;                                // recorded while crating the throats
    
    int newNumThroats = readAndCreateThroats(input, connectingPores, throatsToInlet, throatsToOutlet, 
        poreHash, throatHash, newNumPores);    
    readAndCreatePores(input, connectingPores, poreHash, throatHash, newNumPores);
    input.finishedLoadingNetwork();

    m_xSize *= m_shavingFraction;
    m_numPores = newNumPores;
    m_rockVolume = m_xSize * m_ySize * m_zSize * (m_satBoxEnd - m_satBoxStart);

    double yMid(m_ySize/2.0), zMid(m_zSize/2.0);
    createInAndOutletPore(0, -1.0E-15, yMid, zMid, throatsToInlet);
    createInAndOutletPore(m_numPores + 1, m_xSize+1.0E-15, yMid, zMid, throatsToOutlet);
        
    vector< double > lenToRadRatio(newNumThroats);
    int runIdx(0);
    for(int i = 0; i < m_numThroats; ++i)                               // Adding the pore pointers to the throats had
    {                                                                   // to be delayed until after having created 
        int poreIndex1 = checkPoreIndex(connectingPores[i].first);      // the pores. The network should now be
        int poreIndex2 = checkPoreIndex(connectingPores[i].second);     // properly initialized.
        
        if(poreIndex1 >= 0 && poreIndex2 >= 0)
        {
            assert(poreIndex1 < m_numPores+2 && poreIndex2 < m_numPores+2);
            RockElem* pore1 = m_rockLattice[poreIndex1];       
            RockElem* pore2 = m_rockLattice[poreIndex2];     
            assert(pore1 != NULL && pore2 != NULL);

            m_rockLattice[m_numPores + 2 + runIdx]->addConnections(pore1, pore2, m_xSize*m_solverBoxStart, 
                m_xSize*m_solverBoxEnd, !m_useAvrPrsAsBdr);
            lenToRadRatio[runIdx] = m_rockLattice[m_numPores + 2 + runIdx]->lenToRadRatio();
            ++runIdx;
        }
    }
    m_numThroats = newNumThroats;
    sort(lenToRadRatio.begin(), lenToRadRatio.end()); 
    m_medianLenToRadRatio = lenToRadRatio[m_numThroats/2];

    removeConnectionsFromNetwork(input);                                // Reduce connection number    

    m_rockLattice[0]->identifyConnectedRockElems();                     // Possibly remove singlets
    int numSingletsRemoved(0);
    
    if(!drainSinglets) 
    {
        for(int pr = 1; pr <= m_numPores; ++pr)
        {
            if(m_rockLattice[pr]->connectedToNetwork() && m_rockLattice[pr]->connectionNum() == 1)
                numSingletsRemoved += m_rockLattice[pr]->removeFromNetwork();
        }
    }
    
    for(size_t elem = 0; elem < m_rockLattice.size(); ++elem)
    {
        assert(m_rockLattice[elem] != NULL);
        m_rockLattice[elem]->checkNetworkIntegrity(m_netVolume, m_clayVolume, m_maxNonZeros, m_numIsolatedElems);
        if(m_rockLattice[elem]->connectedToNetwork())
        {
            if(m_rockLattice[elem]->isOnInletSlvrBdr()) m_krInletBoundary.push_back(m_rockLattice[elem]);
            if(m_rockLattice[elem]->isOnOutletSlvrBdr()) m_krOutletBoundary.push_back(m_rockLattice[elem]);
            m_rockLattice[elem]->sortConnectingElems(); 
        }
    }
    
    if(RockElem::errorState())
    {
        cerr << "===================================="  << endl
            << "Cannot recover from errors. Exiting"    << endl
            << "===================================="   << endl;
        exit(-1);
    }
    
    if(m_useAvrPrsAsBdr && m_numPressurePlanes < 2) m_numPressurePlanes = 2;
    m_pressurePlanes.resize(m_numPressurePlanes);

    for(int p = 0; p < m_numPressurePlanes; ++p)
    {
        double loc = m_satBoxStart + p * (m_satBoxEnd - m_satBoxStart) / (m_numPressurePlanes - 1);
        m_pressurePlanesLoc.push_back(loc);
        for(size_t t = m_numPores + 2; t < m_rockLattice.size(); ++t)
        {
            if(m_rockLattice[t]->crossesPlaneAt(loc*m_xSize) && m_rockLattice[t]->connectedToNetwork())
                m_pressurePlanes[p].push_back(m_rockLattice[t]);
        }
    }
    return numSingletsRemoved;
}

void Netsim::removeConnectionsFromNetwork(InputData& input)
{
    string modelToChangeCN;
    double targetConnNum(-1.0);
    int totNumConn(0);
    
    input.modifyConnNum(targetConnNum, modelToChangeCN);

    for(int i = 1; i <= m_numPores; ++i) 
        totNumConn += m_rockLattice[i]->connectionNum();

    if(targetConnNum <= 0.0) 
        return;
    else if(targetConnNum > static_cast< double >(totNumConn)/m_numPores)
    {
        ostringstream out;
        out << endl
            << "==========================================================="    << endl
            << "Warning: The requested connection number is higher than the"    << endl
            << "original. Connections can only be removed, not added."          << endl
            << "==========================================================="    << endl
            << endl;
        writePrtData(out);
    }

    vector< RockElem* > throats(m_rockLattice.begin()+2+m_numPores, m_rockLattice.end());
    
    if(modelToChangeCN[1] == 'o' || modelToChangeCN[1] == 'O')      // Net Volume 'volume'
        sort(throats.begin(), throats.end(), ElemVolCmpRed());
    else if(modelToChangeCN[1] == 'h' || modelToChangeCN[1] == 'H') // Shape Factor 'shape'
        sort(throats.begin(), throats.end(), ElemGCmpRed());
    else if(modelToChangeCN[2] == 'n' || modelToChangeCN[2] == 'N') // Random 'rand'
        sort(throats.begin(), throats.end(), ElemGCmpRed());
    else                                                            // Radius 'radius'
        sort(throats.begin(), throats.end(), ElemRadCmpRed());

    int numToRemove(static_cast< int >((totNumConn-targetConnNum*m_numPores)/2));
    while(numToRemove > 0 && !throats.empty())
    {
        RockElem *damned = throats.back();
        throats.pop_back();
        
        if(!damned->connectedToEntryOrExit())
        {
            RockElem *poreOne = damned->connection(0); 
            RockElem *poreTwo = damned->connection(1);

            poreOne->severConnection(damned);
            poreTwo->severConnection(damned);

            assert(m_rockLattice[damned->latticeIndex()] == damned);
            m_rockLattice[damned->latticeIndex()] = 0;
            delete damned;
            --numToRemove;
            --m_numThroats;
        }
    }
    RockElem *ghost = NULL; 
    m_rockLattice.erase(remove(m_rockLattice.begin()+m_numPores+2, m_rockLattice.end(), ghost),
        m_rockLattice.end());
    assert(static_cast< int >(m_rockLattice.size()) == m_numPores+m_numThroats+2);
    
    for(int newTIdx = m_numPores+2; newTIdx < static_cast< int >(m_rockLattice.size()); ++newTIdx)
        m_rockLattice[newTIdx]->updateLatticeIndex(newTIdx);
}

void Netsim::modifyNetwork(InputData& input)
{
    int throatRadModel(0), poreRadModel(0), throatGModel(0), poreGModel(0);
    int numPtsGDist(0), numPtsRDist(0);
    string throatRadOptions, poreRadOptions, throatGOptions, poreGOptions;
    double netPoroTrgt(-1.0), clayPoroTrgt(-1.0), scaleFactor(-1.0);
    bool writeRDistToFile(false), writeGDistToFile(false), maintainLtoR(false);
    
    input.modifyRadDist(throatRadModel, poreRadModel, throatRadOptions, poreRadOptions, 
        maintainLtoR, writeRDistToFile, numPtsRDist);
    input.modifyPoro(netPoroTrgt, clayPoroTrgt);
    input.modifyGDist(throatGModel, poreGModel, throatGOptions, poreGOptions, writeGDistToFile, 
        numPtsGDist);
    input.modifyModelSize(scaleFactor);
    
    vector< RockElem* > pores(m_rockLattice.begin()+1, m_rockLattice.begin()+1+m_numPores);
    vector< RockElem* > throats(m_rockLattice.begin()+2+m_numPores, m_rockLattice.end());

    if(poreGModel == -1 || throatGModel == -1)
    {
        string fileName("ShapeFactDist.csv");
        vector< RockElem* > elems(m_rockLattice.begin()+1, m_rockLattice.begin()+1+m_numPores);
        elems.insert(elems.end(), m_rockLattice.begin()+2+m_numPores, m_rockLattice.end());
        if(throatGModel == -1)
            modifyShapeFactor(poreGModel, poreGOptions, elems, fileName, writeGDistToFile, numPtsGDist);
        else
            modifyShapeFactor(throatGModel, throatGOptions, elems, fileName, writeGDistToFile, numPtsGDist);
    }
    else
    {
        string fileNamePores("ShapeFactDist_pores.csv"), fileNameThroats("ShapeFactDist_throats.csv");
        modifyShapeFactor(throatGModel, throatGOptions, throats, fileNameThroats, writeGDistToFile, numPtsGDist);
        modifyShapeFactor(poreGModel, poreGOptions, pores, fileNamePores, writeGDistToFile, numPtsGDist);
    }
        
    if(scaleFactor > 0.0)
    {
        for(size_t elm = 0; elm < m_rockLattice.size(); ++elm)
        {
            double oldRad(m_rockLattice[elm]->shape()->radius());
            m_rockLattice[elm]->shape()->setRadius(oldRad*scaleFactor);
        }
        poreRadModel = 0;
        throatRadModel = 0;
    }
    else if(poreRadModel == -1 || throatRadModel == -1)
    {
        string fileName("RadDist.csv");
        vector< RockElem* > elems(m_rockLattice.begin()+1, m_rockLattice.begin()+1+m_numPores);
        elems.insert(elems.end(), m_rockLattice.begin()+2+m_numPores, m_rockLattice.end());
        if(throatRadModel == -1)
            modifyInscribedRadii(poreRadModel, poreRadOptions, elems, fileName, writeRDistToFile, numPtsRDist);
        else
            modifyInscribedRadii(throatRadModel, throatRadOptions, elems, fileName, writeRDistToFile, numPtsRDist);
    }
    else if(throatRadModel != 2)
    {
        string fileNamePores("RadDist_pores.csv"), fileNameThroats("RadDist_throats.csv");
        modifyInscribedRadii(throatRadModel, throatRadOptions, throats, fileNameThroats, writeRDistToFile, numPtsRDist);
        modifyInscribedRadii(poreRadModel, poreRadOptions, pores, fileNamePores, writeRDistToFile, numPtsRDist, true);
    }
    else
    {
        string fileNamePores("RadDist_pores.csv"), fileNameThroats("RadDist_throats.csv");
        modifyInscribedRadii(poreRadModel, poreRadOptions, pores, fileNamePores, writeRDistToFile, numPtsRDist, true);
        modifyInscribedRadii(throatRadModel, throatRadOptions, throats, fileNameThroats, writeRDistToFile, numPtsRDist);
    }
    sort(throats.begin(), throats.end(), lenToRadRatioCmp());
    double oldmMedianLenToRadRatio(m_medianLenToRadRatio);
    m_medianLenToRadRatio = throats[throats.size()/2]->lenToRadRatio();
    if(poreRadModel && maintainLtoR) scaleFactor = oldmMedianLenToRadRatio/m_medianLenToRadRatio;  
      
    if(scaleFactor > 0.0)
    {
        for(int i = 0; i < m_numPores + 2; ++i)
            m_rockLattice[i]->node()->rePosition(scaleFactor);

        for(size_t j = m_numPores + 2; j < m_rockLattice.size(); ++j)
            m_rockLattice[j]->modifyLength(scaleFactor);

        if(netPoroTrgt < 0) netPoroTrgt = m_netVolume/m_rockVolume;
        if(clayPoroTrgt < 0) clayPoroTrgt = m_clayVolume/m_rockVolume;
        m_xSize *= scaleFactor;
        m_ySize *= scaleFactor;
        m_zSize *= scaleFactor;
        m_rockVolume = m_xSize*m_ySize*m_zSize*(m_satBoxEnd-m_satBoxStart);
        m_medianLenToRadRatio = throats[throats.size()/2]->lenToRadRatio();
    }

    double targetNetVol = netPoroTrgt >= 0.0 ? m_rockVolume*netPoroTrgt: m_netVolume;
    double targetClayVol = clayPoroTrgt >= 0.0 ? m_rockVolume*clayPoroTrgt: m_clayVolume;
    
    if(netPoroTrgt >= 0.0 || clayPoroTrgt >= 0.0)
    {
        double changeNetFrac((targetNetVol-m_netVolume)/m_netVolume);
        double changeClayFrac((targetClayVol-m_clayVolume)/m_clayVolume);
        double changeClayFromNet((targetClayVol-m_netVolume)/m_netVolume);
        
        double netVolSum(0.0), clayVolSum(0.0);
        for(size_t i = 0; i < m_rockLattice.size(); ++i)
        {
            double newNetVol = m_rockLattice[i]->netVolume()*(1.0+changeNetFrac);
            double newClayVol(0.0);
            if(clayPoroTrgt >= 0.0 && m_clayVolume > 0.0)
                newClayVol = m_rockLattice[i]->clayVolume()*(1.0+changeClayFrac);
            else if(clayPoroTrgt >= 0.0 && m_clayVolume == 0.0)
                newClayVol = m_rockLattice[i]->netVolume()*(1.0+changeClayFromNet);

            m_rockLattice[i]->adjustVolume(newNetVol, newClayVol, netVolSum, clayVolSum);
        }
        
        m_netVolume = netVolSum;
        m_clayVolume = clayVolSum;
    }    
}

void Netsim::modifyShapeFactor(int model, const string& options, vector< RockElem* >& elems, 
                               const string& fileName, bool writeToFile, int numPts)
{
    istringstream in(options);

    if(model == 0)
    {
        // Do nothing option
    }
    else if(model == 1)
    {
        string distFile;
        double lowCutOffG, highCutOffG;
        in >> distFile >> lowCutOffG >> highCutOffG;
        setShapeFactFromFile(elems, distFile, lowCutOffG, highCutOffG);
    }
    else if(model == 3)
    {
        double multFactor;
        in >> multFactor;
        for(size_t i = 0; i < elems.size(); ++i)
        {
            double gFact = min(sqrt(3.0)/36.0-1.0E-6, elems[i]->shape()->shapeFactor()*multFactor);
            elems[i]->shape()->setShapeFactor(gFact);
        }
    }
    else if(model == 4)     // Not a very good model this.....
    {                       // NEED TO CHANGE THIS
        double aConst;
        in >> aConst;
        sort(elems.begin(), elems.end(), ElemGCmpRed());
        double medianG = elems[elems.size()/2]->shape()->shapeFactor();
        for(size_t i = 0; i < elems.size(); ++i)
        {
            double gFact = aConst*(elems[i]->shape()->shapeFactor()-medianG)+elems[i]->shape()->shapeFactor();
            if(gFact > sqrt(3.0)/36.0) gFact = sqrt(3.0)/36.0-1.0E-6;
            if(gFact <= 0.0) gFact = 0.0001;
            elems[i]->shape()->setShapeFactor(gFact);
        }        
    }
    else if(model == 5)
    {
        double minVal, maxVal, deltaExp, etaExp;
        in >> minVal >> maxVal >> deltaExp >> etaExp;
        if(maxVal > sqrt(3.0)/36.0)
        {
            ostringstream out;
            out << endl
                << "==========================================================="    << endl
                << "Warning: Maximum shape factor should be less than 0.048113"     << endl
                << "which represents a equilateral triangle."                       << endl
                << "==========================================================="    << endl
                << endl;
            writePrtData(out);
            maxVal = sqrt(3.0)/36.0 - 1.0E-6;
        }

        for(size_t i = 0; i < elems.size(); ++i)
        {
            elems[i]->shape()->setShapeFactor(weibull(minVal, maxVal, deltaExp, etaExp));
        }
    }
    else
    {
        cerr << "==========================================" << endl
             << "Error: Uknown shape factor modifier option" << endl
             << "==========================================" << endl;
        exit(-1);
    }
    
    if(writeToFile)
    {
        ofstream fout(fileName.c_str());
        fout.flags(ios::showpoint);
        fout.flags(ios::scientific);
        fout.precision(6);

        int totNum(0);
        for(size_t i = 0; i < elems.size(); ++i)
        {
            if(elems[i]->shape()->shapeFactor() <= sqrt(3.0)/36.0)
                ++totNum;
        }
                
        sort(elems.begin(), elems.end(), ElemGCmpRed());
        
        double step(0.0), stepDelta(1.0/numPts);
        int runIdx(0);
        for(size_t j = 0; j < elems.size(); ++j)
        {
            if(elems[j]->shape()->shapeFactor() <= sqrt(3.0)/36.0)
            {
                ++runIdx;
                double frac = static_cast< double >(runIdx)/static_cast< double >(totNum);
                if(frac > step || frac == 1.0)
                {
                    fout << elems[j]->shape()->shapeFactor() << ",   "  << frac << endl;
                    step += stepDelta;
                }
            }
        }
    }
}

void Netsim::modifyInscribedRadii(int model, const string& options, vector< RockElem* >& elems, 
                                  const string& fileName, bool writeToFile, int numPts, bool forPores)
{
    istringstream in(options);
    sort(elems.begin(), elems.end(), ElemRadCmpRed());

    if(model == 0)
    {
        // Do nothing option
    }
    else if(model == 1)
    {
        string distFile;
        double lowCutOffDiam, highCutOffDiam;
        in >> distFile >> lowCutOffDiam >> highCutOffDiam;
        setRadiiFromFile(elems, distFile, lowCutOffDiam*1.0E-6, highCutOffDiam*1.0E-6, forPores);
    }
    else if(model == 2)
    {
        string whichAspectRatioType;
        double aspectRatioMin, aspectRatioMax, delta, eta;
        in  >> aspectRatioMin >> aspectRatioMax >> delta >> eta;
        for(size_t i = 0; i < elems.size(); ++i)
        {
            double aspectRatio(-1.0);
            if(aspectRatioMin > 0.0 && aspectRatioMax > 0.0)
                aspectRatio = weibull(aspectRatioMin, aspectRatioMax, delta, eta);
                
            elems[i]->setRadiusFromAspectRatio(aspectRatio);
        }
    }
    else if(model == 3)
    {
        double multFactor;
        in >> multFactor;
        for(size_t i = 0; i < elems.size(); ++i)
        {
            double maxRadius(elems[i]->shape()->radius()*multFactor);
            if(forPores)
            {
                for(int j = 0; j < elems[i]->connectionNum(); ++j)
                {
                    maxRadius = max(maxRadius, elems[i]->connection(j)->shape()->radius());
                }
            }
            elems[i]->shape()->setRadius(maxRadius);
        }
    }
    else if(model == 4)
    {
        double aConst;
        bool applyAboveMedian, applyBelowMedian;
        char above, below;
        in >> aConst >> above >> below;
        applyAboveMedian = (above == 't' || above == 'T');
        applyBelowMedian = (below == 't' || below == 'T');

        vector< pair< double, double > > poreSizeDist;
        double totVolume(0.0);
        for(size_t i = 0; i < elems.size(); ++i)
        {
            totVolume += elems[i]->netVolume();
            pair< double, double > entry(elems[i]->shape()->radius(), totVolume);
            poreSizeDist.push_back(entry);
        }
        double volBasedAvr = tableLookUpX(totVolume/2.0, poreSizeDist);
        for(size_t j = 0; j < elems.size(); ++j)
        {
            double rad(elems[j]->shape()->radius());
            if((rad > volBasedAvr && applyAboveMedian) || (rad < volBasedAvr && applyBelowMedian))
                rad = pow(elems[j]->shape()->radius(), aConst)/pow(volBasedAvr, aConst-1.0);
            elems[j]->shape()->setRadius(rad);
        }        
    }
    else if(model == 5)
    {
        double minVal, maxVal, deltaExp, etaExp;
        in >> minVal >> maxVal >> deltaExp >> etaExp;
        minVal *= 1.0E-6;
        maxVal *= 1.0E-6;
        for(size_t i = 0; i < elems.size(); ++i)
        {
            elems[i]->shape()->setRadius(weibull(minVal, maxVal, deltaExp, etaExp));
        }
    }
    else
    {
        cerr << "====================================" << endl
             << "Error: Uknown radius modifier option" << endl
             << "====================================" << endl;
        exit(-1);
    }    

    if(writeToFile)
    {
        ofstream fout(fileName.c_str());
        fout.flags(ios::showpoint);
        fout.flags(ios::scientific);
        fout.precision(6);

        double cumVol(0.0), totCumVol(0.0);
        sort(elems.begin(), elems.end(), ElemRadCmpRed());
        for(size_t i = 0; i < elems.size(); ++i)
            totCumVol += elems[i]->netVolume();

        double step(0.0), stepDelta(1.0/numPts);
        for(size_t j = 0; j < elems.size(); ++j)
        {
            cumVol += elems[j]->netVolume();
            double poreVolFrac = cumVol/totCumVol;
            if(poreVolFrac > step || poreVolFrac == 1.0)
            {
                fout << setw(15) << elems[j]->shape()->radius()*2.0E6 << "," << setw(15) << poreVolFrac << endl;
                step += stepDelta;
            }
        }
    }
}

void Netsim::setRadiiFromFile(vector< RockElem* >& elems, const string& radiiFileName, 
                              double lowCutOffDiam, double highCutOffDiam, bool forPores)
{
    ifstream fin(radiiFileName.c_str());
    if (!fin)
    {
        cerr << "==========================================================="       << endl 
             << "Error: Unable to open radius distribution file " << radiiFileName  << endl
             << "==========================================================="       << endl;
        exit( -1 );
    }

    vector< pair< double, double > > targetDist;        // diameter (dec) vs. pore vol fraction (inc)
    pair< double, double > funcPt, oldPt(1.0, -1.0);
    while(fin >> funcPt.first)
    {
        fin >> funcPt.second;
        funcPt.first *= 1.0E-6;                    // Convert from micro.m to m
        if(funcPt.second < oldPt.second || funcPt.first >= oldPt.first)
        {
            cerr << "==========================================================="   << endl 
                << "Error: Fractional pore volume should increase monotonically."   << endl
                << "==========================================================="    << endl;
            exit( -1 );
        }
        else if(funcPt.second != oldPt.second)
        {
            targetDist.push_back(funcPt);
        }
        oldPt = funcPt;
    }
   
    if(lowCutOffDiam > targetDist.back().first && lowCutOffDiam < targetDist.front().first)
    {
        double volCutoff = tableLookUpY(lowCutOffDiam, targetDist);
        while(!targetDist.empty() && targetDist.back().first < lowCutOffDiam)
            targetDist.pop_back();

        for(size_t i = 0; i < targetDist.size(); ++i)
            targetDist[i].second /= volCutoff;
        pair< double, double > endPoint(lowCutOffDiam, 1.0);
        targetDist.push_back(endPoint);
    }

    if(highCutOffDiam > targetDist.back().first && highCutOffDiam < targetDist.front().first)
    {
        double volCutoff = tableLookUpY(highCutOffDiam, targetDist);
        vector< pair< double, double > > newTargetDist;
        pair< double, double > pt(highCutOffDiam, 0.0);
        newTargetDist.push_back(pt);
        
        for(size_t i = 0; i < targetDist.size(); ++i)
        {
            if(targetDist[i].first < highCutOffDiam)
            {
                pair< double, double > pt(targetDist[i].first, 
                    (targetDist[i].second-volCutoff)/(1.0-volCutoff)); 
                newTargetDist.push_back(pt);
            }
        }
        targetDist = newTargetDist;
    }

    if(targetDist.empty() || targetDist.front().second != 0.0 || targetDist.back().second != 1.0)
    {
        cerr << endl
            << "============================================================"   << endl
            << "Error: Target distribution contains errors. The cumulative"     << endl
            << "volume should go from 0.0 to 1.0."                              << endl
            << "============================================================"   << endl;
        exit(-1);
    }

    double totElemVol(0.0), runningVol(0.0);
    for(size_t i = 0; i < elems.size(); ++i)
    {
        totElemVol += elems[i]->netVolume();
    }

    for(size_t j = 0; j < elems.size(); ++j)
    {
        runningVol += elems[j]->netVolume();
        double poreVolFrac = runningVol/totElemVol;
        double maxRadius(tableLookUpX(poreVolFrac, targetDist)/2.0);
        if(forPores)
        {
            for(int k = 0; k < elems[j]->connectionNum(); ++k)
            {
                maxRadius = max(maxRadius, elems[j]->connection(k)->shape()->radius());
            }
        }
        elems[j]->shape()->setRadius(maxRadius);
    }
}

void Netsim::setShapeFactFromFile(vector< RockElem* >& elems, const string& gFileName, 
                              double lowCutOffDiam, double highCutOffDiam)
{
    ifstream fin(gFileName.c_str());
    if (!fin)
    {
        cerr << "==========================================================="   << endl 
             << "Error: Unable to open G distribution file " << gFileName       << endl
             << "==========================================================="   << endl;
        exit( -1 );
    }

    vector< pair< double, double > > targetDist;        // g (dec) vs. numeric fraction (inc)
    pair< double, double > funcPt, oldPt(1.0, -1.0);
    while(fin >> funcPt.first)
    {
        fin >> funcPt.second;
        if(funcPt.second < oldPt.second || funcPt.first > oldPt.first)
        {
            cerr << "==========================================================="   << endl 
                << "Error: Fractional index should increase monotonically."         << endl
                << "==========================================================="    << endl;
            exit( -1 );
        }
        else if(funcPt.second != oldPt.second)
        {
            targetDist.push_back(funcPt);
        }
        oldPt = funcPt;
    }
   
    if(lowCutOffDiam > targetDist.back().first && lowCutOffDiam < targetDist.front().first)
    {
        double volCutoff = tableLookUpY(lowCutOffDiam, targetDist);
        while(!targetDist.empty() && targetDist.back().first < lowCutOffDiam)
            targetDist.pop_back();

        for(size_t i = 0; i < targetDist.size(); ++i)
            targetDist[i].second /= volCutoff;
        pair< double, double > endPoint(lowCutOffDiam, 1.0);
        targetDist.push_back(endPoint);
    }

    if(highCutOffDiam > targetDist.back().first && highCutOffDiam < targetDist.front().first)
    {
        double volCutoff = tableLookUpY(highCutOffDiam, targetDist);
        vector< pair< double, double > > newTargetDist;
        pair< double, double > pt(highCutOffDiam, 0.0);
        newTargetDist.push_back(pt);
        
        for(size_t i = 0; i < targetDist.size(); ++i)
        {
            if(targetDist[i].first < highCutOffDiam)
            {
                pair< double, double > pt(targetDist[i].first, 
                    (targetDist[i].second-volCutoff)/(1.0-volCutoff)); 
                newTargetDist.push_back(pt);
            }
        }
        targetDist = newTargetDist;
    }

    if(targetDist.empty() || targetDist.front().second != 0.0 || targetDist.back().second != 1.0)
    {
        cerr << endl
            << "============================================================"   << endl
            << "Error: Target distribution contains errors. The cumulative"     << endl
            << "volume should go from 0.0 to 1.0."                              << endl
            << "============================================================"   << endl;
        exit(-1);
    }

    int totNum(0);
    for(size_t i = 0; i < elems.size(); ++i)
    {
        if(elems[i]->shape()->shapeFactor() <= sqrt(3.0)/36.0)
            ++totNum;
    }

    int runIdx(0);
    for(size_t j = 0; j < elems.size(); ++j)
    {
        if(elems[j]->shape()->shapeFactor() <= sqrt(3.0)/36.0)
        {
            ++runIdx;
            double frac = static_cast< double >(runIdx)/static_cast< double >(totNum);
            double newG = tableLookUpX(frac, targetDist);
            elems[j]->shape()->setShapeFactor(newG);
        }
    }
}

void Netsim::setContactAngles(vector< RockElem* >& pores, vector< RockElem* >& throats, double min, double max, 
                              double delta, double eta, bool attractToMarked)
{
    vector< double > conAngles(pores.size());
    for(size_t i = 0; i < conAngles.size(); ++i)
    {
        conAngles[i] = weibull(min, max, delta, eta);
    }

    sort(conAngles.begin(), conAngles.end(), greater< double >());
    if(m_angDistScheme[2] == 'a' || m_angDistScheme[2] == 'A')      // rMax
        sort(pores.begin(), pores.end(), ElemRadCmpRed());
    else if(m_angDistScheme[2] == 'i' || m_angDistScheme[2] == 'I') // rMix
        sort(pores.begin(), pores.end(), ElemRadCmpInc());
    else
        sort(pores.begin(), pores.end(), ElemRandCmp());            // Random

    for(size_t j = 0; j < pores.size(); ++j)
    {
        double rad = pores[j]->shape()->radius();
        double conAng = conAngles[j];
        pores[j]->shape()->setContactAngles(conAngles[j], m_wettingClass, m_modelTwoSepAng);
    }

    for(size_t k = 0; k < throats.size(); ++k)
    {
        const RockElem* pOne = throats[k]->connection(0);
        const RockElem* pTwo = throats[k]->connection(1);
        if((pOne->shape()->fractionalWetPopl(0) && pOne->shape()->fractionalWetPopl(0)) ||    // None or both are marked ->
            (!pOne->shape()->fractionalWetPopl(0) && !pOne->shape()->fractionalWetPopl(0)))   // assign on 50/50 basis
        {
            if(static_cast<double>(rand())/static_cast<double>(RAND_MAX) > 0.5)  
                throats[k]->shape()->setContactAngles(pOne->shape()->conAngEquil(), m_wettingClass, m_modelTwoSepAng);
            else
                throats[k]->shape()->setContactAngles(pTwo->shape()->conAngEquil(), m_wettingClass, m_modelTwoSepAng);
        }
        else if((pOne->shape()->fractionalWetPopl(0) && attractToMarked) || (pTwo->shape()->fractionalWetPopl(0) && !attractToMarked))
        {
            throats[k]->shape()->setContactAngles(pTwo->shape()->conAngEquil(), m_wettingClass, m_modelTwoSepAng);
        }
        else
        {
            throats[k]->shape()->setContactAngles(pOne->shape()->conAngEquil(), m_wettingClass, m_modelTwoSepAng);
        }
    }
}


/////////////////////////////////////////////////////////////////////////////////
// Change the advancing contact angle depending on several criteria set out in
// input file. Wettability alterations will only occur in elems that have been
// drained
/////////////////////////////////////////////////////////////////////////////////
void Netsim::applyFractionalWetting(InputData& input)
{
    bool applyFracWetting(false), volBased(false), oilClustInWat(true);
    double fraction, minAng, maxAng, deltaExp, etaExp, totVolume(0.0), absVolume(0.0), fracWetted(0.0);
    int totElem(0), clustDiam(0);
    string fracWettingModel;
    vector< RockElem* > pores2BAltered, throats2BAltered;

    input.fracWetting(applyFracWetting, fraction, volBased, minAng, maxAng, deltaExp, etaExp, 
        fracWettingModel, clustDiam, oilClustInWat);

    if(!applyFracWetting) return;

    ostringstream out;
    out.flags(ios::showpoint);
    out.flags(ios::fixed);
    out.precision(3);

    for(size_t el = 0; el < m_rockLattice.size(); ++el)
    {
        if(m_rockLattice[el]->shape()->containsOil())
        {
            if(el < static_cast< size_t >(m_numPores+2)) ++totElem;                        // Only count pores
            totVolume += m_rockLattice[el]->netVolume();
        }
        absVolume += m_rockLattice[el]->netVolume();
    }

    if(fracWettingModel[1] == 'o' || fracWettingModel[1] == 'O')    // Spatial correlation approach
    {
        int numFracWetted(0), clusterIdx(1);
        bool allSystemsAreGo(true);
        double targetFrac = oilClustInWat ? fraction: 1.0-fraction;
        while(allSystemsAreGo)
        {
            double clusterVol(0.0);
            int clusterElem(0);
            set< RockElem * > frontier, oldFrontier;
            double randNum = static_cast<double>(rand()) / static_cast<double>(RAND_MAX);
            //double randNum = 0.5;   // delete me
            int randPoreIdx = static_cast< int >(randNum*m_numPores);
            randPoreIdx = max(1, randPoreIdx);
            frontier.insert(m_rockLattice[randPoreIdx]);
            typedef set< RockElem* >::iterator ItrSet;
            int frontExpansions(0);
            while(!frontier.empty() && allSystemsAreGo && frontExpansions < clustDiam)  // Front expansions will go over thraots as well as pores, wheras
            {                                                                           // cluster diam is wrt pores => Don't divide by 2
                oldFrontier = frontier;
                frontier.clear();
                ++frontExpansions;                
                for(ItrSet itrElem = oldFrontier.begin(); itrElem != oldFrontier.end(); ++itrElem)
                {
                    if(!(*itrElem)->shape()->fractionalWetPopl(clusterIdx) && allSystemsAreGo) 
                    {
                        if((*itrElem)->shape()->fractionalWetPopl(0) && (*itrElem)->shape()->containsOil())
                        {
                            if(oilClustInWat && dynamic_cast< Pore* >(*itrElem) != 0)
                            {
                                assert((*itrElem)->iAmAPore());
                                pores2BAltered.push_back(*itrElem);
                            }
                            else if(oilClustInWat)
                            {
                                assert(!(*itrElem)->iAmAPore());
                                throats2BAltered.push_back(*itrElem);
                            }
                            clusterVol += (*itrElem)->netVolume();
                            fracWetted += (*itrElem)->netVolume();
                            ++clusterElem;     
                            if(dynamic_cast< Pore* >(*itrElem) != 0) ++numFracWetted;   // Only count pores

                            m_wettingFraction = volBased ? fracWetted/totVolume: numFracWetted/totElem;
                            allSystemsAreGo = ((volBased && fracWetted/totVolume < targetFrac) || 
                                (!volBased && static_cast<double>(numFracWetted)/totElem < targetFrac));
                            if(!allSystemsAreGo) 
                                break;
                        }

                        (*itrElem)->shape()->setFracWetPopl(clusterIdx);                    // Set flag
                        for(int i = 0; i < (*itrElem)->connectionNum(); ++i)
                        {
                            if(!(*itrElem)->connection(i)->isExitOrEntryRes() && 
                                !(*itrElem)->connection(i)->shape()->fractionalWetPopl(clusterIdx))
                            {
                                frontier.insert((*itrElem)->connection(i));
                            }
                        }			
                    }
                }
            }
            ++clusterIdx;
        }
        if(!oilClustInWat)
        {
            numFracWetted = 0;
            fracWetted = totVolume-fracWetted;
            for(size_t elm = 0; elm < m_rockLattice.size(); ++elm)
            {
                if(m_rockLattice[elm]->shape()->containsOil() && m_rockLattice[elm]->shape()->fractionalWetPopl(0))
                {
                    ++numFracWetted;
                    if(static_cast< int >(elm) < m_numPores+2)
                    {
                        assert(m_rockLattice[elm]->iAmAPore());
                        pores2BAltered.push_back(m_rockLattice[elm]);
                    }
                    else
                    {
                        assert(!m_rockLattice[elm]->iAmAPore());
                        throats2BAltered.push_back(m_rockLattice[elm]);
                    }
                }
            }
        }

        out << "================================================================"       << endl
            << "Fractional wetting was applied"                                         << endl
            << "Number of correlated regions: " << clusterIdx-1                         << endl
            << "Altered volume (of oil invaded volume): " << fracWetted/totVolume       << endl
            << "Altered volume (of total net pore volume): " << fracWetted/absVolume    << endl
            << "Number of pores altered: " << numFracWetted                             << endl
            << "================================================================="      << endl;
    }
    else
    {
        oilClustInWat = true;   // Ensure this is set for non-spatial approach
        vector< pair<double, RockElem*> > toBeAltered;

        for(int i = 1; i <= m_numPores; ++i)
        {
            if(m_rockLattice[i]->shape()->containsOil())
            {
                pair<double, RockElem*> rockEntry;
                rockEntry.second = m_rockLattice[i];
                if(fracWettingModel[1] == 'a' || fracWettingModel[1] == 'A')        // Random
                {
                    rockEntry.first = static_cast<double>(rand()) / static_cast<double>(RAND_MAX);
                    //rockEntry.first = 0.5;  // delete me
                }
                else if(fracWettingModel[0] == 'r' || fracWettingModel[0] == 'R')   // Radius
                    rockEntry.first = m_rockLattice[i]->shape()->radius();
                else
                {
                    cerr << endl
                        << "==============================================================="            << endl
                        << "Error: Did not recognize fractional wetting model: " << fracWettingModel  << endl   
                        << "==============================================================="            << endl
                        << endl;
                    exit(-1);
                }
                toBeAltered.push_back(rockEntry);
            }
        }
        if(fracWettingModel[2] == 'a' || fracWettingModel[2] == 'A')
            sort(toBeAltered.begin(), toBeAltered.end(), FracWettInc());
        else
            sort(toBeAltered.begin(), toBeAltered.end(), FracWettDec());
        
        int numElem(0), targetNum(static_cast< int >(fraction*toBeAltered.size()));
        double poreVol(0.0);
                
        while(!toBeAltered.empty() &&
            ((volBased && poreVol/totVolume < fraction) || (!volBased && numElem < targetNum)))
        {
            ++numElem;
            RockElem* elem = toBeAltered.back().second;
            poreVol += elem->netVolume();
            double newConAng = weibull(minAng, maxAng, deltaExp, etaExp);
            pores2BAltered.push_back(elem);
            elem->shape()->setFracWetPopl(1);
            toBeAltered.pop_back();
            m_wettingFraction = volBased ? poreVol/totVolume: numElem*fraction/targetNum;
            for(int t = 0; t < elem->connectionNum(); ++t)
            {
                RockElem* throat = elem->connection(t);
                if(throat->shape()->fractionalWetPopl(0) && throat->shape()->containsOil())
                {
                    double randNum = static_cast<double>(rand()) / static_cast<double>(RAND_MAX);
                    //double randNum = 0.5;   // delete me
                    if(randNum < fraction)
                    {
                        throats2BAltered.push_back(throat);
                        throat->shape()->setFracWetPopl(1);
                        poreVol += throat->netVolume();
                        if(volBased) m_wettingFraction = poreVol/totVolume;
                    }
                }
            }
        }
        out << "================================================================"   << endl
            << "Fractional wetting was applied"                                     << endl
            << "Altered volume (of oil invaded volume): " << poreVol/totVolume      << endl
            << "Altered volume (of total net pore volume): " << poreVol/absVolume   << endl
            << "Number of pores altered: " << numElem                               << endl
            << "================================================================="  << endl;
    }    
    setContactAngles(pores2BAltered, throats2BAltered, minAng, maxAng, deltaExp, etaExp, oilClustInWat);    
    writePrtData(out);
}

/////////////////////////////////////////////////////////////////////////////////
// Does the first flow through of the network
/////////////////////////////////////////////////////////////////////////////////
void Netsim::initFlow(InputData& input, bool writeMatInitOnly, bool createLocData, bool drainSinglets)
{
    bool writeNetToFile(false), writeInBinary(false);
    string optNetFileName;

    input.writeNetwork(writeNetToFile, writeInBinary, optNetFileName);

    double singlePhaseErr(0.0), cpuTmp(0.0), currentErr(0.0);
        
    if((m_writeWatVelocity || m_writeOilVelocity) && m_writeSlvMatrixAsMatlab) createMatlabLocationData();
    
    string matFile = m_matrixFileName + "_init";
    Solver singlePhaseSolve(m_rockLattice, m_krInletBoundary, m_krOutletBoundary, m_numPores+1, m_maxNonZeros, 
        matFile, m_writeSlvMatrixAsMatlab);
    
    m_singlePhaseWaterQ = singlePhaseSolve.flowrate(m_inletSolverPrs, m_outletSolverPrs, m_water, singlePhaseErr, 
        cpuTmp, 1.0, m_writeWatVelocity, m_writeWatMatrix);
    m_singlePhaseOilQ = m_singlePhaseWaterQ * (m_water->viscosity() / m_oil->viscosity());
    if(m_useAvrPrsAsBdr)
    {
        prsOrVoltDrop(m_water, false, m_singlePhaseDprs);
        m_deltaPw = m_singlePhaseDprs;
    }
    if(m_prtPressureProfile && m_calcRelPerm) recordPrsProfiles(m_water);

    m_watFlowRate = m_singlePhaseWaterQ;
    m_oilFlowRate = 0.0;
    
    m_absPermeability = (m_singlePhaseWaterQ * m_water->viscosity() * m_xSize * (m_solverBoxEnd-m_solverBoxStart)) 
        / (m_ySize * m_zSize * 9.869233E-16 * (m_inletSolverPrs - m_outletSolverPrs));
        
    m_singlePhaseCurrent = singlePhaseSolve.flowrate(m_inletSolverPrs, m_outletSolverPrs, m_water, currentErr, cpuTmp, 1.0, 
        m_writeResVelocity, m_writeResMatrix, true);
    if(m_useAvrPrsAsBdr)prsOrVoltDrop(m_water, true, m_singlePhaseDvolt);

    if(fabs(singlePhaseErr) > 0.05 || fabs(currentErr) > 0.05)
    {
        ostringstream out;
        out << endl 
            << "====================================================="  << endl
            << "Warning. Large errors were detected when solving for"   << endl
            << "single phase conditions."                               << endl
            << "Flow Error: " << singlePhaseErr                         << endl
            << "Resistivity Error: " << currentErr                      << endl
            << "====================================================="  << endl;
        writePrtData(out);
    }

    m_formationFactor = ((m_ySize * m_zSize) * (m_inletSolverPrs - m_outletSolverPrs)) 
        / (m_water->resistivity() * m_singlePhaseCurrent * m_xSize * (m_solverBoxEnd-m_solverBoxStart)); 
    
    if(writeNetToFile) writeNetworkToFile(writeInBinary, drainSinglets, optNetFileName);
    if(writeMatInitOnly)
    {
        m_writeWatMatrix = false;
        m_writeOilMatrix = false; 
        m_writeResMatrix = false; 
        m_writeWatVelocity = false;
        m_writeOilVelocity = false; 
        m_writeResVelocity = false; 
    }

    if(createLocData) createMatlabLocationData();
}

void Netsim::createMatlabLocationData() const 
{
    ofstream outp("poreLocation.m");
    outp.flags(ios::scientific);
    outp.precision(3);
    outp << "% The physical dimensions (m) of the network is: " << endl;
    outp << "modSize(1) = " << m_xSize << ";" << endl;
    outp << "modSize(2) = " << m_ySize << ";" << endl;
    outp << "modSize(3) = " << m_zSize << ";" << endl;      

    outp << endl << "% The location of individual pores (m). The array is 1-based" << endl;
    outp << "poreLoc = [";
    for(int i = 1; i <= m_numPores; ++i)
    {
        outp << m_rockLattice[i]->node()->xPos() << ", "
            << m_rockLattice[i]->node()->yPos() << ", "
            << m_rockLattice[i]->node()->zPos() << "; ..."
            << endl;
    }
    outp << "];" << endl;
    outp.close();

    ofstream outt("throatConnection.m");
    outt << "% Pores to which the throats are connected. The indexing is" << endl
        << "% 1-based, which is the same as that used by the Oren format" << endl;

    outt << "throatConn = [";
    for(size_t j = m_numPores+2; j < m_rockLattice.size(); ++j)
    {
        outt << m_rockLattice[j]->connection(0)->orenIndex() << ", ";
        if(m_rockLattice[j]->connectionNum() == 2)
            outt << m_rockLattice[j]->connection(1)->orenIndex() << "; " << endl;
        else
            outt << m_rockLattice[j]->connection(0)->orenIndex() << "; " << endl;            
    }
    outt << "];" << endl;
    outt.close();
}

/////////////////////////////////////////////////////////////////////////////////
// Writes the optimized network to file
/////////////////////////////////////////////////////////////////////////////////
void Netsim::writeNetworkToFile(bool writeInBinary, bool drainSinglets, const string& fileNameBase)
{
    if(!drainSinglets)
    {
        cerr << "==================================================" << endl
            << "The option not to drain dangling ends (option 4 in" << endl
            << "keyword TRAPPING) will modify the network and can " << endl
            << "not be used together with the option to write the " << endl
            << "optimized network to file."                         << endl
            << "==================================================" << endl;
        exit(-1);
    }
    
    if(writeInBinary)
    {
        string poreFileName(fileNameBase + "_node.bin"), throatFileName(fileNameBase + "_link.bin");
        ofstream pOut(poreFileName.c_str(), ios::binary), tOut(throatFileName.c_str(), ios::binary);
        if(!pOut || !tOut)
        {
            cout << endl
                << "====================================================="  << endl
                << "Warning: Could not open " << poreFileName               << endl   
                << "for writing. Network is not written to file."           << endl
                << "====================================================="   << endl
                << endl;
        }
        
        pOut.write((char *)(&m_numPores), sizeof(int));
        pOut.write((char *)(&m_xSize), sizeof(double));
        pOut.write((char *)(&m_ySize), sizeof(double));
        pOut.write((char *)(&m_zSize), sizeof(double));
        tOut.write((char *)(&m_numThroats), sizeof(int));
        
        for(int i = 1; i <= m_numPores; ++i)
            m_rockLattice[i]->writeNetworkDataBinary(pOut);
        
        for(size_t j = m_numPores+2; j < m_rockLattice.size(); ++j)
            m_rockLattice[j]->writeNetworkDataBinary(tOut);
        
        pOut.close();
        tOut.close();
    }
    else
    {
        string pOut1FileName(fileNameBase + "_node1.dat"), pOut2FileName(fileNameBase + "_node2.dat");
        
        ofstream pOut1, pOut2;
        pOut1.open(pOut1FileName.c_str());
        pOut2.open(pOut2FileName.c_str());
        pOut1.flags(ios::showpoint);
        pOut1.flags(ios::scientific);
        if(!pOut1 || !pOut2)
        {
            cout << endl
                << "====================================================="  << endl
                << "Warning: Could not open " << pOut1FileName              << endl   
                << "for writing. Network is not written to file."           << endl
                << "====================================================="  << endl
                << endl;
        }
        
        pOut1 << m_numPores << "   " << m_xSize << "   " << m_ySize << "   " << m_zSize << '\n';
        
        for(int i = 1; i <= m_numPores; ++i)
            m_rockLattice[i]->writeNetworkData(pOut1, pOut2);
        
        pOut1.close();
        pOut2.close();
        
        string tOut1FileName(fileNameBase + "_link1.dat"), tOut2FileName(fileNameBase + "_link2.dat");
        ofstream tOut1, tOut2;
        tOut1.open(tOut1FileName.c_str());
        tOut2.open(tOut2FileName.c_str());
        
        tOut1 << m_numThroats << "   " << '\n';
        
        for(size_t j = m_numPores+2; j < m_rockLattice.size(); ++j)
            m_rockLattice[j]->writeNetworkData(tOut1, tOut2);
        
        tOut1.close();
        tOut2.close();
    }
}

/////////////////////////////////////////////////////////////////////////////////
// Gets the pressure drop between two planes in the network, used for kr
/////////////////////////////////////////////////////////////////////////////////
bool Netsim::prsOrVoltDrop(const Fluid* fluid, bool resistSolve, double& prsDrop) const
{
    double prsOut(0.0), stdOut(0.0);
    int numOut(0);
        
    bool btOut = avrPrsOrVolt(fluid, resistSolve, m_numPressurePlanes-1, prsOut, stdOut, numOut);

    double prsIn(0.0), stdIn(0.0);
    int numIn(0);
        
    bool btIn = avrPrsOrVolt(fluid, resistSolve, 0, prsIn, stdIn, numIn);

    prsDrop = prsIn - prsOut;
    return btOut && btIn;
}

/////////////////////////////////////////////////////////////////////////////////
// Calculates the average pressure in throats crossing a plane. Assumes the pore
// pressures to be point distributed. Will also calculate the number of pressure
// points along with standard deviation
/////////////////////////////////////////////////////////////////////////////////
bool Netsim::avrPrsOrVolt(const Fluid* fluid, bool resistSolve, int prsPlane, double& res, 
                          double& stdDev, int& numVal) const
{
    double loc(m_pressurePlanesLoc[prsPlane]*m_xSize), resSum(0.0), resSumSq(0.0), flowSum(0.0), flowTarget;
    vector< pair< double, double > > valArray;
    string phase("water");
    
    if(resistSolve)
    {
        flowTarget = m_current;
        phase = "current";
    }
    else if(dynamic_cast< const Oil* >(fluid) != NULL)
    {
        flowTarget = m_oilFlowRate;
        phase = "oil";
    }
    else 
        flowTarget = m_watFlowRate;
    
    
    flowSum = 0.0;
    for(size_t i = 0; i < m_pressurePlanes[prsPlane].size(); ++i)
    {
        double val(0.0), flowRate(0.0);
        if(m_pressurePlanes[prsPlane][i]->prevSolvrRes(fluid, resistSolve, loc, val, flowRate))
        {
            pair< double, double > valPair(val, flowRate);
            valArray.push_back(valPair);
            flowSum += flowRate;
        }
    }

    double flowError(fabs((flowTarget-flowSum)/flowSum));
    
    if(resistSolve && flowTarget > 0.0)
        m_maxResIdxErr = max(m_maxResIdxErr, flowError);
    else if(dynamic_cast< const Oil* >(fluid) != NULL && flowTarget > 0.0)
        m_maxOilFlowErr = max(m_maxOilFlowErr, flowError);
    else if(flowTarget > 0.0)
        m_maxWatFlowErr = max(m_maxWatFlowErr, flowError);
   
    if(flowError > MAX_FLOW_ERR && flowTarget > 0.0) 
        cout << "Large flow error (" << flowError*100.0 << "%) for " << phase << " at " << loc << endl;
    
    int numPts(static_cast< int >(valArray.size()));
    numVal = 0;
    for(int j = 0; j < numPts; ++j)
    {
        resSum += valArray[j].first;
        resSumSq += valArray[j].first*valArray[j].first;
        ++numVal;
    }

    if(numVal > 0) res = resSum / numVal;
    if(numVal > 1) stdDev = sqrt((numVal*resSumSq - resSum*resSum)/(numVal*(numVal-1)));
       
    return numVal > 0;
}


/////////////////////////////////////////////////////////////////////////////////////////////////////
// The data for the throats are read from the link files. Since the pores are not yet created their
// indicies are stored in a temporary vector and the pointers will be initialized later. Since 
// in/outlet does not have separete entries in the data files, we keep track of connecting throats.
// The strucure of the link files are as follows:
//
// *_link1.dat:
// index, pore 1 index, pore 2 index, radius, shape factor, total length (pore center to pore center)
//
// *_link2.dat:
// index, pore 1 index, pore 2 index, length pore 1, length pore 2, length throat, volume, clay volume
/////////////////////////////////////////////////////////////////////////////////////////////////////
int Netsim::readAndCreateThroats(InputData& input, vector< pair<int, int> >& connectingPores, 
                                  vector< RockElem* >& throatsToInlet, vector< RockElem* >& throatsToOutlet,
                                  vector< pair< int, double> >& poreHash, vector< int >& throatHash, int newNumPores)
{   
    double minConAng(0.0), maxConAng(0.0), delta, eta;
    input.initConAng(minConAng, maxConAng, delta, eta);

    int numLengthErrors(0), newNumThroats(0);
    for(int index = 1; index <= m_numThroats; ++index)
    {
        int poreOneIdx, poreTwoIdx;
        double radius, shapeFactor, lenTot, lenPore1, lenPore2, lenThroat, volume, clayVolume;

        input.throatData(index, poreOneIdx, poreTwoIdx, volume, clayVolume, radius, shapeFactor, 
            lenPore1, lenPore2, lenThroat, lenTot);

        if(poreOneIdx > 0)
            connectingPores[index - 1].first = poreHash[poreOneIdx-1].first;
        else
            connectingPores[index - 1].first = poreOneIdx;

        if(poreTwoIdx > 0)
            connectingPores[index - 1].second = poreHash[poreTwoIdx-1].first;
        else
            connectingPores[index - 1].second = poreTwoIdx;

        if(connectingPores[index - 1].first > 0 || connectingPores[index - 1].second > 0)
        {
            ++newNumThroats;
            throatHash[index-1] = newNumThroats;

            if(connectingPores[index - 1].first == DUMMY_IDX)
            {
                if(poreHash[poreOneIdx-1].second < m_xSize/2.0) 
                    connectingPores[index - 1].first = -1;
                else
                    connectingPores[index - 1].first = 0;
            }
            else if(connectingPores[index - 1].second == DUMMY_IDX)
            {
                if(poreHash[poreTwoIdx-1].second < m_xSize/2.0) 
                    connectingPores[index - 1].second = -1;
                else
                    connectingPores[index - 1].second = 0;
            }

            double initConAng = weibull(minConAng, maxConAng, delta, eta);       
            double adjustingVol(m_clayAdjust*(volume+clayVolume));
            adjustingVol = min(adjustingVol, volume);
            adjustingVol = -min(-adjustingVol, clayVolume);

            volume -= adjustingVol;
            clayVolume += adjustingVol;

            if(fabs(lenPore1+lenPore2+lenThroat-lenTot)/lenTot > 0.01)
                ++numLengthErrors;

            RockElem *throat = new Throat(m_commonData, m_oil, m_water, radius, volume, clayVolume, shapeFactor, initConAng, 
                lenThroat, lenPore1, lenPore2, newNumPores + 1 + newNumThroats);

            m_rockLattice.push_back(throat);

            if(connectingPores[index - 1].first == -1 || connectingPores[index - 1].second == -1)          
                throatsToInlet.push_back(throat);                        
            else if(connectingPores[index - 1].first == 0 || connectingPores[index - 1].second == 0)                  
                throatsToOutlet.push_back(throat);
        }
    }

    if(numLengthErrors > 0)
    {
        ostringstream out;

        out << endl
            << "==================================================="                << endl
            << "Warning: For " << numLengthErrors << " throats the lengths of the"  << endl
            << "pore-throat-pore did not match the total length."                   << endl
            << "This is generally only an artifact of the network"                  << endl
            << "reconstruction process, and is not serious."                        << endl
            << "==================================================="                << endl
            << endl;

        writePrtData(out);
    }

    return newNumThroats;
}

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// We're using different indicies for out and inlet. Paal-Eric uses -1 and 0 wheras we use 0 and (numPores+1), hence
// we need to renumber these. The reason for this is that -1 is not a good index when storing the element 
// pointers in a vector.
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
int Netsim::checkPoreIndex(int index) const
{
    if(index == -1)
        return 0;
    else if(index == 0)
        return m_numPores + 1;
    else
        return index;
}

int Netsim::setupPoreHashing(InputData& input, vector< pair< int, double > >& poreHash) const
{
    int numPores(0);
    for(int index = 1; index <= m_numPores; ++index)
    {
        pair< int, double > entry(DUMMY_IDX, 0.0);
        input.poreLocation(index, entry.second);
        if(entry.second >= m_xSize*(1.0-m_shavingFraction)/2.0 && 
            entry.second <= m_xSize-m_xSize*(1.0-m_shavingFraction)/2.0)
        {
            entry.first = ++numPores;
        }       
        poreHash[index-1] = entry;
    }
    return numPores;
}


////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// The pore data is read from the node files. At this point the throats are already created and the pointers 
// can be set. The strucure of the node files are as follows:
//
// *_node1.dat:
// index, x_pos, y_pos, z_pos, connection num, connecting nodes..., at inlet?, at outlet?, connecting links...
//
// *_node2.dat:
// index, volume, radius, shape factor, clay volume
////////////////////////////////////////////////////////////////////////////////////////////////////////////////
void Netsim::readAndCreatePores(InputData& input, vector< pair<int, int> >& connectingPores, 
                                vector< pair< int, double> >& poreHash, vector< int >& throatHash, int newNumPores)
{
    double minConAng(0.0), maxConAng(0.0), delta, eta, shaveOff(m_xSize*(1.0-m_shavingFraction)/2.0);
    input.initConAng(minConAng, maxConAng, delta, eta);
    int newIndex(0);

    for(int index = 1; index <= m_numPores; ++index)
    {
        int connNumber;
        double xPos, yPos, zPos, volume, radius, shapeFactor, clayVolume;
        vector< int > connThroats, connPores;
        vector< RockElem* > connectingThroats;

        input.poreData(index, xPos, yPos, zPos, connNumber, connThroats, connPores, volume, clayVolume, 
            radius, shapeFactor);
                
        if(xPos >= shaveOff && xPos <= m_xSize-shaveOff)
        {
            ++newIndex;
            double initConAng = weibull(minConAng, maxConAng, delta, eta);       
            double adjustingVol(m_clayAdjust*(volume+clayVolume));
            adjustingVol = min(adjustingVol, volume);
            adjustingVol = -min(-adjustingVol, clayVolume);

            volume -= adjustingVol;
            clayVolume += adjustingVol;

            connectingThroats.resize(connNumber);        
            for(int j = 0; j < connNumber; ++j)
            {
                int hashedThroatIdx = throatHash[connThroats[j]-1];
                
                int hashedPoreIdx = connPores[j];
                if(hashedPoreIdx > 0 && poreHash[connPores[j]-1].first != DUMMY_IDX) 
                    hashedPoreIdx = poreHash[connPores[j]-1].first;
                else if(hashedPoreIdx > 0 && poreHash[connPores[j]-1].second < m_xSize/2.0)
                    hashedPoreIdx = -1;
                else if(hashedPoreIdx > 0 && poreHash[connPores[j]-1].second > m_xSize/2.0)
                    hashedPoreIdx = 0;
                
                assert(newIndex == connectingPores[connThroats[j]-1].first 
                    || newIndex == connectingPores[connThroats[j]-1].second);

                assert(hashedPoreIdx == connectingPores[connThroats[j]-1].first 
                    || hashedPoreIdx == connectingPores[connThroats[j]-1].second);            

                connectingThroats[j] = m_rockLattice[newNumPores + 1 + hashedThroatIdx];
            }

            double initSolvPrs = (m_outletSolverPrs + m_inletSolverPrs)/2.0;

            Node *currNode = new Node(newIndex, newNumPores, xPos-shaveOff, yPos, zPos, m_xSize*m_shavingFraction);
            bool insideSlvrBox(currNode->isInsideBox(m_solverBoxStart, m_solverBoxEnd));
            bool insideSatBox(currNode->isInsideBox(m_satBoxStart, m_satBoxEnd));

            RockElem *pore = new Pore(m_commonData, currNode, m_oil, m_water, radius, volume, clayVolume, shapeFactor, 
                initConAng, insideSlvrBox, insideSatBox, initSolvPrs, connectingThroats);                      
            m_rockLattice[newIndex] = pore;
        }
    }
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////
// In and outlet only need to know what throats are connected to it. Their indicies are 0 and (n_pores + 1)
///////////////////////////////////////////////////////////////////////////////////////////////////////////
void Netsim::createInAndOutletPore(int index, double xPos, double yPos, double zPos, vector< RockElem* >& connThroats)
{    
    if(xPos > 0.0 && xPos < m_xSize)
    {
        cerr << endl 
            << "========================================="  << endl
            << "Error: Entry and exit reservoirs cannot be" << endl
            << "within the network model area."             << endl
            << "========================================="  << endl;
        exit(-1);
    }
    
    Node *currNode = new Node(index, m_numPores, xPos, yPos, zPos, m_xSize);
    RockElem *pore = new EndPore(m_commonData, currNode, m_oil, m_water, connThroats);                      
    m_rockLattice[currNode->index()] = pore;
}

/////////////////////////////////////////////////////////////////////////////////
// Sets up the solver, fills the sorted lists etc
/////////////////////////////////////////////////////////////////////////////////
void Netsim::initDrainage(bool calcRelPerm, bool calcResIdx)
{
    m_commonData->incrementDrainCycle();
    m_commonData->injectant(m_oil);
    m_commonData->drainagePhase(true);
    
    m_cpuTimeTotal = 0.0;
    m_cpuTimeCoal = 0.0;
    m_cpuTimeKrw = 0.0;
    m_cpuTimeKro = 0.0;
    m_cpuTimeTrapping = 0.0;
    m_cpuTimeResIdx = 0.0;
    m_totNumFillings = 0;
    m_maxCappPress = m_cappPress;
    m_commonData->minDatumPc(m_minCappPress);

    m_maxOilFlowErr = 0.0;
    m_maxWatFlowErr = 0.0;
    m_maxResIdxErr = 0.0;

    if(calcRelPerm || calcResIdx) 
    {
        ostringstream matDBGFileName;
        matDBGFileName << m_matrixFileName << "_draincycle_" << m_commonData->drainageCycle();
        m_solver = new Solver(m_rockLattice, m_krInletBoundary, m_krOutletBoundary, m_numPores+1, m_maxNonZeros, 
            matDBGFileName.str(), m_writeSlvMatrixAsMatlab);
    }

    ostringstream tmp;
    if(!m_calcRelPerm) solveLaplaceEq(calcRelPerm, false, tmp);     // Create starting points if not previously
    if(!m_calcResIdx) solveLaplaceEq(false, calcResIdx, tmp);       // available

    recordRes(calcRelPerm, calcResIdx);
    m_calcRelPerm = calcRelPerm;
    m_calcResIdx = calcResIdx;

    if(m_injAtExitRes && m_rockLattice[m_numPores+1]->shape()->bulkFluid() != m_oil) 
        m_rockLattice[m_numPores+1]->drain();
    else if(!m_injAtExitRes && m_rockLattice[m_numPores+1]->shape()->bulkFluid() == m_oil) 
        m_rockLattice[m_numPores+1]->imbide();
       
    if(m_injAtEntryRes && m_rockLattice[0]->shape()->bulkFluid() != m_oil) 
        m_rockLattice[0]->drain();
    else if(!m_injAtEntryRes && m_rockLattice[0]->shape()->bulkFluid() == m_oil) 
        m_rockLattice[0]->imbide();

    if(m_sourceNode != 0 && m_rockLattice[m_sourceNode]->shape()->bulkFluid() != m_oil)
        m_rockLattice[m_sourceNode]->drain();
        
    m_drainageEvents.clear();
    m_layerReformEvents.clear();
    
    for(size_t i = 1; i < m_rockLattice.size(); ++i)
    {
        if(i != static_cast< size_t >(m_numPores + 1) && m_rockLattice[i]->connectedToNetwork())
        {
            double gravCorr(m_rockLattice[i]->shape()->gravityCorrection());
            m_rockLattice[i]->shape()->initDrainage(m_cappPress-gravCorr);

            if(m_rockLattice[i]->addToEventVec(m_oil))
            {
                m_drainageEvents.quickInsert(m_rockLattice[i]);
                m_rockLattice[i]->isInOilFloodVec(true);
            }
            
            Polygon* polyShape = NULL;
            vector<int> addCrns;
            if(m_rockLattice[i]->addToLayerVec(m_oil, &polyShape, addCrns, m_cappPress-gravCorr))
            {
                for(size_t j = 0; j < addCrns.size(); ++j)
                {
                    pair<Polygon*, int> elem(polyShape, addCrns[j]);
                    m_layerReformEvents.quickInsert(elem);
                    polyShape->oilInCorner(addCrns[j])->isInReformVec(true);
                }
            }
        }
    }
    m_drainageEvents.sortEvents();
    m_layerReformEvents.sortEvents();

    if(m_injAtEntryRes)
    {
        for(int inT = 0; inT < m_rockLattice[0]->connectionNum(); ++inT)
        {
            coalesceOil(m_rockLattice[0]->connection(inT));
        }
    }

    if(m_injAtExitRes)
    {
        for(int outT = 0; outT < m_rockLattice[m_numPores+1]->connectionNum(); ++outT)
        {
            coalesceOil(m_rockLattice[m_numPores+1]->connection(outT));
        }
    }

    if(m_sourceNode != 0)
    {
        for(int sourceT = 0; sourceT < m_rockLattice[m_sourceNode]->connectionNum(); ++sourceT)
        {
            coalesceOil(m_rockLattice[m_sourceNode]->connection(sourceT));
        }
    }

    m_amottDataDrainage[0] = m_satWater;
    m_amottDataDrainage[1] = -1.0;
    m_usbmDataDrainage.clear();
    recordUSBMData(true);

    if(m_createDrainList)
    {
        ostringstream fileName;
        fileName << "fill_draincycle_" << m_commonData->drainageCycle() << ".m";
        m_drainListOut.open(fileName.str().c_str());

        m_drainListOut << "% The backbone identifies which pores/throats are oil filled at the start of drainage." << endl
            << "% The first row is pore/throat index, followed by 1 for pores and 0 for thoats." << endl;

        m_drainListOut << "backbone = [";
        for(size_t i = 0; i < m_rockLattice.size(); ++i)
        {
            if(!m_rockLattice[i]->isExitOrEntryRes() && m_rockLattice[i]->shape()->bulkOil())
            {
                bool isAPore(dynamic_cast< Pore* >(m_rockLattice[i]) != 0);
                m_drainListOut << m_rockLattice[i]->orenIndex() << ", ";
                m_drainListOut << isAPore << "; ..." << endl; 
            }
        }
        m_drainListOut << "];" << endl;

        m_drainListOut << endl << "% The filling list identifies the order through which pores/throats get filled by oil" << endl;
              
        m_drainListOut << "fill = [";
    }
}

/////////////////////////////////////////////////////////////////////////////////////////
// Continue injecting oil until a target saturation is reached, draiange queue is empty
// or capillary pressure target is reached. Writing results to output file. Oil injection
// needs to occur at least at one face.
/////////////////////////////////////////////////////////////////////////////////////////
void Netsim::drainage(double& targetWaterSat, double& targetPc, double& krw, double& kro, 
                             double& resIdx, bool& residualSat)
{
    clock_t startDrainage(clock());
    ostringstream out;
    out.flags(ios::showpoint);
    out.precision(3);

    if(targetWaterSat > m_satWater || targetPc < m_cappPress)
    {
        out << "================================================="              << endl
            << "Nothing to be done:"                                            << endl;

        if(targetWaterSat > m_satWater)
            out << "Target water saturation (" << targetWaterSat << ")"         << endl           
                << "is higher than current saturation (" << m_satWater << ")"   << endl;
        else
            out << "Target capillary pressure (" << targetPc << ")"             << endl           
                << "is lower than current pressure (" << m_cappPress << ")"  << endl;
        
        out << "=================================================="             << endl;
        
        writePrtData(out);
        targetWaterSat = m_satWater;
        targetPc = m_cappPress;
        krw = m_relPermWater;
        kro = m_relPermOil;
        resIdx = m_resistivityIdx;
        return;
    }    
    
    out << "Sw = "<< setw(6) << m_satWater << " -> "; 

    m_boundPress = 1.0E21;
    singleWaterDisplacement(targetWaterSat, targetPc, residualSat, out);
    recordRes(m_calcRelPerm, m_calcResIdx);

    writePrtData(out);
    m_cpuTimeTotal += cpuTimeElapsed(startDrainage);

    targetWaterSat = m_satWater;
    targetPc = m_cappPress;
    krw = m_relPermWater;
    kro = m_relPermOil;
    resIdx = m_resistivityIdx;
}

///////////////////////////////////////////////////////////////////////////////////////////
// Do a single displacement with oil and recalculate saturation and relative permeability
//////////////////////////////////////////////////////////////////////////////////////////
void Netsim::singleWaterDisplacement(double& targetWaterSat, double& targetPc, bool& residualSat, ostream& out)
{
    int numSteps(0), totNumFill(0);
    int fillTarget = max(m_minNumFillings, 
        static_cast< int >(m_initStepSize*(m_numPores+m_numThroats)*(m_satWater-targetWaterSat)));

    while(m_satWater > targetWaterSat && m_cappPress < targetPc && !m_drainageEvents.empty())
    {
        double oldCappPress(m_cappPress);
        double oldWaterSat(m_satWater);
        int numInv(0), invInsideBox(0);
        bool insideBox(false);

        while(invInsideBox < fillTarget && !m_drainageEvents.empty() && m_cappPress < targetPc) 
        {                                                                       
            popAndUpdateDrainVec(insideBox);    
            m_commonData->datumCappPress(m_cappPress);
            ++numInv;
            if(insideBox) ++invInsideBox;
            if(m_cappPress*oldCappPress < 0.0) 
            {
                ostringstream outXO;
                outXO.flags(ios::showpoint);
                outXO.precision(3);
                
                checkForUnstableConfigs();
                recordAmottData(true);
                outXO << "Pc cross over at Sw = " << setw(6) << m_satWater << "; "; 
                solveLaplaceEq(m_calcRelPerm, m_calcResIdx, outXO);
                recordRes(m_calcRelPerm, m_calcResIdx);
                writePrtData(outXO);
            }
            oldCappPress = m_cappPress;
            
            while(!m_drainageEvents.empty() && m_stableFilling && 
                m_cappPress > m_drainageEvents.peek()->shape()->gravCorrectedEntryPress())
            {                                       
                popAndUpdateDrainVec(insideBox);
                ++numInv;                                                   
                if(insideBox) ++invInsideBox;
            }
        }
 
        ++numSteps;        
        checkForUnstableConfigs();
        calculateWaterSat();
        recordUSBMData(true);
        fillTarget = max(m_minNumFillings, min(static_cast< int >(fillTarget*m_maxFillIncrease), 
            static_cast< int >(m_extrapCutBack*(invInsideBox/(m_satWater-oldWaterSat))*(targetWaterSat-m_satWater))));
        totNumFill += numInv;
    }

    residualSat = m_drainageEvents.empty();     
    m_maxCappPress = max(m_cappPress, m_maxCappPress);    

    out << setw(6) << m_satWater << "; " 
        << setw(3) << numSteps << " steps; " 
        << setw(5) << totNumFill << " fillings; "; 
    
    solveLaplaceEq(m_calcRelPerm, m_calcResIdx, out);
    m_totNumFillings += totNumFill;
}

//////////////////////////////////////////////////////////////////////////////////////////////
// Pops the elements with the highest priority acording to the compare function and updates
// that element which now contains oil. New elements now available for injection are inserted
// into the queue. The entry pressure rquired to do that invasion is returned.
/////////////////////////////////////////////////////////////////////////////////////////////
void Netsim::popAndUpdateDrainVec(bool& insideBox)
{
    popAndUpdateReformVec();
    if(m_drainageEvents.empty()) return;
       
    RockElem *currElem = m_drainageEvents.pop();    
    m_currentPc = currElem->shape()->gravCorrectedEntryPress();
    m_cappPress = max(m_cappPress, m_currentPc);
    m_boundPress = min(m_boundPress, m_currentPc);
    
    currElem->isInOilFloodVec(false);
    bool didContainOil(currElem->shape()->containsOil());
    currElem->drain();
    bool doesContainWat(currElem->shape()->containsWat());

    for(int i = 0; i < currElem->connectionNum(); ++i)
    {
        RockElem *conn = currElem->connection(i);
        if(!conn->isExitOrEntryRes())
        {
            if(conn->shape()->bulkFluid() == m_water)
            {
                if(conn->shape()->allOilLayers() || !doesContainWat)
                {
                    trappWaterCheck(conn, bulkBlob);
                }
                if(!doesContainWat && conn->shape()->allOilLayers())
                {
                    trappWaterCheck(conn, filmBlob);
                }
                updateElemDrainage(conn);
                addElemToDrainEvents(conn);
                addElemToReformEvents(conn);
            }
            else if(!doesContainWat && conn->shape()->containsWat())
            {
                trappWaterCheck(conn, filmBlob);
            }
        }
    }
    
    if(!didContainOil) 
    {
        for(int j = 0; j < currElem->connectionNum(); ++j)
        {
            coalesceOil(currElem->connection(j));
        }
    }

    if(m_createDrainList)
    {
        bool isAPore(dynamic_cast< Pore* >(currElem) != 0);
        m_drainListOut << currElem->orenIndex() << ", ";
        m_drainListOut << isAPore << "; ..." << endl;
    }

    clearTrappedWat();
    insideBox = currElem->isInsideSatBox();
}

void Netsim::popAndUpdateReformVec()
{
    double nextEventPc = m_drainageEvents.peek()->shape()->gravCorrectedEntryPress();
    while(!m_layerReformEvents.empty() && 
        m_layerReformEvents.peek().first->oilInCorner(m_layerReformEvents.peek().second)->gravCorrectedCollapsePc()
        < nextEventPc)
    {
        pair<Polygon*, int> polyEvent = m_layerReformEvents.pop();
        assert(!polyEvent.first->parent()->trappedOil());
        assert(!polyEvent.first->bulkOil());

        m_currentPc = polyEvent.first->reformOilLayer(polyEvent.second)+polyEvent.first->gravityCorrection();
        m_cappPress = max(m_currentPc, m_cappPress);
        polyEvent.first->addFillingEventToHistory(7);   // See shape.h for event indices
        assert(m_currentPc < nextEventPc);
        
        if(polyEvent.second == 0)                       // Sharpest corner => new oil connection
        {                                               // need to check for oil coalescence
            updateElemDrainage(polyEvent.first->parent());
            addElemToDrainEvents(polyEvent.first->parent());
            for(int i = 0; i < polyEvent.first->parent()->connectionNum(); ++i)
            {
                coalesceOil(polyEvent.first->parent()->connection(i));
                addElemToReformEvents(polyEvent.first->parent()->connection(i));
            }
        }
        else if(polyEvent.second == polyEvent.first->numCorners()-1)    // Oblique corner => water in center and        
        {                                                               // corner is separated, check for trapping            
            trappWaterCheck(polyEvent.first->parent(), filmBlob);    
            trappWaterCheck(polyEvent.first->parent(), bulkBlob);
        }

        clearTrappedWat();
        if(m_drainageEvents.empty()) return;
        nextEventPc = m_drainageEvents.peek()->shape()->gravCorrectedEntryPress();
    }
}

void Netsim::trappWaterCheck(RockElem* elem, FluidBlob startPt)
{
    vector< pair<RockElem*, FluidBlob> > trappingStorage;
    double gravCorr(elem->shape()->gravityCorrection());
    elem->checkForWatTrapping(m_cappPress-gravCorr, startPt, trappingStorage, m_cpuTimeTrapping, m_trappingCriteria);
    if(!trappingStorage.empty())
    {
        for(size_t i = 0; i < trappingStorage.size(); ++i)
        {
            updateLayerReform(trappingStorage[i].first, m_cappPress);
            addElemToReformEvents(trappingStorage[i].first);
        }
        sort(trappingStorage.begin(), trappingStorage.end(), TrappingWatStorageCmp());  // Having it sorted hels us when we want to coalesce the blob
        m_commonData->addTrappedRegionWat(trappingStorage);
    }
}

void Netsim::coalesceOil(RockElem* elem)
{
    clock_t startCoalesce(clock());
    pair<int, double> trapOilInfo =  elem->trappingOil();
    trapOilInfo.second += elem->shape()->gravityCorrection();
    if(trapOilInfo.first > -1)                                                      // We need to untrap this oil blob
    {
        vector<RockElem*> newElems = m_commonData->trappedRegionsOil(trapOilInfo.first);
        SortedEvents< ThreeSome<Polygon*, int, double>, CoalesceWatFillCmp > waterFillingEvents;
        SortedEvents< ThreeSome<Polygon*, int, double>, CoalesceOilFillCmp > oilFillingEvents;
        if(trapOilInfo.second > m_cappPress) m_commonData->injectant(m_water);      // Gonna drop pressure in oil blob...

        for(size_t i = 0; i < newElems.size(); ++i)                                 // First stage: check for unstable configurations
        {                                                                           // when the pressure in the coalesced blob is 
            Polygon* polyShape = dynamic_cast< Polygon* >(newElems[i]->shape());    // equilibrated to the global pressure            
            newElems[i]->unTrapOil();
            if(polyShape)
            {               
                double gravCorr(polyShape->gravityCorrection());
                updateLayerReform(newElems[i], m_cappPress);
                if(trapOilInfo.second > m_cappPress)
                {                                                                               
                    polyShape->checkForUnstableWaterConfigs(waterFillingEvents, m_cappPress-gravCorr);   // => water snap off + oil layer collapse 
                }
                else
                {                                                                               
                    polyShape->checkForUnstableOilConfigs(oilFillingEvents, m_cappPress-gravCorr);       // => oil snap off     
                }
            }
        }
        
        if(trapOilInfo.second > m_cappPress)                            // Second stage: Once all possible unstable configurations
        {                                                               // have been identified it's time to increase/reduce 
            waterFillingEvents.sortEvents();                            // pressure in the blob. Dropping pressure will in fact be
            dropPressureInCoalescedBlob(waterFillingEvents, false);     // water injection process (water snap off, layer collapse).
        }                                                               // Increasing pressure is simpler in that the only change 
        else                                                            // that might occur are oil snap off. 
        {
            oilFillingEvents.sortEvents();
            increasePressureInCoalescedBlob(oilFillingEvents, false); 
        }
        
        for(size_t elm = 0; elm < newElems.size(); ++elm)                           // Third stage: Once the pressure in the blob 
        {                                                                           // has been equilibrated it is time to calculate 
            Polygon* polyShape = dynamic_cast< Polygon* >(newElems[elm]->shape());  // entry pressures and pin interfaces in their 
            if(polyShape)                                                           // correct places.
            {                
                double gravCorr(polyShape->gravityCorrection());
                polyShape->updateState(m_cappPress-gravCorr); 
                
                if(trapOilInfo.second > m_cappPress) 
                {
                    polyShape->initDrainage(m_cappPress-gravCorr);
                }
                else if(polyShape->oilInCorner(0)->exists()) 
                {
                    double snapOffPrs = polyShape->calcSnapOffPressureDrain(); 
                    polyShape->snapOffPrs(snapOffPrs);
                    updateElemDrainage(newElems[elm]);
                }
            }
        }
        if(trapOilInfo.second > m_cappPress) m_commonData->injectant(m_oil);    // Resume oil injection
        
        for(size_t j = 0; j < newElems.size(); ++j)                         // Final stage: Now that all is peach, we can
        {                                                                   // possibly start adding new elements to the 
            if(!newElems[j]->trappedOil())                                  // displacement vectors
            {
                addElemToReformEvents(newElems[j]);           
                addElemToDrainEvents(newElems[j]);  
                for(int k = 0; k < newElems[j]->connectionNum(); ++k)
                {
                    addElemToReformEvents(newElems[j]->connection(k));
                    addElemToDrainEvents(newElems[j]->connection(k));
                }
            }
        }
        m_commonData->removeTrappedRegionOil(trapOilInfo.first);
    }
    m_cpuTimeCoal += cpuTimeElapsed(startCoalesce);
}

void Netsim::updateElemDrainage(RockElem* elem)
{
    double entryPrs(0.0);
    if(elem->shape()->updateDrainEntryPrs(entryPrs))
    {
        if(elem->isInOilFloodVec())
        {
            m_drainageEvents.remove(elem);
            elem->shape()->entryPress(entryPrs);
            m_drainageEvents.insert(elem);
        }
        else
        {
            elem->shape()->entryPress(entryPrs);
        }
    }
}

void Netsim::updateLayerReform(RockElem* elem, double cappPress)
{
    Polygon* polyShape = dynamic_cast< Polygon* >(elem->shape());
    if(polyShape)
    {
        for(int i = 0; i < polyShape->numCorners(); ++i)
        {
            pair< Polygon*, int > elem(polyShape, i);
            if(polyShape->oilInCorner(i)->isInReformVec()) 
            {
                m_layerReformEvents.remove(elem);
            }
        }
                        
        polyShape->updateFilmsForTrapping(cappPress-polyShape->gravityCorrection());

        for(int j = 0; j < polyShape->numCorners(); ++j)
        {
            pair< Polygon*, int > elem(polyShape, j);
            if(polyShape->oilInCorner(j)->isInReformVec()) m_layerReformEvents.insert(elem);
        }
    }
}

/////////////////////////////////////////////////////////////////////////////////
// Done with drainage => now clean up
/////////////////////////////////////////////////////////////////////////////////
void Netsim::finaliseDrainage()
{
    ostringstream out;
    out.flags(ios::showpoint);
    out.flags(ios::fixed);
    out.precision(3);

    if(m_maxOilFlowErr > 0.1 || m_maxWatFlowErr > 0.1 || m_maxResIdxErr > 0.1)
    {
        out << endl
            << "===================================================="   << endl
            << "Warning: For some rel perm calculations there were"     << endl
            << "more than 10% difference between flowrates computed"    << endl
            << "at the inlet and outlet. Perhaps try to reduce"         << endl
            << "solver tolerance"                                       << endl
            << "Max water flowrate error:    "  << m_maxWatFlowErr      << endl
            << "Max oil flowrate error:      "  << m_maxOilFlowErr      << endl
            << "Max resistivity index error: "  << m_maxResIdxErr       << endl
            << "===================================================="   << endl
            << endl;
    }

    out << endl
        << "=====================  Drainage  ====================="                                     << endl            
        << "Total elapsed time for drainage:         "  << m_cpuTimeTotal                               << endl
        << "Solving for water relative permeability: "  << m_cpuTimeKrw                                 << endl
        << "Solving for oil relative permeability:   "  << m_cpuTimeKro                                 << endl
        << "Solving for resistivity index:           "  << m_cpuTimeResIdx                              << endl   
        << "Identifying trapped elements:            "  << m_cpuTimeTrapping                            << endl
        << "Coalesceing trapped oil:                 "  << m_cpuTimeCoal                                << endl
        << "Max water flowrate error:                "  << m_maxWatFlowErr*100.0 << " %"                << endl
        << "Max oil flowrate error:                  "  << m_maxOilFlowErr*100.0 << " %"                << endl
        << "Max resistivity index error:             "  << m_maxResIdxErr*100.0 << " %"                 << endl
        << endl
        << "===================  Network State  ==================="                                    << endl
        << "Maximum capillary pressure reached (Pa): "  << m_maxCappPress                               << endl
        << "Water saturation:                        "  << m_satWater                                   << endl
        << "Number of elements invaded:              "  << m_totNumFillings                             << endl
        << "Remaining uninvaded elements:            "  << (int)m_rockLattice.size()-2-m_totNumFillings << endl
        << "Number of trapped water regions:         "  << (int)m_commonData->numTrappedWatRegions()    << endl
        << "Total number of trapped water elements:  "  << (int)m_commonData->numTrappedWatElems()      << endl
        << "Number of trapped oil regions:           "  << (int)m_commonData->numTrappedOilRegions()    << endl
        << "Total number of trapped oil elements:    "  << (int)m_commonData->numTrappedOilElems()      << endl
        << endl;

    m_amottDataDrainage[2] = m_satWater;
    if(m_cappPress < 0.0 && m_amottDataDrainage[1] < 0.0)
        m_amottOilIdx = 1.0;
    else if(m_amottDataDrainage[1] < 0.0)
        m_amottOilIdx = 0.0;
    else
    {
        m_amottOilIdx = (m_amottDataDrainage[0]-m_amottDataDrainage[1])/
            (m_amottDataDrainage[0]-m_amottDataDrainage[2]);
    }

    if(m_commonData->drainageCycle() > 1)
    {
        out << endl
            << "=================  Wettability State  ================"                     << endl
            << "Amott water index, Iw:                  " << m_amottWaterIdx                << endl
            << "Amott oil index, Io:                    " << m_amottOilIdx                  << endl
            << "Amott wettability index, I = Iw - Io:   " << m_amottWaterIdx-m_amottOilIdx  << endl
            << "USBM wettability index:                 " << calcUSBMindex()                << endl
            << endl;
    }

    writePrtData(out);    
    writeResultData(m_calcRelPerm, m_calcResIdx, m_drainResultsOut);
    if(m_prtPressureProfile && m_calcRelPerm)
    {
        writePrsProfileData(m_water, m_drainResultsOut);
        writePrsProfileData(m_oil, m_drainResultsOut);
    }

    if(m_createDrainList)
    {
        m_drainListOut << "];" << endl;
        m_drainListOut.close();
    }

    if(m_solver != 0)
    {
        delete m_solver;
        m_solver = 0;
    }

    m_drainageEvents.clear();
    m_layerReformEvents.clear();
}


//////////////////////////////////////////////////////////////////////////////////////////
// At the end of draiange imbibition displacement is initialized as max Pc is set. This 
// involves determening entry pressures for all elements. 
//////////////////////////////////////////////////////////////////////////////////////////
void Netsim::initImbibition(bool calcRelPerm, bool calcResIdx, InputData& input)
{
    m_commonData->incrementImbCycle();
    m_commonData->injectant(m_water);
    m_commonData->drainagePhase(false);
    
    m_cpuTimeTotal = 0.0;
    m_cpuTimeCoal = 0.0;
    m_cpuTimeKrw = 0.0;
    m_cpuTimeKro = 0.0;
    m_cpuTimeTrapping = 0.0;
    m_cpuTimeResIdx = 0.0;
    m_totNumFillings = 0;
    m_minCappPress = m_cappPress;
    m_boundPress = m_cappPress;
    m_commonData->maxDatumPc(m_maxCappPress);

    m_maxOilFlowErr = 0.0;
    m_maxWatFlowErr = 0.0;
    m_maxResIdxErr = 0.0;

    if(m_commonData->imbibitionCycle() == 2) applyFractionalWetting(input);

    if(calcRelPerm || calcResIdx) 
    {
        ostringstream matDBGFileName;
        matDBGFileName << m_matrixFileName << "_imbcycle_" << m_commonData->imbibitionCycle();
        m_solver = new Solver(m_rockLattice, m_krInletBoundary, m_krOutletBoundary, m_numPores+1, m_maxNonZeros, 
            matDBGFileName.str(), m_writeSlvMatrixAsMatlab);
    }

    ostringstream tmp;
    if(!m_calcRelPerm) solveLaplaceEq(calcRelPerm, false, tmp);
    if(!m_calcResIdx) solveLaplaceEq(false, calcResIdx, tmp);

    recordRes(calcRelPerm, calcResIdx);    
    m_calcRelPerm = calcRelPerm;
    m_calcResIdx = calcResIdx;

    if(m_injAtExitRes && m_rockLattice[m_numPores+1]->shape()->bulkFluid() != m_water) 
        m_rockLattice[m_numPores+1]->imbide();
    else if(!m_injAtExitRes && m_rockLattice[m_numPores+1]->shape()->bulkFluid() == m_water) 
        m_rockLattice[m_numPores+1]->drain();
   
    if(m_injAtEntryRes && m_rockLattice[0]->shape()->bulkFluid() != m_water) 
        m_rockLattice[0]->imbide();
    else if(!m_injAtEntryRes && m_rockLattice[0]->shape()->bulkFluid() == m_water) 
        m_rockLattice[0]->drain();


    if(m_sourceNode != 0 && m_rockLattice[m_sourceNode]->shape()->bulkFluid() != m_water) 
        m_rockLattice[m_sourceNode]->imbide();

    m_imbibitionEvents.clear();
    m_layerCollapseEvents.clear();
       
    for(int i = 1; i < static_cast< int >(m_rockLattice.size()); ++i)
    {
        if(i != m_numPores + 1 && m_rockLattice[i]->connectedToNetwork())
        {            
            double gravCorr(m_rockLattice[i]->shape()->gravityCorrection());
            m_rockLattice[i]->shape()->initImbibition(m_cappPress-gravCorr);

            if(m_rockLattice[i]->addToEventVec(m_water))
            {
                m_imbibitionEvents.quickInsert(m_rockLattice[i]);
                m_rockLattice[i]->isInWatFloodVec(true);
            }
            
            Polygon* polyShape = 0;
            vector<int> addCrns;
            if(m_rockLattice[i]->addToLayerVec(m_water, &polyShape, addCrns, m_cappPress-gravCorr))
            {
                assert(!addCrns.empty());
                for(size_t j = 0; j < addCrns.size(); ++j)
                {
                    pair<Polygon*, int> elem(polyShape, addCrns[j]);
                    assert(!polyShape->oilInCorner(addCrns[j])->isInCollapseVec());
                    m_layerCollapseEvents.quickInsert(elem);
                    polyShape->oilInCorner(addCrns[j])->isInCollapseVec(true);
                }
            }
        }
    }
    
    m_imbibitionEvents.sortEvents();
    m_layerCollapseEvents.sortEvents();
    
    if(m_injAtEntryRes)
    {
        for(int inT = 0; inT < m_rockLattice[0]->connectionNum(); ++inT)
        {
            coalesceWater(m_rockLattice[0]->connection(inT), filmBlob);
            coalesceWater(m_rockLattice[0]->connection(inT), bulkBlob);
        }
    }

    if(m_injAtExitRes)
    {
        for(int outT = 0; outT < m_rockLattice[m_numPores+1]->connectionNum(); ++outT)
        {
            coalesceWater(m_rockLattice[m_numPores+1]->connection(outT), filmBlob);
            coalesceWater(m_rockLattice[m_numPores+1]->connection(outT), bulkBlob);
        }
    }

    if(m_sourceNode != 0)
    {
        for(int sourceT = 0; sourceT < m_rockLattice[m_sourceNode]->connectionNum(); ++sourceT)
        {
            coalesceWater(m_rockLattice[m_sourceNode]->connection(sourceT), filmBlob);
            coalesceWater(m_rockLattice[m_sourceNode]->connection(sourceT), bulkBlob);
        }
    }

    m_amottDataImbibition[0] = m_satWater;
    m_amottDataImbibition[1] = -1.0;

    m_usbmDataImbibition.clear();
    recordUSBMData(false);

    if(m_createImbList)
    {
        ostringstream fileName;
        fileName << "fill_imbcycle_" << m_commonData->imbibitionCycle() << ".m";
        m_imbListOut.open(fileName.str().c_str());

        m_imbListOut << "% The backbone identifies which pores/throats are water filled at the start of water flooding." << endl
            << "% The first row is pore/throat index, followed by 1 for pores and 0 for thoats." << endl;

        m_imbListOut << "backbone = [";
        for(size_t i = 0; i < m_rockLattice.size(); ++i)
        {
            if(!m_rockLattice[i]->isExitOrEntryRes() && !m_rockLattice[i]->shape()->bulkOil())
            {
                bool isAPore(dynamic_cast< Pore* >(m_rockLattice[i]) != 0);
                m_imbListOut << m_rockLattice[i]->orenIndex() << ", ";
                m_imbListOut << isAPore << "; ..." << endl; 
            }
        }
        m_imbListOut << "];" << endl; 
        
        m_imbListOut << endl << "% The filling list identifies the order through which pores/throats get filled by water" << endl;
        m_imbListOut << "fill = [";
    }
}


///////////////////////////////////////////////////////////////////////////////////////////////////////////
// Inject water (imbibition) until target water saturation is reached or alternatively that the queue
// containing elements to be popped is empty. After drainage throats connected to the outlets have a 
// maximum of one oil neighbour. If all elements were drained water injection will then occur at both
// faces. We can remove an injection face by increasing the oil neighbour count in throats connected to
// that that face. If we remove both faces and all elements were previously drained, the firts imbibition
// event will be a snap off.
///////////////////////////////////////////////////////////////////////////////////////////////////////////
void Netsim::injectWater(double& targetWaterSat, double& targetPc, double& krw, double& kro, double& resIdx, bool& residualSat)
{
    clock_t startImbibition(clock());
    ostringstream out;
    out.flags(ios::showpoint);
    out.precision(3);
    
    if(targetWaterSat < m_satWater || targetPc > m_cappPress)
    {
        out << "================================================="              << endl
            << "Nothing to be done:"                                            << endl;

        if(targetWaterSat < m_satWater)
            out << "Target water saturation (" << targetWaterSat << ")"         << endl           
                << "is lower than current saturation (" << m_satWater << ")"    << endl;
        else
            out << "Target capillary pressure (" << targetPc << ")"             << endl           
                << "is higher than current pressure (" << m_cappPress << ")"    << endl;
        
        out << "=================================================="             << endl;
        
        writePrtData(out);
        targetWaterSat = m_satWater;
        targetPc = m_cappPress;
        krw = m_relPermWater;
        kro = m_relPermOil;
        resIdx = m_resistivityIdx;
        return;
    }    
    
    out << "Sw = "<< setw(6) << m_satWater << " -> "; 

    m_boundPress = -1.0E21;
    singleOilDisplacement(targetWaterSat, targetPc, residualSat, out);
    recordRes(m_calcRelPerm, m_calcResIdx);

    writePrtData(out);
    m_cpuTimeTotal += cpuTimeElapsed(startImbibition);

    targetWaterSat = m_satWater;
    targetPc = m_cappPress;
    krw = m_relPermWater;
    kro = m_relPermOil;
    resIdx = m_resistivityIdx;
}

///////////////////////////////////////////////////////////////////////////////////////////////
// Do a single displacement with water and recalculate saturation and relative permeability.
// We do not allow for empty filling Vec as this just might create a mess in imbibition. The
// reason is that as Pc increases (negatively) more and more regions might get trapped as oil
// layers collapse, and we don't want to spend huge amounts of time checking on this.
///////////////////////////////////////////////////////////////////////////////////////////////
void Netsim::singleOilDisplacement(double& targetWaterSat, double& targetPc, bool& residualSat, ostream& out)
{
    int numSteps(0), totNumFill(0);
    int fillTarget = max(m_minNumFillings, 
        static_cast< int >(m_initStepSize*(m_numPores+m_numThroats)*(targetWaterSat-m_satWater)));
    
    while(m_satWater < targetWaterSat && m_cappPress > targetPc && !m_imbibitionEvents.empty())
    {
        double oldCappPress(m_cappPress);
        double oldWaterSat(m_satWater);
        int numInv(0), invInsideBox(0);
        bool insideBox(false);

        while(invInsideBox < fillTarget && !m_imbibitionEvents.empty() && m_cappPress > targetPc) 
        {                                                                       
            popAndUpdateImbVec(insideBox);
            m_commonData->datumCappPress(m_cappPress);                   // Use local maxima for pc           
            ++numInv;
            if(insideBox) ++invInsideBox;
            if(m_cappPress*oldCappPress < 0.0) 
            {
                ostringstream outXO;
                outXO.flags(ios::showpoint);
                outXO.precision(3);
                
                checkForUnstableConfigs();
                recordAmottData(false);
                cout << endl << "trapped water wet oil " << trappedWWoil() << endl;
                outXO << "Pc cross over at Sw = " << setw(6) << m_satWater << "; "; 
                solveLaplaceEq(m_calcRelPerm, m_calcResIdx, outXO);
                recordRes(m_calcRelPerm, m_calcResIdx);
                writePrtData(outXO);
            }
            oldCappPress = m_cappPress;
            
            while(!m_imbibitionEvents.empty() && m_stableFilling &&
                m_cappPress < m_imbibitionEvents.peek()->shape()->gravCorrectedEntryPress())
            {                                       
                popAndUpdateImbVec(insideBox);
                ++numInv;                                                   
                if(insideBox) ++invInsideBox;
            }        
        }
        
        ++numSteps; 
        checkForUnstableConfigs();
        calculateWaterSat();
        recordUSBMData(false);
        fillTarget = max(m_minNumFillings, min(static_cast< int >(fillTarget*m_maxFillIncrease), 
            static_cast< int >(m_extrapCutBack*(invInsideBox/(m_satWater-oldWaterSat))*(targetWaterSat-m_satWater))));
        totNumFill += numInv;
    }

    residualSat = m_imbibitionEvents.empty();
    m_minCappPress = min(m_cappPress, m_minCappPress);    

    out << setw(6) << m_satWater << "; " 
        << setw(3) << numSteps << " steps; " 
        << setw(5) << totNumFill << " fillings; "; 
    
    solveLaplaceEq(m_calcRelPerm, m_calcResIdx, out);
    m_totNumFillings += totNumFill;    
}

double Netsim::trappedWWoil() const
{
    double volOil(0.0);
    for(size_t i = 0; i < m_rockLattice.size(); ++i)
    {
        RockElem* el = m_rockLattice[i];
        if(el->isInsideSatBox() && 
            el->shape()->waterWet() &&
            el->shape()->bulkOil() &&
            el->trappedOil())
        {
            volOil += el->oilVolume();
        }
    }
    return volOil / (m_netVolume + m_clayVolume);
}

//////////////////////////////////////////////////////////////////////////////////////////
// Since a single element have many possible imbibition entry pressures, a priority Vec
// is unsuitable for keeping track of the filling sequence. Hence we use a sorted list.
// For each imbibition event the entry pressures for the neighbours might change since 
// the number of oil filled neighbours have decreased. We need to remove these elements 
// from the list and then reinsert them with the new entry pressure, while keeping the
// computaional cost as low as possible. Just resorting the list would cost O(N) whreas 
// extracting the elements, O(logN), and then reinserting them, O(logN), have a total 
// cost of 2*connectionNum*O(logN). For large networks this should be cheaper.
/////////////////////////////////////////////////////////////////////////////////////////
void Netsim::popAndUpdateImbVec(bool& insideBox)
{
    popAndUpdateCollapseVec();
    if(m_imbibitionEvents.empty()) return;

    RockElem *currElem = m_imbibitionEvents.pop();    
    m_currentPc = currElem->shape()->gravCorrectedEntryPress();
    m_cappPress = min(m_cappPress, m_currentPc);
    m_boundPress = max(m_boundPress, m_currentPc);
    currElem->isInWatFloodVec(false);

    bool didContainWat(currElem->shape()->containsWat());
    currElem->imbide();
    bool doesContainOil(currElem->shape()->containsOil());

    for(int i = 0; i < currElem->connectionNum(); ++i)
    {
        RockElem *conn = currElem->connection(i);
        if(!conn->isExitOrEntryRes())
        {
            if(!doesContainOil) trappOilCheck(conn);
            
            if(conn->shape()->bulkFluid() == m_oil)
            {
                updateElemImbibition(conn);
                addElemToImbEvents(conn);
            }
        }
    }

    if(currElem->trappedWat(filmBlob) && !currElem->shape()->allOilLayers())
        coalesceWater(currElem, filmBlob);

    for(int j = 0; j < currElem->connectionNum(); ++j)
    {
        if(!didContainWat) coalesceWater(currElem->connection(j), filmBlob);
        coalesceWater(currElem->connection(j), bulkBlob);
    }

    addElemToCollapseEvents(currElem); 

    if(m_createImbList)
    {
        bool isAPore(dynamic_cast< Pore* >(currElem) != 0);
        m_imbListOut << currElem->orenIndex() << ", ";
        m_imbListOut << isAPore << "; ..." << endl; 
    }

    clearTrappedOil();
    insideBox = currElem->isInsideSatBox();    
}

void Netsim::popAndUpdateCollapseVec()
{
    double nextEventPc = m_imbibitionEvents.peek()->shape()->gravCorrectedEntryPress();
    
    while(!m_layerCollapseEvents.empty() && 
        m_layerCollapseEvents.peek().first->oilInCorner(m_layerCollapseEvents.peek().second)->gravCorrectedCollapsePc()
        > nextEventPc)
    {
        pair<Polygon*, int> polyEvent = m_layerCollapseEvents.pop();
        double gravCorr(polyEvent.first->gravityCorrection());
        m_currentPc = polyEvent.first->collapseOilLayer(polyEvent.second)+gravCorr;

        assert(!polyEvent.first->parent()->trappedOil());
     
        m_cappPress = min(m_cappPress, m_currentPc);
        polyEvent.first->addFillingEventToHistory(6);
        assert(m_currentPc > nextEventPc);

        if(polyEvent.second == 0)                       // Sharpest corner => no more oil, check
        {                                               // for oil trapping
            for(int i = 0; i < polyEvent.first->parent()->connectionNum(); ++i)
            {
                trappOilCheck(polyEvent.first->parent()->connection(i));
            }
        }
        else if(polyEvent.second == polyEvent.first->numCorners()-1)    
        {                                                           // Oblique corner => water in center and       
            coalesceWater(polyEvent.first->parent(), filmBlob);     // corner is joined, check for coalescing    
            coalesceWater(polyEvent.first->parent(), bulkBlob);     
        }

        clearTrappedOil();
        if(m_imbibitionEvents.empty()) return;

        nextEventPc = m_imbibitionEvents.peek()->shape()->gravCorrectedEntryPress();
    }
}

void Netsim::coalesceWater(RockElem* elem, FluidBlob blob)
{
    clock_t startCoalesce(clock());
    pair<int, double> trapWatInfo =  elem->trappingWat(blob);
    trapWatInfo.second += elem->shape()->gravityCorrection();
    if(trapWatInfo.first > -1)                                                      // We need to untrap this water blob
    {
        vector< pair<RockElem*,FluidBlob> > newElems = m_commonData->trappedRegionsWat(trapWatInfo.first);
        SortedEvents< ThreeSome<Polygon*, int, double>, CoalesceWatFillCmp > waterFillingEvents;
        SortedEvents< ThreeSome<Polygon*, int, double>, CoalesceOilFillCmp > oilFillingEvents;
        if(trapWatInfo.second < m_cappPress) m_commonData->injectant(m_oil);            // Gonna increase pressure in water blob...

        for(size_t k = 0; k < newElems.size(); ++k) newElems[k].first->unTrapWat(newElems[k].second);
                
        Polygon* oldPolyShape = 0;
        for(size_t i = 0; i < newElems.size(); ++i)                                     // First stage: check for unstable configurations
        {                                                                               // when the pressure in the coalesced blob is 
            Polygon* polyShape = dynamic_cast< Polygon* >(newElems[i].first->shape());  // equilibrated with the rest            
            
            if(polyShape && polyShape != oldPolyShape)                                  
            {               
                double gravCorr(polyShape->gravityCorrection());
                updateLayerCollapse(newElems[i].first, m_cappPress);
                if(trapWatInfo.second > m_cappPress)
                {                                                                                       // which are water snap offs and
                    polyShape->checkForUnstableWaterConfigs(waterFillingEvents, m_cappPress-gravCorr);  // oil layer collapses
                }
                else
                {                                                                               
                    polyShape->checkForUnstableOilConfigs(oilFillingEvents, m_cappPress-gravCorr);      // which are oil snap offs
                }
            }
            oldPolyShape = polyShape;       // Prevent adding the same element twice to the event storage
        }
        
        if(trapWatInfo.second > m_cappPress)                            // Second stage: Once all possible unstable configurations
        {                                                               // have been identified it's time to increase/reduce 
            waterFillingEvents.sortEvents();                            // pressure in the blob. Dropping pressure will in fact be
            dropPressureInCoalescedBlob(waterFillingEvents, true);      // water injection process (water snap off, layer collapse).
        }                                                               // Increasing pressure is simpler in that the only change 
        else                                                            // that might occur are oil snap off. 
        {
            oilFillingEvents.sortEvents();
            increasePressureInCoalescedBlob(oilFillingEvents, true); 
        }
        
        oldPolyShape = 0;
        for(size_t elm = 0; elm < newElems.size(); ++elm)                                   // Third stage: Once the pressure in the blob 
        {                                                                                   // has been equilibrated it is time to calculate 
            Polygon* polyShape = dynamic_cast< Polygon* >(newElems[elm].first->shape());    // entry pressures and pin interfaces in their 
            if(polyShape && polyShape != oldPolyShape)                                      // correct places.
            {         
                double gravCorr(polyShape->gravityCorrection());
                polyShape->updateState(m_cappPress-gravCorr); 
                
                if(trapWatInfo.second < m_cappPress) 
                {
                    if(polyShape->parent()->isInWatFloodVec())
                    {
                        m_imbibitionEvents.remove(polyShape->parent());
                        polyShape->parent()->isInWatFloodVec(false);
                    }

                    for(int i = 0; i < polyShape->numCorners(); ++i)
                    {
                        if(polyShape->oilInCorner(i)->isInCollapseVec())
                        {
                            pair<Polygon*, int> gonner(polyShape, i); 
                            m_layerCollapseEvents.remove(gonner);
                            polyShape->oilInCorner(i)->isInCollapseVec(false);
                        }
                    }
                    
                    assert(!newElems[elm].first->isInWatFloodVec());
                    assert(!polyShape->oilInCorner(0)->isInCollapseVec());
                    polyShape->initImbibition(m_cappPress-gravCorr);
                }
            }
            oldPolyShape = polyShape;
        }
        if(trapWatInfo.second < m_cappPress) m_commonData->injectant(m_water);  // Resume water injection

        for(size_t j = 0; j < newElems.size(); ++j)                         // Final stage: Now that all is peach, we can
        {                                                                   // possibly start adding new elements to the 
            addElemToCollapseEvents(newElems[j].first);           
            addElemToImbEvents(newElems[j].first);    
            for(int k = 0; k < newElems[j].first->connectionNum(); ++k)
            {
                addElemToCollapseEvents(newElems[j].first->connection(k));
                addElemToImbEvents(newElems[j].first->connection(k));
            }
        }
        m_commonData->removeTrappedRegionWat(trapWatInfo.first);
    }
    m_cpuTimeCoal += cpuTimeElapsed(startCoalesce);
}

void Netsim::dropPressureInCoalescedBlob(SortedEvents< ThreeSome<Polygon*, int, double>, CoalesceWatFillCmp >& waterFillingEvents, bool watFloodCycle)
{
    while(!waterFillingEvents.empty())
    {
        ThreeSome<Polygon*, int, double> watEvent = waterFillingEvents.pop();
        double blobPc(watEvent.third());
        assert(blobPc > m_cappPress);
        if(!watEvent.first()->parent()->trappedOil())
        {
            if(watEvent.second() == -1 && !watEvent.first()->parent()->trappedWat(filmBlob))         // Water snap off
            {
                watEvent.first()->parent()->imbide(true);
                for(int i = 0; i < watEvent.first()->parent()->connectionNum(); ++i)
                {
                    RockElem *conn = watEvent.first()->parent()->connection(i);
                    if(!conn->isExitOrEntryRes() && conn->shape()->bulkFluid() == m_water && !watFloodCycle)
                    {
                        updateElemDrainage(conn);
                    }
                    else if(!conn->isExitOrEntryRes() && conn->shape()->bulkFluid() == m_oil && watFloodCycle)
                    {
                        updateElemImbibition(conn);
                    }
                }
            }
            else if(watEvent.second() >= 0)     // Oil layer collapse
            {
                if(watEvent.first()->oilInCorner(watEvent.second())->isInCollapseVec())
                {
                    pair< Polygon*, int > toBeRemoved(watEvent.first(), watEvent.second());
                    m_layerCollapseEvents.remove(toBeRemoved);
                }
                watEvent.first()->addFillingEventToHistory(6);
                watEvent.first()->collapseOilLayer(watEvent.second());
            }

            if(watEvent.second() <= 0 && !watFloodCycle)      // Element no longer contains oil => check for trapping
            {
                for(int i = 0; i < watEvent.first()->parent()->connectionNum(); ++i)
                { 
                    RockElem* conn = watEvent.first()->parent()->connection(i);
                    vector<RockElem*> trappingStorage;
                    double gravCorr(conn->shape()->gravityCorrection());
                    
                    TrappingCriteria trpCrit(escapeToEither);       // Anything connected to the entry reservoir is 
                    if(m_injAtEntryRes && !m_injAtExitRes)          // by default connected. ie we need less restrictive
                        trpCrit = escapeToInlet;                    // trapping criteria.
                    else if(!m_injAtEntryRes && m_injAtExitRes) 
                        trpCrit = escapeToOutlet;

                    conn->checkForOilTrapping(blobPc-gravCorr, trappingStorage, m_cpuTimeTrapping, trpCrit);
                    if(!trappingStorage.empty()) 
                    {
                        m_commonData->addTrappedRegionOil(trappingStorage);
                        for(size_t j = 0; j < trappingStorage.size(); ++j)
                        {
                            updateLayerReform(trappingStorage[j], blobPc);
                        }
                    }
                }
            }
            else if(watEvent.second() == watEvent.first()->numCorners()-1)      // New possible connection between water in corner
            {                                                                   // and that in the centre    
                coalesceWater(watEvent.first()->parent(), filmBlob);         
                coalesceWater(watEvent.first()->parent(), bulkBlob);     
            }
        }
    }
}

void Netsim::increasePressureInCoalescedBlob(SortedEvents< ThreeSome<Polygon*, int, double>, CoalesceOilFillCmp >& oilFillingEvents, bool watFloodCycle)
{
    while(!oilFillingEvents.empty())
    {
        ThreeSome<Polygon*, int, double> oilEvent = oilFillingEvents.pop();
        assert(oilEvent.second() == -1);
        double blobPc(oilEvent.third());
        assert(blobPc < m_cappPress);

        if(!oilEvent.first()->parent()->trappedOil() && !oilEvent.first()->parent()->trappedWat(bulkBlob))
        {       
            bool wasAllLayers(oilEvent.first()->allOilLayers());
            assert(oilEvent.first()->anyOilLayers());

            if(watFloodCycle)
            {
                for(int corn = 0; corn < oilEvent.first()->numCorners(); ++corn)
                {
                    if(oilEvent.first()->oilInCorner(corn)->isInCollapseVec())
                    {
                        pair<Polygon*, int> collapseEvent(oilEvent.first(), corn);
                        m_layerCollapseEvents.remove(collapseEvent);
                        oilEvent.first()->oilInCorner(corn)->isInCollapseVec(false);
                    }
                }
            }
            oilEvent.first()->parent()->drain();

            for(int i = 0; i < oilEvent.first()->parent()->connectionNum(); ++i)
            {
                RockElem *conn = oilEvent.first()->parent()->connection(i);
                if(!conn->isExitOrEntryRes() && conn->shape()->bulkFluid() == m_water && !watFloodCycle)
                {
                    updateElemDrainage(conn);
                }
                else if(!conn->isExitOrEntryRes() && conn->shape()->bulkFluid() == m_oil && watFloodCycle)
                {
                    updateElemImbibition(conn);
                }
            }
               
            if(wasAllLayers && watFloodCycle)                                           // Need to check for trapping in the bulk 
            {                                                                           // where water has been displaced by oil
                for(int i = 0; i < oilEvent.first()->parent()->connectionNum(); ++i)
                {
                    RockElem* conn = oilEvent.first()->parent()->connection(i);
                    vector< pair<RockElem*, FluidBlob> > trappingStorage;
                    double gravCorr(conn->shape()->gravityCorrection());

                    TrappingCriteria trpCrit(escapeToEither);       // Anything connected to the entry reservoir is 
                    if(m_injAtEntryRes && !m_injAtExitRes)          // by default connected. ie we need less restrictive
                        trpCrit = escapeToInlet;                    // trapping criteria.
                    else if(!m_injAtEntryRes && m_injAtExitRes) 
                        trpCrit = escapeToOutlet;

                    conn->checkForWatTrapping(blobPc-gravCorr, bulkBlob, trappingStorage, m_cpuTimeTrapping, trpCrit);
                    if(!trappingStorage.empty()) 
                    {
                        m_commonData->addTrappedRegionWat(trappingStorage);
                        for(size_t j = 0; j < trappingStorage.size(); ++j)
                        {
                            updateLayerCollapse(trappingStorage[j].first, blobPc);
                         }
                    }
                }
            }
        }
    }
}

void Netsim::trappOilCheck(RockElem* elem)
{
    vector<RockElem*> trappingStorage;
    double gravCorr(elem->shape()->gravityCorrection());
    elem->checkForOilTrapping(m_cappPress-gravCorr, trappingStorage, m_cpuTimeTrapping, m_trappingCriteria);
    if(!trappingStorage.empty()) 
    {
        for(size_t i = 0; i < trappingStorage.size(); ++i)
        {
            updateLayerCollapse(trappingStorage[i], m_cappPress);
        }
        m_commonData->addTrappedRegionOil(trappingStorage);
    }
}

void Netsim::updateElemImbibition(RockElem* elem)
{
    double entryPrs(0.0);
    if(elem->shape()->updateImbEntryPrs(entryPrs))
    {
        if(elem->isInWatFloodVec())
        {
            m_imbibitionEvents.remove(elem);
            elem->shape()->entryPress(entryPrs);
            m_imbibitionEvents.insert(elem);
        }
        else
        {
            elem->shape()->entryPress(entryPrs);
        }
    }
}

void Netsim::updateLayerCollapse(RockElem* elem, double cappPress)
{
    Polygon* polyShape = dynamic_cast< Polygon* >(elem->shape());
    if(polyShape)
    {
        for(int i = 0; i < polyShape->numCorners(); ++i)
        {
            pair< Polygon*, int > entry(polyShape, i);
            if(polyShape->oilInCorner(i)->isInCollapseVec())
            {
                m_layerCollapseEvents.remove(entry);
            }
        }

        polyShape->updateFilmsForTrapping(cappPress-polyShape->gravityCorrection());

        for(int j = 0; j < polyShape->numCorners(); ++j)
        {
            pair< Polygon*, int > entry(polyShape, j);
            if(polyShape->oilInCorner(j)->isInCollapseVec())
            {
                m_layerCollapseEvents.insert(entry);
            }
        }
    }
}

/////////////////////////////////////////////////////////////////////////////////
// Done with imbibition => clean up
/////////////////////////////////////////////////////////////////////////////////
void Netsim::finaliseImbibition()
{
    ostringstream out;
    out.flags(ios::showpoint);
    out.flags(ios::fixed);
    out.precision(3);

    if(m_maxOilFlowErr > 0.1 || m_maxWatFlowErr > 0.1 || m_maxResIdxErr > 0.1)
    {
        out << endl
            << "===================================================="   << endl
            << "Warning: For some rel perm calculations there were"     << endl
            << "more than 10% difference between flowrates computed"    << endl
            << "at the inlet and outlet. Perhaps try to reduce"         << endl
            << "solver tolerance"                                       << endl
            << "Max water flowrate error:    "  << m_maxWatFlowErr      << endl
            << "Max oil flowrate error:      "  << m_maxOilFlowErr      << endl
            << "Max resistivity index error: "  << m_maxResIdxErr       << endl
            << "===================================================="   << endl
            << endl;
    }

    vector< int > imbEventP(6);
    vector< int > imbEventT(3);
    
    for(int elm = 1; elm < static_cast< int >(m_rockLattice.size()); ++elm)
    {
        int event = m_rockLattice[elm]->fillingEvent();
        
        if(elm <= m_numPores)
        {
            if(event == -1) ++imbEventP[0];
            else if(event == 0 || event == 1) ++imbEventP[1];
            else if(event == 2) ++imbEventP[2];
            else if(event == 3) ++imbEventP[3];
            else if(event > 3) ++imbEventP[4];
            else ++imbEventP[5];
        }
        else if (elm > m_numPores+1)
        {
            if(event == -1) ++imbEventT[0];
            else if(event == 0 || event == 1) ++imbEventT[1];
            else ++imbEventT[2];
        }
    }

    out << endl
        << "=====================  Imbibition  ====================="                                   << endl            
        << "Total elapsed time for imbibition:       "  << m_cpuTimeTotal                               << endl
        << "Solving for water relative permeability: "  << m_cpuTimeKrw                                 << endl
        << "Solving for oil relative permeability:   "  << m_cpuTimeKro                                 << endl
        << "Solving for resistivity Index:           "  << m_cpuTimeResIdx                              << endl
        << "Identifying trapped elements:            "  << m_cpuTimeTrapping                            << endl
        << "Coalesceing trapped water:               "  << m_cpuTimeCoal                                << endl
        << "Max water flowrate error:                "  << m_maxWatFlowErr*100.0 << " %"                << endl
        << "Max oil flowrate error:                  "  << m_maxOilFlowErr*100.0 << " %"                << endl
        << "Max resistivity index error:             "  << m_maxResIdxErr*100.0 << " %"                 << endl
        << endl
        << "=====================  Network State  ====================="                                << endl
        << "Minimum capillary pressure reached (Pa): "  << m_cappPress                                  << endl
        << "Water saturation:                        "  << m_satWater                                   << endl
        << "Number of elements invaded:              "  << m_totNumFillings                             << endl
        << "Number of trapped water regions:         "  << (int)m_commonData->numTrappedWatRegions()    << endl
        << "Total number of trapped water elements:  "  << (int)m_commonData->numTrappedWatElems()      << endl
        << "Number of trapped oil regions:           "  << (int)m_commonData->numTrappedOilRegions()    << endl
        << "Total number of trapped oil elements:    "  << (int)m_commonData->numTrappedOilElems()      << endl
        << endl
        << "==================  Pore Filling Process  ================="                                << endl
        << "Uninvaded:                               "  << imbEventP[5]                                 << endl 
        << "Snap off:                                "  << imbEventP[0]                                 << endl
        << "Piston type displacement:                "  << imbEventP[1]                                 << endl 
        << "Pore body filling, I2:                   "  << imbEventP[2]                                 << endl
        << "Pore body filling, I3:                   "  << imbEventP[3]                                 << endl
        << "Pore body filling, I4+:                  "  << imbEventP[4]                                 << endl
        << endl
        << "=================  Throat Filling Process  ================="                               << endl
        << "Uninvaded:                               "  << imbEventT[2]                                 << endl 
        << "Snap off:                                "  << imbEventT[0]                                 << endl
        << "Piston type displacement:                "  << imbEventT[1]                                 << endl 
        << endl;

    m_amottDataImbibition[2] = m_satWater;
    if(m_cappPress > 0.0)
        m_amottWaterIdx = 1.0;
    else if(m_cappPress < 0.0 && m_amottDataImbibition[1] < 0.0)
        m_amottWaterIdx = 0.0;
    else
    {
        m_amottWaterIdx = (m_amottDataImbibition[1]-m_amottDataImbibition[0])/
            (m_amottDataImbibition[2]-m_amottDataImbibition[0]);
    }

    if(m_commonData->drainageCycle() > 1)
    {
        out << endl
            << "=================  Wettability State  ================"                     << endl
            << "Amott water index, Iw:                  " << m_amottWaterIdx                << endl
            << "Amott oil index, Io:                    " << m_amottOilIdx                  << endl
            << "Amott wettability index, I = Iw - Io:   " << m_amottWaterIdx-m_amottOilIdx  << endl
            << "USBM wettability index:                 " << calcUSBMindex()                << endl
            << endl;
    }

    writePrtData(out);    
    writeResultData(m_calcRelPerm, m_calcResIdx, m_imbResultsOut);

    if(m_createImbList)
    {
        m_imbListOut << "]; " << endl;
        m_imbListOut.close();
    }

    if(m_prtPressureProfile && m_calcRelPerm)
    {
        writePrsProfileData(m_water, m_imbResultsOut);
        writePrsProfileData(m_oil, m_imbResultsOut);
    }
    
    if(m_solver != NULL)
    {
        delete m_solver;
        m_solver = NULL;
    }

    m_imbibitionEvents.clear();
    m_layerCollapseEvents.clear();
}

/////////////////////////////////////////////////////
// Do a single calculation of relative permeability
/////////////////////////////////////////////////////
void Netsim::solveLaplaceEq(bool calcRelPerm, bool calcResIdx, ostream& out)
{   
    double oilErr(0.0), watErr(0.0), resErr(0.0); 
    bool chk;

    out.flags(ios::scientific);

    if(calcRelPerm)
    {

        m_watFlowRate = m_solver->flowrate(m_inletSolverPrs, m_outletSolverPrs, m_water, watErr, m_cpuTimeKrw, 
            m_satWater, m_writeWatVelocity, m_writeWatMatrix);

        m_relPermWater = m_watFlowRate / m_singlePhaseWaterQ;
        if(m_watFlowRate != 0.0)
        {
            if(m_useAvrPrsAsBdr)
            {
                chk = prsOrVoltDrop(m_water, false, m_deltaPw);
                assert(chk);
                m_relPermWater = (m_watFlowRate * m_singlePhaseDprs) / (m_singlePhaseWaterQ * m_deltaPw);
            }
            if(m_prtPressureProfile) recordPrsProfiles(m_water);
        }

        m_oilFlowRate = m_solver->flowrate(m_inletSolverPrs, m_outletSolverPrs, m_oil, oilErr, m_cpuTimeKro, 
            m_satWater, m_writeOilVelocity, m_writeOilMatrix); 
                
        m_relPermOil = m_oilFlowRate / m_singlePhaseOilQ;
        if(m_oilFlowRate != 0.0)
        {
            if(m_useAvrPrsAsBdr)
            {
                chk = prsOrVoltDrop(m_oil, false, m_deltaPo);
                assert(chk);
                m_relPermOil = (m_oilFlowRate * m_singlePhaseDprs) / (m_singlePhaseOilQ * m_deltaPo);
            }
            if(m_prtPressureProfile) recordPrsProfiles(m_oil);
        }
               
        out << "Pc = " << m_cappPress << "; krw = " << m_relPermWater << "; kro = " << m_relPermOil;
        
        m_maxOilFlowErr = max(m_maxOilFlowErr, oilErr);
        m_maxWatFlowErr = max(m_maxWatFlowErr, watErr);
        
    }

    if(calcResIdx)
    {
        m_current = m_solver->flowrate(m_inletSolverPrs, m_outletSolverPrs, m_water, resErr, m_cpuTimeResIdx, m_satWater, 
            m_writeResVelocity, m_writeResMatrix, true);
        m_resistivityIdx = m_singlePhaseCurrent / m_current;
        if(m_current > 0.0)
        {
            if(m_useAvrPrsAsBdr)
            {
                double deltaV(0.0);
                bool chk = prsOrVoltDrop(m_water, true, deltaV);
                assert(chk);
                m_resistivityIdx = (m_current * m_singlePhaseDvolt) / (m_singlePhaseCurrent * deltaV);
            }
        }
        
        m_maxResIdxErr = max(m_maxResIdxErr, resErr);
    }
}

///////////////////////////////////////////////////////////////////////////////////////////
// Itherates across all nework elements (not in/outlets) and computes the water saturation
//////////////////////////////////////////////////////////////////////////////////////////
void Netsim::calculateWaterSat()
{
    double volWater(0.0);
    for(size_t elem = 0; elem < m_rockLattice.size(); ++elem)
    {
        volWater += m_rockLattice[elem]->updateSatAndConductance(m_cappPress-
            m_rockLattice[elem]->shape()->gravityCorrection());
    }

    m_satWater = volWater / (m_netVolume + m_clayVolume);
}

void Netsim::checkForUnstableConfigs()
{
    for(size_t i = 0; i < m_rockLattice.size(); ++i)
    {
        RockElem* elem = m_rockLattice[i];
        double gravCorr(elem->shape()->gravityCorrection());
        if(elem->shape()->unstableWaterConfig(m_cappPress-gravCorr))
        {
            if(elem->isInWatFloodVec()) 
            {
                m_imbibitionEvents.remove(elem);
                elem->isInWatFloodVec(false);
            }
            elem->imbide(true);
            
            if(elem->trappedOil())
            {
                int trpIdx = elem->trappingIndexOil();
                elem->unTrapOil();
                m_commonData->removeTrappedOilElem(trpIdx, elem);
            }
            
            if(elem->trappedWat(filmBlob))
            {
                int trpIdx = elem->trappingIndexWat(filmBlob);
                elem->copyWatFilmTrappingToBulk();
                pair< RockElem*, FluidBlob > trpElem(elem, bulkBlob);
                m_commonData->addTrappeWatElem(trpIdx, trpElem);
            }
            for(int j = 0; j < elem->connectionNum(); ++j)
            {
                updateElemImbibition(elem->connection(j));
                addElemToImbEvents(elem->connection(j));
            }               
        }
        else if(elem->shape()->unstableOilConfig(m_cappPress-gravCorr))
        {
            if(elem->isInOilFloodVec()) 
            {
                m_drainageEvents.remove(elem);
                elem->isInOilFloodVec(false);
            }
            Polygon* polyShape = dynamic_cast< Polygon* >(elem->shape());
            if(polyShape)
            {
                for(int corn = 0; corn < polyShape->numCorners(); ++corn)
                {
                    if(polyShape->oilInCorner(corn)->isInReformVec())
                    {
                        pair<Polygon*, int> cornElem(polyShape, corn);
                        m_layerReformEvents.remove(cornElem);
                        polyShape->oilInCorner(corn)->isInReformVec(false);
                    }
                }
            }
            elem->drain();
            
            if(elem->trappedWat(bulkBlob))
            {
                int trpIdx = elem->trappingIndexWat(bulkBlob);
                elem->unTrapWat(bulkBlob);
                pair< RockElem*, FluidBlob > trpElem(elem, bulkBlob);
                m_commonData->removeTrappeWatElem(trpIdx, trpElem);
            }
            for(int j = 0; j < elem->connectionNum(); ++j)
            {
                updateElemDrainage(elem->connection(j));
                addElemToDrainEvents(elem->connection(j));
            }               
        }
    }
}

////////////////////////////////////////////////////////////////////////////////////////////////
// Writes significant data about pores/throats to file. Included data are: radius, shape factor,
// half angles and wheter corners contain water.
////////////////////////////////////////////////////////////////////////////////////////////////
void Netsim::writePoreDataToFile(const string& baseFileName) const
{
    string filePore(baseFileName + "_pore.out"), fileThroat(baseFileName + "_throat.out");
    ofstream poreOut(filePore.c_str()), throatOut(fileThroat.c_str());
       
    if(!poreOut || !throatOut)
    {
        cout << endl
            << "======================================" << endl
            << "Warning: Could not open files for size" << endl 
            << "and property output"                    << endl   
            << "======================================" << endl
            << endl;
    }
    else
    {
        ostringstream strOut;   
        strOut << "Radius [micro.m]"  << endl
            << "Area [micro.m^2]"               << endl
            << "Shape Factor"                   << endl
            << "Rec con ang [deg]"   << endl
            << "Adv con ang [deg]"  << endl
            << "Volume [micro.m^3]"             << endl
            << "Clay volume [micro.m^3]"        << endl
            << "Inside calculation box"         << endl;
        
        
        poreOut << m_numPores 
            << "  " << m_xSize << "  " << m_ySize << "  " << m_zSize                                       
            << " (num pores, x size, y size, z size)"       << endl
            << 15                           << endl
            << "X position [micro.m]"       << endl
            << "Y position [micro.m]"       << endl
            << "Z position [micro.m]"       << endl
            << strOut.str()
            << "Connection number"          << endl
            << "Inlet pressure boundary"    << endl
            << "Outlet pressure boundary"   << endl
            << "Radius aspect ratio"        << endl;
        
        throatOut << static_cast< int >(m_rockLattice.size()) - m_numPores - 2 
            << "  " << m_xSize << "  " << m_ySize << "  " << m_zSize                                       
            << " (num throats, x size, y size, z size)" << endl
            << 15                                   << endl
            << "Pore 1 index"                       << endl
            << "Pore 2 index"                       << endl
            << strOut.str()
            << "Connected inlet pressure boundary"  << endl
            << "Connected outlet pressure boundary" << endl
            << "Throat length [micro.m]"            << endl
            << "Pore 1 half length [micro.m]"       << endl
            << "Pore 2 half length [micro.m]"       << endl;
        
        
        for(size_t elem = 0; elem < m_rockLattice.size(); ++elem)
        {        
            if(dynamic_cast< Throat * >(m_rockLattice[elem]) != NULL)
                throatOut << *m_rockLattice[elem] << endl;
            else if(dynamic_cast< EndPore * >(m_rockLattice[elem]) != NULL)
            {}
            else
                poreOut << *m_rockLattice[elem] << endl;
        }
        
        poreOut.close();
        throatOut.close();
    }
}

////////////////////////////////////////////////////////
// Record the water saturation of all pores/throats for
// subsequent 3D visualization.
////////////////////////////////////////////////////////
void Netsim::recordWaterSatMap()
{
    m_waterSatHistory.push_back(m_satWater);
    for(size_t i = 0; i < m_rockLattice.size(); ++i)
        m_rockLattice[i]->shape()->recordSw();
}

////////////////////////////////////////////////////////////////////////////////////////////
// Writes water saturation data to file. The pore file contains position data for each node, 
// whereas the throat file contains indecies to connecting nodes. These files can become
// quite large if there are lots of data points. 
////////////////////////////////////////////////////////////////////////////////////////////
void Netsim::writeSwHistoryToFile(const string& baseFileName) const
{
    string filePore(baseFileName + "_pore_sw.out"); 
    string fileThroat(baseFileName + "_throat_sw.out");
    ofstream poreOut(filePore.c_str()), throatOut(fileThroat.c_str());

    if(!poreOut || !throatOut)
    {
        cout << endl
            << "======================================="    << endl
            << "Warning: Could not open files for water"    << endl   
            << "saturation output."                         << endl
            << "======================================="    << endl
            << endl;
    }
    else
    {
        int numRecords = static_cast< int >(m_waterSatHistory.size());
        
        ostringstream strOut;
        strOut.flags(ios::showpoint);
        strOut.flags(ios::fixed);
        for(int i = 0; i < numRecords; ++i)
        {
            strOut << m_waterSatHistory[i]  << " = Water saturation (Sw record no: " << i+1 << ")" << endl;
        }
        
        poreOut << m_numPores 
            << "  " << m_xSize << "  " << m_ySize << "  " << m_zSize                                       
            << " (num pores, x size, y size, z size)"   << endl
            << 5 + numRecords                           << endl
            << "Index"                                  << endl
            << "X position"                             << endl
            << "Y position"                             << endl
            << "Z position"                             << endl
            << "Inside calculation box"                 << endl
            << strOut.str();
        
        throatOut << static_cast< int >(m_rockLattice.size()) - m_numPores - 2 
            << "  " << m_xSize << "  " << m_ySize << "  " << m_zSize                                       
            << " (num throats, x size, y size, z size)" << endl
            << 3 + numRecords                           << endl
            << "Pore index 1"                           << endl
            << "Pore index 2"                           << endl
            << "Inside calculation box"                 << endl
            << strOut.str();
        
        
        poreOut.flags(ios::showpoint);
        throatOut.flags(ios::showpoint);
        for(size_t j = 0; j < m_rockLattice.size(); ++j)
        {
            if(dynamic_cast< Throat* >(m_rockLattice[j]) != NULL)
            {
                throatOut << setw(10) << m_rockLattice[j]->connection(0)->node()->indexOren()
                    << setw(10) << m_rockLattice[j]->connection(1)->node()->indexOren()
                    << setw(4)  << m_rockLattice[j]->isInsideSatBox();
                
                throatOut.flags(ios::fixed);
                const vector< double > swHistory(m_rockLattice[j]->shape()->waterSatHistory());
                for(size_t event = 0; event < swHistory.size(); ++event)
                    throatOut << setw(10) << setprecision(4) << swHistory[event];
                throatOut << endl;
            }
            else if(dynamic_cast< EndPore * >(m_rockLattice[j]) != NULL)
            {}
            else
            {
                poreOut << setw(10) << m_rockLattice[j]->node()->indexOren() 
                    << *m_rockLattice[j]->node()
                    << setw(4) << m_rockLattice[j]->isInsideSatBox();
                
                poreOut.flags(ios::fixed);
                const vector< double > swHistory(m_rockLattice[j]->shape()->waterSatHistory());
                for(size_t event = 0; event < swHistory.size(); ++event)
                    poreOut << setw(10) << setprecision(4) << swHistory[event];
                poreOut << endl;
            }        
        }
        
        poreOut.close();
        throatOut.close();
    }
}

/////////////////////////////////////////////////////////////////////////////////
// We might want to retain the pressure profiles through the network
/////////////////////////////////////////////////////////////////////////////////
void Netsim::recordPrsProfiles(const Fluid* fluid)
{
    char prefixChar(' ');
    if(m_excelFormat) prefixChar = ',';

    ostringstream outData;
    outData.flags(ios::showpoint);
    outData.flags(ios::fixed);
    outData.precision(4);
    outData << setw(8) << m_satWater << prefixChar;

    ostringstream outStd;
    outStd.flags(ios::showpoint);
    outStd.flags(ios::fixed);
    outStd.precision(4);
    outStd << setw(8) << m_satWater << prefixChar;;
    
    ostringstream outNum;
    outNum.flags(ios::showpoint);
    outNum.flags(ios::fixed);
    outNum.precision(4);
    outNum << setw(8) << m_satWater << prefixChar;;

    for(int i = 0; i < m_numPressurePlanes; ++i)
    {
        double avgPrs(0.0), std(0.0); 
        int numVal(0);
        
        avrPrsOrVolt(fluid, false, i, avgPrs, std, numVal);
        
        if(!m_useAvrPrsAsBdr && i == 0)     // Make sure average pressure at in/outlet is infact
        {                                   // 0 and 1 when we have previously moved the boundary
            avgPrs = m_inletSolverPrs;
            std = 0.0;
        }
        else if(!m_useAvrPrsAsBdr && i == m_numPressurePlanes-1) 
        {
            avgPrs = m_outletSolverPrs;
            std = 0.0;
        }

        outData << setw(8) << avgPrs << prefixChar << "  "; 
        outStd << setw(8) << std << prefixChar << "  ";
        outNum << setw(8) << numVal << prefixChar << "  ";
    }

    if(m_matlabFormat)
    {
        outData << "; ..."; 
        outStd << "; ..."; 
        outNum << "; ..."; 
    }

    vector< string > prsDat(3);
    prsDat[0] = outData.str();
    prsDat[1] = outStd.str();
    prsDat[2] = outNum.str();
    
    if(dynamic_cast< const Oil* >(fluid) != NULL)
        m_oilPrsProfiles.push_back(prsDat);
    else
        m_watPrsProfiles.push_back(prsDat);
}

void Netsim::writePrsProfileData(const Fluid* fluid, vector< ostream* >& resStream)  
{
    string fluidName("\nWater");
    vector< vector< string > >& data = m_watPrsProfiles;
    
    if(dynamic_cast< const Oil* >(fluid) != NULL)
    {
        fluidName = "\nOil";
        data = m_oilPrsProfiles;
    }

    vector< string > propertyName(3);
    if(m_matlabFormat)
    {
        propertyName[0] = "Prs = [";
        propertyName[1] = "Std = [";
        propertyName[2] = "NoSamp = [";
    }
    else
    {
        propertyName[0] = " Average Pressure \n";
        propertyName[1] = " Standard Deviation \n";
        propertyName[2] = " Number of sample points \n";
    }

    ostringstream out;    
    out.flags(ios::showpoint);
    out.flags(ios::fixed);
    out.precision(4);
    if(m_matlabFormat)
    {
        out << fluidName <<"Loc  =  [";
        for(int l = 0; l < m_numPressurePlanes; ++l)
            out << setw(9) << m_pressurePlanesLoc[l];
        out << "];" << endl;
    }
    else
    {
        char seperator = m_excelFormat ? ',': ' ';
        out << "Loc   " << seperator; 
        for(int l = 0; l < m_numPressurePlanes; ++l)
            out << setw(10) << m_pressurePlanesLoc[l] << seperator;
        out << endl;
    }
    
    for(size_t j = 0; j < resStream.size(); ++j)
    {
        if(m_matlabFormat) *resStream[j] << out.str();

        for(int i = 0; i < 3; ++i)
        {
            *resStream[j] << fluidName << propertyName[i];
            if(!m_matlabFormat) *resStream[j] << out.str();
            
            for(size_t k = 0; k < data.size(); ++k)
                *resStream[j] << data[k][i] << endl;
            
            if(m_matlabFormat) *resStream[j] << "];" << endl;
        }
    }
    data.clear();
}

void Netsim::writeResultData(bool relPermIncluded, bool resIdxIncluded, vector< ostream* >& resStream)
{
    formatResults();
    ostringstream out;    
    string legend;
    int cycle;
    if(m_commonData->injectant() == m_oil)
    {
        legend = m_matlabFormat ? "draincycle_" : "(draincycle ";
        cycle = m_commonData->drainageCycle();
    }
    else
    {
        legend = m_matlabFormat ? "imb_": "  (imbcycle ";
        cycle = m_commonData->imbibitionCycle();
    }
       
    char seperator = m_excelFormat ? ',': ' ';
    if(!m_matlabFormat) 
    { 
        out << static_cast< unsigned int >(m_results.size()) << seperator << "number of data points" << endl; 
        out << "Sw                 " << seperator << "Pc (Pa) " << legend << cycle << ")    ";
        
        if(m_apexPrsReported)
        {
            out << seperator << "Pc_b (Pa) " << legend << cycle << ")  ";
        }
        
        
        if(relPermIncluded) 
        {
            out << seperator << "Krw " << legend << cycle << ")        "
                << seperator << "Kro " << legend << cycle << ")        ";
        }
        
        if(resIdxIncluded)
        {
            out << "  " << seperator << "I " << legend << cycle << ")        ";
        }

        if(m_reportMaterialBal)
        {
            out << seperator << "Mass_w (kg) " << legend << cycle << ")"
                << seperator << "Mass_o (kg) " << legend << cycle << ")";
        }
        out << endl;
    }
    else
    {
        out << "DataLegend = {'Sw " << legend << cycle << "' 'Pc (Pa) " << legend << cycle << "'";
        if(relPermIncluded) out << "'Krw " << legend << cycle << "' 'Kro " << legend << cycle << "'";
        if(resIdxIncluded)  out << "'I " << legend << cycle << "'";
        if(m_reportMaterialBal) out << "'Mass_w (kg) " << legend << cycle << "' 'Mass_o (kg) " << legend << cycle << "'";
        out << "};" << endl << "Res_" << legend << cycle << " = [";
    }

    for(size_t i = 0; i < resStream.size(); ++i)
    {
        *resStream[i] << out.str();
        
        for(size_t j = 0; j < m_results.size(); ++j)
            *resStream[i] << m_results[j] << endl;

        if(m_matlabFormat) *resStream[i] << "];" << endl;
    }
    m_results.clear();
    m_resultWaterFlowRate.clear();
    m_resultOilFlowRate.clear();
    m_resultWaterSat.clear();
    m_resultCappPress.clear();
    m_resultBoundingPc.clear();
    m_resultResistivityIdx.clear();
    m_resultWaterMass.clear();
    m_resultOilMass.clear();
}

void Netsim::recordRes(bool calcRelPerm, bool calcResIdx)
{
    m_resultWaterSat.push_back(m_satWater);
    m_resultCappPress.push_back(m_cappPress);
    if(m_apexPrsReported)
    {
        m_resultBoundingPc.push_back(m_boundPress);
    }
    if(calcRelPerm)
    {
        pair<double, double> watElem(m_watFlowRate, m_deltaPw), oilElem(m_oilFlowRate, m_deltaPo);
        m_resultWaterFlowRate.push_back(watElem);
        m_resultOilFlowRate.push_back(oilElem);
    }
    if(calcResIdx)
    {
        m_resultResistivityIdx.push_back(m_resistivityIdx);
    }
    if(m_reportMaterialBal)
    {
        m_resultWaterMass.push_back((m_netVolume+m_clayVolume)*m_satWater*m_water->density());
        m_resultOilMass.push_back((m_netVolume+m_clayVolume)*(1-m_satWater)*m_oil->density());
    }
}

void Netsim::formatResults()
{
    bool calcRelPerm(m_resultWaterFlowRate.size() > 0);
    bool calcResIdx(m_resultResistivityIdx.size() > 0);
        
    string whiteSpace;   
    if(m_matlabFormat) whiteSpace = "  ";
    else if(m_excelFormat) whiteSpace = ",           ";
    else whiteSpace = "            ";

    double krwRefFlow(m_singlePhaseWaterQ), kroRefFlow(m_singlePhaseOilQ);
    double deltaPwRef(m_singlePhaseDprs), deltaPoRef(m_singlePhaseDprs);
    if(calcRelPerm && (m_relPermDef[2] == 's' || m_relPermDef[2] == 'S'))   // Scale wrt residual cond
    {
        double maxQW(0.0), maxQO(0.0), delPw(0.0), delPo(0.0);
        for(size_t el = 0; el < m_resultWaterSat.size(); ++el)
        {
            if(m_resultWaterFlowRate[el].first > maxQW)
            {
                maxQW = m_resultWaterFlowRate[el].first;
                delPw = m_resultWaterFlowRate[el].second;
            }
            if(m_resultOilFlowRate[el].first > maxQO)
            {
                maxQO = m_resultOilFlowRate[el].first;
                delPo = m_resultOilFlowRate[el].second;
            }
        }
        krwRefFlow = maxQW;
        kroRefFlow = maxQO;
        deltaPwRef = delPw;
        deltaPoRef = delPo;
    }
    else if(calcRelPerm && m_relPermDef[2] != 'n' && m_relPermDef[2] == 'N')
    {
        cout << "========================================================" << endl
            << "Warning. Uknown reference flag for relative permeability" << endl
            << "calculations. Use 'single' or 'residual'. Defaulting" << endl
            << "to single phase reference." << endl
            << "========================================================" << endl;
    }
    
    for(size_t entry = 0; entry < m_resultWaterSat.size(); ++entry)
    {
        ostringstream out;
        out.flags(ios::showpoint);
        out.flags(ios::fixed);

        out << setw(10) << m_resultWaterSat[entry] << whiteSpace;    
        out.flags(ios::scientific);    
        out << setw(15) << m_resultCappPress[entry] << whiteSpace;

        if(m_apexPrsReported)
        {
            out << setw(15) << m_resultBoundingPc[entry] << whiteSpace;
        }

        if(calcRelPerm)
        {
            double krw(0.0), kro(0.0);
            if(m_resultWaterFlowRate[entry].first > 0.0)
            {
                if(m_useAvrPrsAsBdr) 
                    krw = m_resultWaterFlowRate[entry].first*deltaPwRef/(krwRefFlow*m_resultWaterFlowRate[entry].second);
                else
                    krw = m_resultWaterFlowRate[entry].first / krwRefFlow;
            }
            if(m_resultOilFlowRate[entry].first > 0.0)
            {
                if(m_useAvrPrsAsBdr) 
                    kro = m_resultOilFlowRate[entry].first*deltaPoRef/(kroRefFlow*m_resultOilFlowRate[entry].second);
                else
                    kro = m_resultOilFlowRate[entry].first / kroRefFlow;
            }
            out << setw(15) << krw << whiteSpace << setw(15) << kro << whiteSpace;
        }

        if(calcResIdx)
            out << setw(15) << m_resultResistivityIdx[entry] << whiteSpace;

        if(m_reportMaterialBal)
        {
            out << setw(15) << m_resultWaterMass[entry] << whiteSpace 
                << setw(15) << m_resultOilMass[entry] << whiteSpace;
        }

        if(m_matlabFormat) out << "; ...";

        m_results.push_back(out.str());
    }
}

string Netsim::calcUSBMindex() const
{
    ostringstream out;
    out.flags(ios::showpoint);
    out.flags(ios::fixed);
    out.precision(3);

    size_t numDrainPts(m_usbmDataDrainage.size()), numImbPts(m_usbmDataImbibition.size());
    double areaDrain(0.0), areaImb(0.0);
   
    if(numDrainPts > 1)
    {
        for(size_t i = 1; i < numDrainPts; ++i)
        {
            pair< double, double > ptOne(m_usbmDataDrainage[i]), ptTwo(m_usbmDataDrainage[i-1]);
            double deltaSw(ptTwo.second-ptOne.second);
            double averagePc((ptTwo.first+ptOne.first)/2.0);
            areaDrain += deltaSw*averagePc;
        }
    }

    if(numImbPts > 1)
    {
        for(size_t i = 1; i < numImbPts; ++i)
        {
            pair< double, double > ptOne(m_usbmDataImbibition[i-1]), ptTwo(m_usbmDataImbibition[i]);
            double deltaSw(ptTwo.second-ptOne.second);
            double averagePc(-(ptTwo.first+ptOne.first)/2.0);
            areaImb += deltaSw*averagePc;
        }
    }

    if(areaDrain == 0.0)
        out << "-INF";
    else if(areaImb == 0.0)
        out << "+INF";
    else
        out << log10(areaDrain/areaImb); 

    return out.str();
}



