#ifndef NETSIM_H
#define NETSIM_H

#ifdef WIN32   
#pragma warning(disable:4786)
#endif

class Solver;
class Fluid;
class CommonData;
class Shape;

#include "rockElem.h"
#include "compareFuncs.h"
#include "sortedEvents.h"

class Netsim
{
public:

    Netsim(unsigned seed);
    Netsim(const Netsim& net);
    ~Netsim();

    void init(InputData& input);

    void initDrainage(bool calcRelPerm, bool calcResIdx);
    int drainageCycle() const {return m_commonData->drainageCycle();}
    void drainage(double& swt, double& pct, double& krw, double& kro, double& resIdx, bool& swr);
    void finaliseDrainage();
    
    int imbibitionCycle() const {return m_commonData->imbibitionCycle();}
    void initImbibition(bool calcRelPerm, bool calcResIdx, InputData& input);
    void injectWater(double& swt, double& pct, double& krw, double& kro, double& resIdx, bool& swr);
    void finaliseImbibition();
     
    void addOStreamForDrainRes(ostream* out) {m_drainResultsOut.push_back(out);}
    void addOStreamForImbRes(ostream* out) {m_imbResultsOut.push_back(out);}
    void clearOStreamForDrainRes() {m_drainResultsOut.clear();}
    void clearOStreamForImbRes() {m_imbResultsOut.clear();}
    
    void addOStreamForPrt(ostream* out) {m_prtOut.push_back(out);}
    
    void recordWaterSatMap();
    void writeSwHistoryToFile(const string& baseFileName) const;
    void writePoreDataToFile(const string& baseFileName) const;   
    
private:

    void formatResults();
    int initNetwork(InputData& input, bool drainSinglets);
    void modifyNetwork(InputData& input);
    void applyFractionalWetting(InputData& input);
    void initFlow(InputData& input, bool writeMatInitOnly, bool createLocData, bool drainSinglets);
    int readAndCreateThroats(InputData& input, vector< pair<int, int> >& pores, 
        vector< RockElem* >& inT, vector< RockElem* >& outT, vector< pair< int, double > >& poreHash, 
        vector< int >& throatHash, int newNumPores);
    void writeNetworkToFile(bool writeInBinary, bool drainSinglets, const string& fileNameBase);
    void readAndCreatePores(InputData& input, vector< pair<int, int> >& connectingPores,  
        vector< pair< int, double > >& poreHash, vector< int >& throatHash, int newNumPores);
    void createInAndOutletPore(int index, double xPos, double yPos, double zPos, 
        vector< RockElem* >& connThroats);
    int checkPoreIndex(int index) const;
    
    bool prsOrVoltDrop(const Fluid* fluid, bool resistSolve, double& prsDrop) const;
    bool avrPrsOrVolt(const Fluid* fluid, bool resist, int pPlane, double& res, 
        double& stdDev, int& numVal) const;
    void solveLaplaceEq(bool calcRelPerm, bool calcResIdx, ostream& out);
    void calculateWaterSat();
    
    void singleWaterDisplacement(double& targetWaterSat, double& targetPc, 
        bool& residualSat, ostream& out);
    void popAndUpdateDrainVec(bool& insideBox);
    void popAndUpdateReformVec();
    void trappWaterCheck(RockElem* elem, FluidBlob startPt);
    void coalesceOil(RockElem* elem);
    void updateElemDrainage(RockElem* elem);
    void updateLayerReform(RockElem* elem, double cappPress);
    void singleOilDisplacement(double& targetWaterSat, double& targetPc, 
        bool& residualSat, ostream& out);
    void popAndUpdateImbVec(bool& insideBox);
    void popAndUpdateCollapseVec();
    void coalesceWater(RockElem* elem, FluidBlob blob);
    void trappOilCheck(RockElem* elem);
    void updateElemImbibition(RockElem* elem);
    void updateLayerCollapse(RockElem* elem, double cappPress);
    void checkForUnstableConfigs();
    int setupPoreHashing(InputData& input, vector< pair< int, double > >& poreHash) const;
    void createMatlabLocationData() const;
    void modifyInscribedRadii(int model, const string& options, vector< RockElem* >& elems, 
        const string& fileName, bool writeToFile, int numPts, bool forPores = false);
    void modifyShapeFactor(int model, const string& options, vector< RockElem* >& elems, 
        const string& fileName, bool writeToFile, int numPts);
    void setRadiiFromFile(vector< RockElem* >& elems, const string& fileName, double lowCO, double hiCO, bool forPores);
    void setShapeFactFromFile(vector< RockElem* >& elems, const string& fileName, double lowCO, double hiCO);
    void removeConnectionsFromNetwork(InputData& input);
    void recordPrsProfiles(const Fluid* fluid);
    void recordRes(bool calcRelPerm, bool calcResIdx);
    void writePrsProfileData(const Fluid* fluid, vector< ostream* >& resStream);  
    void writeResultData(bool relPermIncluded, bool resIdxIncluded, 
        vector< ostream* >& resStream);
    void dropPressureInCoalescedBlob(SortedEvents< ThreeSome<Polygon*, int, double>, CoalesceWatFillCmp >& watEvents, bool watFloodCycle);
    void increasePressureInCoalescedBlob(SortedEvents< ThreeSome<Polygon*, int, double>, CoalesceOilFillCmp >& oilEvents, bool watFloodCycle);
    void setContactAngles(vector< RockElem* >& pores, vector< RockElem* >& throats, double min, double max, 
        double delta, double eta, bool attractToMarked = true);
    string calcUSBMindex() const;
    double trappedWWoil() const;
    inline int numInstancesInCollapsevec(pair<Polygon*, int> elem) const;
    inline void assureCollapseVecIntegrity() const;

    inline double weibull(double minVal, double maxVal, double delta, double eta) const;
    inline void writePrtData(ostringstream& out);
    inline double cpuTimeElapsed(clock_t start);
    inline void addElemToDrainEvents(RockElem* elem);
    inline void addElemToImbEvents(RockElem* elem);
    inline void addElemToReformEvents(RockElem* elem);
    inline void addElemToCollapseEvents(RockElem* elem);
    inline void clearTrappedOil();
    inline void clearTrappedWat();
    inline void recordUSBMData(bool drainage);
    inline void recordAmottData(bool drainage);
    inline double tableLookUpX(double valY, const vector< pair<double, double> >& tab) const;
    inline double tableLookUpY(double valX, const vector< pair<double, double> >& tab) const;
    inline void checkStateOfInitConAng();
   
    static const double                             MAX_FLOW_ERR;
    static const int                                DUMMY_IDX;
    const unsigned                                  m_randSeed;
    
    CommonData*                                     m_commonData;
    SortedEvents<RockElem*, PceDrainCmp>            m_drainageEvents;
    SortedEvents< pair<Polygon*, int>, PcColCmp >   m_layerCollapseEvents;
    SortedEvents<RockElem*, PceImbCmp>              m_imbibitionEvents;
    SortedEvents< pair<Polygon*, int>, PcRefCmp >   m_layerReformEvents;
    
    Fluid*                                          m_oil;
    Fluid*                                          m_water;

    vector< RockElem* >                             m_rockLattice; 
    vector< RockElem* >                             m_krInletBoundary;
    vector< RockElem* >                             m_krOutletBoundary;
    vector< double >                                m_waterSatHistory;
    vector< ostream* >                              m_prtOut;
    vector< ostream* >                              m_drainResultsOut;
    vector< ostream* >                              m_imbResultsOut;
    vector< string >                                m_results;
    vector< vector< string > >                      m_watPrsProfiles;
    vector< vector< string > >                      m_oilPrsProfiles;
    vector< pair< double, double > >                m_usbmDataDrainage;
    vector< pair< double, double > >                m_usbmDataImbibition;
    vector< double >                                m_amottDataDrainage;
    vector< double >                                m_amottDataImbibition;
    
    vector< pair<double, double> >                  m_resultWaterFlowRate;
    vector< pair<double, double> >                  m_resultOilFlowRate;
    vector< double >                                m_resultWaterSat;
    vector< double >                                m_resultCappPress;
    vector< double >                                m_resultBoundingPc;
    vector< double >                                m_resultResistivityIdx;
    vector< double >                                m_resultWaterMass;
    vector< double >                                m_resultOilMass;

    int                                             m_maxNonZeros;
    int                                             m_numPores;
    int                                             m_numThroats;
    int                                             m_numIsolatedElems;
    int                                             m_numPressurePlanes;
    int                                             m_minNumFillings;
    int                                             m_totNumFillings;
    int                                             m_wettingClass;
    int                                             m_sourceNode;

    double                                          m_shavingFraction;
    double                                          m_modelTwoSepAng;
    double                                          m_amottOilIdx;
    double                                          m_amottWaterIdx;
    double                                          m_currentPc;
    double                                          m_cpuTimeTotal;
    double                                          m_cpuTimeCoal;
    double                                          m_cpuTimeKrw;
    double                                          m_cpuTimeKro;
	double                                          m_cpuTimeResIdx;
    double                                          m_cpuTimeTrapping;
    double									        m_inletSolverPrs;
    double									        m_outletSolverPrs;
    double                                          m_clayAdjust;
    mutable double                                  m_maxOilFlowErr;
    mutable double                                  m_maxWatFlowErr;
    mutable double                                  m_maxResIdxErr;
    double                                          m_wettingFraction;
    double                                          m_singlePhaseWaterQ;
    double                                          m_singlePhaseOilQ;
    double                                          m_singlePhaseDprs;
    double                                          m_singlePhaseDvolt;
    double                                          m_singlePhaseCurrent;
    double                                          m_oilFlowRate;
    double                                          m_watFlowRate;
    double                                          m_current;
    double                                          m_formationFactor;
    double                                          m_medianLenToRadRatio;
    double                                          m_absPermeability;
    double                                          m_satBoxStart;      
    double                                          m_satBoxEnd;
    double                                          m_solverBoxStart;
    double                                          m_solverBoxEnd;
    double                                          m_netVolume;
    double                                          m_clayVolume;
    double                                          m_rockVolume;
    double                                          m_maxCappPress;
    double                                          m_minCappPress;
    double                                          m_satWater;
    double                                          m_relPermWater;
    double                                          m_relPermOil;
    double                                          m_resistivityIdx;
    double                                          m_cappPress;
    double                                          m_boundPress;
    double                                          m_xSize; 
    double                                          m_ySize; 
    double                                          m_zSize;
    double                                          m_initStepSize;
    double                                          m_extrapCutBack;
    double                                          m_maxFillIncrease;
    double                                          m_deltaPo;
    double                                          m_deltaPw;

    bool                                            m_apexPrsReported;
    bool                                            m_includeGravityInRelPerm;
    bool                                            m_stableFilling;
    bool                                            m_matlabFormat;
    bool                                            m_reportMaterialBal;
    bool                                            m_excelFormat;
    bool                                            m_useAvrPrsAsBdr;
    bool                                            m_prtPressureProfile;
    bool                                            m_calcRelPerm;
    bool                                            m_calcResIdx;
    bool                                            m_injAtEntryRes;
    bool                                            m_injAtExitRes;
    bool                                            m_writeWatMatrix;
    bool                                            m_writeOilMatrix;
    bool                                            m_writeResMatrix;
    bool                                            m_writeWatVelocity;
    bool                                            m_writeOilVelocity;
    bool                                            m_writeResVelocity;
    bool                                            m_writeSlvMatrixAsMatlab;
    bool                                            m_createDrainList; 
    bool                                            m_createImbList;

    string                                          m_matrixFileName;
    string                                          m_angDistScheme;
    string                                          m_relPermDef;
    Solver*                                         m_solver;

    TrappingCriteria                                m_trappingCriteria;
    vector< double >                                m_pressurePlanesLoc;
    vector< vector< RockElem* > >                   m_pressurePlanes;
    
    ofstream                                        m_drainListOut;
    ofstream                                        m_imbListOut;
};

inline double Netsim::weibull(double minVal, double maxVal, double delta, double eta) const
{    
    double randNum = static_cast<double>(rand()) / static_cast<double>(RAND_MAX);
    //double randNum = 0.5; // delete me
    if(delta < 0.0 && eta < 0.0)                    // Uniform Distribution
        return minVal + (maxVal-minVal)*randNum;
    else                                            // Weibull Distribution
        return (maxVal - minVal) * pow(-delta*log(randNum*(1.0-exp(-1.0/delta))+exp(-1.0/delta)), 1.0/eta) + minVal;
}

inline void Netsim::writePrtData(ostringstream& out)
{
    for(size_t i = 0; i < m_prtOut.size(); ++i)
        *m_prtOut[i] << out.str() << endl;
}

inline double Netsim::cpuTimeElapsed(clock_t start)
{
    return (static_cast< double >(clock() - start) / CLOCKS_PER_SEC);
}

inline void Netsim::addElemToDrainEvents(RockElem* elem)
{    
    if(elem->addToEventVec(m_oil))
    {
        m_drainageEvents.insert(elem);
        elem->isInOilFloodVec(true);
    }
}

inline void Netsim::addElemToImbEvents(RockElem* elem)
{    
    if(elem->addToEventVec(m_water))
    {
        m_imbibitionEvents.insert(elem);
        elem->isInWatFloodVec(true);
    }
}

inline void Netsim::addElemToReformEvents(RockElem* elem)
{
    Polygon* polyShape = 0;
    vector<int> addCrns;
    double gravCorr(elem->shape()->gravityCorrection());
    if(elem->addToLayerVec(m_oil, &polyShape, addCrns, m_cappPress-gravCorr))
    {
        for(size_t i = 0; i < addCrns.size(); ++i)
        {
            pair<Polygon*, int> entry(polyShape, addCrns[i]);
            m_layerReformEvents.insert(entry);
            polyShape->oilInCorner(addCrns[i])->isInReformVec(true);
        }
    }
}

inline void Netsim::addElemToCollapseEvents(RockElem* elem)
{
    Polygon* polyShape = 0;
    vector<int> addCrns;
    double gravCorr(elem->shape()->gravityCorrection());
    if(elem->addToLayerVec(m_water, &polyShape, addCrns, m_cappPress-gravCorr))
    {
        assert(!addCrns.empty());
        for(size_t i = 0; i < addCrns.size(); ++i)
        {
            assert(!polyShape->oilInCorner(addCrns[i])->isInCollapseVec());                      
            pair<Polygon*, int> entry(polyShape, addCrns[i]);
            m_layerCollapseEvents.insert(entry);
            polyShape->oilInCorner(addCrns[i])->isInCollapseVec(true);
        }
    }
}

inline void Netsim::clearTrappedOil()
{
    while(!m_imbibitionEvents.empty() && m_imbibitionEvents.peek()->trappedOil())
    {
        RockElem* trpElem = m_imbibitionEvents.pop();
        trpElem->isInWatFloodVec(false);
    }
    
    while(!m_layerCollapseEvents.empty() && m_layerCollapseEvents.peek().first->parent()->trappedOil())
    {
        pair<Polygon*, int> trpElem = m_layerCollapseEvents.pop();
        trpElem.first->oilInCorner(trpElem.second)->isInCollapseVec(false);
    }
}

inline void Netsim::clearTrappedWat()
{
    while(!m_drainageEvents.empty() && m_drainageEvents.peek()->trappedWat(bulkBlob))
    {
        RockElem* trpElem = m_drainageEvents.pop();
        trpElem->isInOilFloodVec(false);
    }
    
    while(!m_layerReformEvents.empty() && 
        !m_layerReformEvents.peek().first->oilInCorner(m_layerReformEvents.peek().second)->exists())
    {
        pair<Polygon*, int> trpElem = m_layerReformEvents.pop();
        trpElem.first->oilInCorner(trpElem.second)->isInReformVec(false);
    }    
}

inline void Netsim::recordAmottData(bool drainage)
{
    calculateWaterSat();
    if(drainage)
        m_amottDataDrainage[1] = m_satWater;
    else
        m_amottDataImbibition[1] = m_satWater;
}

inline void Netsim::recordUSBMData(bool drainage)
{
    pair<double, double> entry(m_cappPress, m_satWater);
    if(drainage && m_cappPress > 0.0)
        m_usbmDataDrainage.push_back(entry);
    else if(!drainage && m_cappPress < 0.0)
        m_usbmDataImbibition.push_back(entry);
}

inline double Netsim::tableLookUpX(double yVal, const vector< pair<double, double> >& lookupTable) const
{
    if(yVal <= lookupTable.front().second) 
        return lookupTable.front().first;
    else if(yVal >= lookupTable.back().second) 
        return lookupTable.back().first;
    
    vector< pair<double, double> >::const_iterator itr = lookupTable.begin()+1;
    while(itr->second < yVal && itr != lookupTable.end()) ++itr;    
    assert(itr->second >= yVal && (itr-1)->second < yVal);
    double xOne((itr-1)->first), xTwo(itr->first), yOne((itr-1)->second), yTwo(itr->second);
    double logResX = log10(xOne)+log10(xTwo/xOne)*(yVal-yOne)/(yTwo-yOne);
    return pow(10.0, logResX);
}

inline void Netsim::checkStateOfInitConAng()
{
    for(size_t i = 0; i < m_rockLattice.size(); ++i)
    {
        m_rockLattice[i]->shape()->checkStateOfInitConAng();
    }
}

inline double Netsim::tableLookUpY(double xVal, const vector< pair<double, double> >& lookupTable) const
{
    vector< pair<double, double> >::const_iterator itr = lookupTable.begin()+1;
    while(itr->first > xVal && itr != lookupTable.end()) ++itr;
    assert(itr->first <= xVal && (itr-1)->first > xVal);
    double xOne((itr-1)->first), xTwo(itr->first), yOne((itr-1)->second), yTwo(itr->second);
    return yOne+log10(xVal/xOne)*(yTwo-yOne)/log10(xTwo/xOne);
}

inline void Netsim::assureCollapseVecIntegrity() const
{
    if(m_layerCollapseEvents.size() < 2) return;

    size_t numElem(m_layerCollapseEvents.size());
    double oldCollP = m_layerCollapseEvents.at(0).first->oilInCorner(m_layerCollapseEvents.at(0).second)->gravCorrectedCollapsePc();
    pair<Polygon*, int> oldElem = m_layerCollapseEvents.at(0);
    for(size_t i = 1; i < numElem; ++i)
    {
        pair<Polygon*, int> elem = m_layerCollapseEvents.at(i);
        double collP = elem.first->oilInCorner(elem.second)->gravCorrectedCollapsePc();
        assert(collP >= oldCollP);
        oldCollP = collP;
        oldElem = elem;
    }
}

inline int Netsim::numInstancesInCollapsevec(pair<Polygon*, int> elem) const
{
    int numInst(0);
    size_t numElem(m_layerCollapseEvents.size());
    for(size_t i = 1; i < numElem; ++i)
    {
        if(elem == m_layerCollapseEvents.at(i)) ++numInst;
    }
    return numInst;
}

#endif
