#ifdef WIN32
#pragma warning(disable:4786)
#endif

#include <iostream>
#include <fstream>
#include <sstream>
#include <iomanip>
#include <cmath>
#include <cstdlib>
#include <algorithm>
#include <functional>
#include <typeinfo>
#include <vector>
#include <ctime>
#include <stack>
#include <set>
#include <cassert>
#include <utility>
#include <map>
using namespace std;

#include "f2c.h"
#include "sortedEvents.h"
#include "threeSome.h"
#include "node.h"
#include "fluid.h"
#include "commonData.h"
#include "apex.h"
#include "shape.h"
#include "compareFuncs.h"
#include "inputData.h"

#include "rockElem.h"

bool RockElem::ERROR_STATE = false;
bool RockElem::USE_GRAV_IN_KR = false;
double RockElem::COND_CUT_OFF = 0.0;
const double RockElem::PI = acos(-1.0);

//////////////////////////////////////////////////////////////////////////////////////
//                                  CONSTRUCTORS
//////////////////////////////////////////////////////////////////////////////////////

RockElem::RockElem(CommonData* common, const Fluid* oil, const Fluid* water, double radius, double volume, 
                   double volClay, double shapeFact, double initConAng, int connNum, bool amPore) : m_commonData(common), 
                   m_netVolume(volume), m_clayVolume(volClay), m_connectionNum(connNum), 
                   m_iAmAPore(amPore), m_randomNum(rand())  
{
    m_volumeWater = m_netVolume + m_clayVolume;
    m_waterSaturation = 1.0;
    m_isInsideSolverBox = false;
    m_isInsideSatBox = false;
    m_isInWatFloodVec = false;
    m_isInOilFloodVec = false;
    m_isOnInletSlvrBdr = false;
    m_isOnOutletSlvrBdr = false;
    m_isExitRes = false;
    m_isEntryRes = false;
    m_isInWatFloodVec = false;
    m_isInOilFloodVec = false;
    m_connectedToNetwork = false;
    m_isTrappingExit = false;
    m_isTrappingEntry = false;
    m_connectedToEntryOrExit = false;
    m_canBePassedToSolver.first = false;
    m_canBePassedToSolver.second = false;
    m_touchedInSearch.first = false;
    m_touchedInSearch.second = false;
    m_trappingIndexOil.first = -1;
    m_trappingIndexOil.second = 0.0;
    m_trappingIndexWatBulk.first = -1;
    m_trappingIndexWatBulk.second = 0.0;
    m_trappingIndexWatFilm.first = -1;
    m_trappingIndexWatFilm.second = 0.0;
    m_poreToPoreCond = 0.0;
    m_numOilNeighbours = 0;
    m_fillingEvent = -2;

    if(shapeFact <= sqrt(3.0)/36.0)                   // Triangular:  0 >= G >= sqrt(3.0)/36.0
    {
        m_elemShape = new Triangle(this, m_commonData, oil, water, radius, shapeFact, 
            initConAng, m_connectionNum);
    }
    else if(shapeFact < 0.07)                         // Square:      G == 1.0/16/0 
    {
        m_elemShape = new Square(this, m_commonData, oil, water, radius, initConAng, m_connectionNum);
    }
    else                                                // Circular:    G == 1.0/4.0*PI
    {
        m_elemShape = new Circle(this, m_commonData, oil, water, radius, initConAng, m_connectionNum);
    }
}

RockElem::RockElem(CommonData* common, const Fluid* oil, const Fluid* water, 
                   const RockElem& elm) : m_commonData(common), m_netVolume(elm.m_netVolume), 
                   m_clayVolume(elm.m_clayVolume), m_connectionNum(elm.m_connectionNum), 
                   m_iAmAPore(elm.m_iAmAPore), m_randomNum(elm.m_iAmAPore) 
{
    m_volumeWater = elm.m_volumeWater; 
    m_waterSaturation = elm.m_waterSaturation;
    m_isInsideSolverBox = elm.m_isInsideSolverBox;
    m_isInsideSatBox = elm.m_isInsideSatBox;
    m_isOnInletSlvrBdr = elm.m_isOnInletSlvrBdr;
    m_isOnOutletSlvrBdr = elm.m_isOnOutletSlvrBdr;
    m_isExitRes = elm.m_isExitRes;
    m_isEntryRes = elm.m_isEntryRes;
    m_isInWatFloodVec = elm.m_isInWatFloodVec;
    m_isInOilFloodVec = elm.m_isInOilFloodVec;
    m_connectedToNetwork = elm.m_connectedToNetwork;
    m_isTrappingExit = elm.m_isTrappingExit;
    m_isTrappingEntry = elm.m_isTrappingEntry;
    m_connectedToEntryOrExit = elm.m_connectedToEntryOrExit;
    m_canBePassedToSolver = elm.m_canBePassedToSolver;
    m_touchedInSearch = elm.m_touchedInSearch;
    m_trappingIndexOil = elm.m_trappingIndexOil;
    m_trappingIndexWatBulk = elm.m_trappingIndexWatBulk;
    m_trappingIndexWatFilm = elm.m_trappingIndexWatFilm;
    m_numOilNeighbours = elm.m_numOilNeighbours;
    m_fillingEvent = elm.m_fillingEvent;
    m_averageAspectRatio = elm.m_averageAspectRatio;
    m_minAspectRatio = elm.m_minAspectRatio;
    m_maxAspectRatio = elm.m_maxAspectRatio;

    if(dynamic_cast< Triangle* >(elm.m_elemShape) != NULL)
        m_elemShape = new Triangle(this, m_commonData, oil, water, *dynamic_cast< Triangle* >(elm.m_elemShape));
    else if(dynamic_cast< Square* >(elm.m_elemShape) != NULL)
        m_elemShape = new Square(this, m_commonData, oil, water, *dynamic_cast< Square* >(elm.m_elemShape));
    else
        m_elemShape = new Circle(this, m_commonData, oil, water, *dynamic_cast< Circle* >(elm.m_elemShape));
}

Pore::Pore(CommonData* common, Node *node, const Fluid* oil, const Fluid* water, double radius, double volume, 
           double volumeClay, double shapeFactor, double initConAng, bool insideSlvBox, bool insideSatBox, 
           double initSolverPrs, vector< RockElem* >& connThroats) : RockElem(common, oil, water, 
           radius, volume, volumeClay, shapeFactor, initConAng, static_cast< int >(connThroats.size()), true), m_node(node)
{
    m_isInsideSatBox = insideSatBox;
    m_isInsideSolverBox = insideSlvBox;
    m_connections = connThroats;
    m_oilSolverPrs = initSolverPrs;
    m_watSolverPrs = initSolverPrs;
    m_watSolverVolt = initSolverPrs;
    m_oilSolverVolt = initSolverPrs;
   
    m_elemShape->setGravityCorrection(m_node);

    double minRad(100), maxRad(0.0), radSum(0.0);
    for(size_t i = 0; i < m_connections.size(); ++i)
    {
        double rad = m_connections[i]->shape()->radius();
        minRad = min(minRad, rad);
        maxRad = max(minRad, rad);
        radSum += rad;
    }

    m_averageAspectRatio = m_elemShape->radius()*m_connections.size()/radSum;
    m_maxAspectRatio = m_elemShape->radius()/maxRad;
    m_minAspectRatio = m_elemShape->radius()/minRad;
}

Pore::Pore(CommonData* common, const Fluid* oil, const Fluid* water, const Pore& pore) : RockElem(common, oil, 
           water, pore)
{
    m_oilSolverPrs = pore.m_oilSolverPrs;
    m_watSolverPrs = pore.m_watSolverPrs;
    m_oilSolverVolt = pore.m_oilSolverVolt;
    m_watSolverVolt = pore.m_watSolverVolt;

    m_node = new Node(*pore.m_node);
}

////////////////////////////////////////////////////////////////////////////////////////
// Endpores are assumed to be triangular and have zero volume and contact angles. 
// Radius and interfacial tension is set to some arbitrary value to prevent any 
// divison by zero that may occur. They are further assumed to be triangular so that 
// they wont suddenly not contain water. 
/////////////////////////////////////////////////////////////////////////////////////////
EndPore::EndPore(CommonData* common, Node *node, const Fluid* oil, const Fluid* water, 
                 vector< RockElem* >& connThroats) : Pore(common, node, oil, water, 
                 1.0E-5, 0.0, 0.0, sqrt(3.0)/36.0, 0.0, false, false, 0.0, connThroats)
{    
    m_isInOilFloodVec = true;
    m_isInWatFloodVec = true;
    Polygon* polyShape = dynamic_cast< Polygon* >(m_elemShape);
    for(int i = 0; i < polyShape->numCorners(); ++i)
    {
        polyShape->oilInCorner(i)->isInCollapseVec(true);
        polyShape->oilInCorner(i)->isInReformVec(true);
    }

    m_waterSaturation = 0.5;
    
    if(node->isExitRes())
        m_isExitRes = true;
    else
        m_isEntryRes = true;

    m_isTrappingExit = m_isExitRes;
    m_isTrappingEntry = m_isEntryRes;
}

Throat::Throat(CommonData* common, const Fluid* oil, const Fluid* water, double radius, double volume, double volumeClay, 
               double shapeFactor, double initConAng, double length, double lengthPore1, double lengthPore2, 
               int index) : RockElem(common, oil, water, radius, volume, volumeClay, shapeFactor, 
               initConAng, 2, false), m_latticeIndex(index)
{
    m_length = length;
    m_poreLength.push_back(lengthPore1);
    m_poreLength.push_back(lengthPore2);
    m_originalPoreLengths = 0;
}

Throat::Throat(CommonData* common, const Fluid* oil, const Fluid* water, 
               const Throat& throat) : RockElem(common, oil, water, throat), m_latticeIndex(throat.m_latticeIndex)
{
    m_connectedToEntryOrExit = throat.m_connectedToEntryOrExit;
    m_poreLength = throat.m_poreLength;
    m_length = throat.m_length;
}

Throat::~Throat()
{
    if(!m_originalPoreLengths)
        delete[] m_originalPoreLengths;
}

/////////////////////////////////////////////////////////////////////////////////
// The connecting pores are added to the throat. From the pores it is also 
// possible to determine if the throat is inside the calculation box. If part of
// the connection (pore-throat-pore) is outside the calculation box that lenght
// is set to zero. This has the effect when solving the pressure field, no 
// pressure loss occurs outside the box. However if the pore inside the box itself
// is on the boundary, pressure losses do occur within that pore. This constraint 
// is needed in the case we're using the whole network for rel perm calculations. 
// The first pores are usually located at x position 0.0. These are still within
// the box. There has to be some length to the outlet to make the problem 
// solveable, hence we allow pressure drop to occur within that pore. 
/////////////////////////////////////////////////////////////////////////////////
void Throat::addConnections(RockElem* first, RockElem* second, double inletBdr, double outletBdr, bool moveBoundary) 
{
    if(first->isEntryRes() || first->isExitRes())
        m_elemShape->setGravityCorrection(second->node());
    else if(second->isEntryRes() || second->isExitRes())
        m_elemShape->setGravityCorrection(first->node());
    else
        m_elemShape->setGravityCorrection(first->node(), second->node());
        
    if(first->isEntryRes() || second->isEntryRes())
    {
        m_isTrappingEntry = true;           
        first->isTrappingEntry(first->isEntryRes());      
        second->isTrappingEntry(second->isEntryRes());
    }                                       
    else if(first->isExitRes() || second->isExitRes())
    {
        m_isTrappingExit = true;           
        first->isTrappingExit(first->isExitRes()); 
        second->isTrappingExit(second->isExitRes());
    }                                       
        
    m_isInsideSolverBox = (first->isInsideSolverBox() || second->isInsideSolverBox());
    m_isInsideSatBox = (first->isInsideSatBox() || second->isInsideSatBox());
    m_connectedToEntryOrExit = (first->isExitOrEntryRes() || second->isExitOrEntryRes());
    
    double oldPOneLen(m_poreLength[0]), oldPTwoLen(m_poreLength[1]), oldThrLen(m_length);    
    
    if(m_isInsideSolverBox && !second->isInsideSolverBox())
    {
        second->isOnInletSlvrBdr(second->node()->xPos() < inletBdr);
        second->isOnOutletSlvrBdr(second->node()->xPos() > outletBdr);
        
        if(moveBoundary)
        {
            double scaleFact = (second->node()->xPos() - first->node()->xPos()) / (m_length+m_poreLength[0]+m_poreLength[1]);        
            if(second->isExitOrEntryRes())      // We don't know position of exit and entry res.
                scaleFact = (second->node()->xPos()-first->node()->xPos()) / fabs(second->node()->xPos()-first->node()->xPos());;
            
            double bdr = (second->node()->xPos() < inletBdr) ? inletBdr: outletBdr;
            double throatStart = first->node()->xPos() + m_poreLength[0]*scaleFact;
            double throatEnd = throatStart + m_length*scaleFact;
            m_originalPoreLengths = new double[3];
            m_originalPoreLengths[0] = m_poreLength[0];
            m_originalPoreLengths[1] = m_length;
            m_originalPoreLengths[2] = m_poreLength[1];

            
            if(second->isExitOrEntryRes())                                          // Keep throat lengths if whole model is being used
                m_poreLength[1] = 0.0;                                              // for calculations
            else if(throatEnd > inletBdr && throatEnd < outletBdr)                  // Both pore1 and throat are within the box
                m_poreLength[1] *= (bdr - throatEnd)/(m_poreLength[1]*scaleFact);
            else if(throatStart > inletBdr && throatStart < outletBdr)              // Onle pore 1 is fully within box
            {
                m_poreLength[1] = 0.0;
                m_length *= (bdr - throatStart)/(m_length*scaleFact);
            }
            else                                                                    // Pore 1 is only partially within box
            {
                m_poreLength[1] = 0.0;
                m_length = 0.0;
            }
        }
    }
    else if(m_isInsideSolverBox && !first->isInsideSolverBox())             // Pore 1 is outside box    
    {
        first->isOnInletSlvrBdr(first->node()->xPos() < inletBdr);
        first->isOnOutletSlvrBdr(first->node()->xPos() > outletBdr);            
        
        if(moveBoundary)
        {
            double scaleFact = (first->node()->xPos() - second->node()->xPos()) / (m_length+m_poreLength[0]+m_poreLength[1]);
            if(first->isExitOrEntryRes())       // We don't know position of exit and entry res.
                scaleFact = (first->node()->xPos()-second->node()->xPos()) / fabs(first->node()->xPos()-second->node()->xPos());
            
            double bdr = (first->node()->xPos() < inletBdr) ? inletBdr: outletBdr;
            double throatStart = second->node()->xPos() + m_poreLength[1]*scaleFact;
            double throatEnd = throatStart + m_length*scaleFact;
            m_originalPoreLengths = new double[3];
            m_originalPoreLengths[0] = m_poreLength[0];
            m_originalPoreLengths[1] = m_length;
            m_originalPoreLengths[2] = m_poreLength[1];
           
            if(first->isExitOrEntryRes())       
                m_poreLength[0] = 0.0;
            else if(throatEnd > inletBdr && throatEnd < outletBdr)               // Both pore 2 and throat are within the box
                m_poreLength[0] *= (bdr - throatEnd)/(m_poreLength[0]*scaleFact);
            else if(throatStart > inletBdr && throatStart < outletBdr)      // Only pore 2 is fully within box
            {
                m_poreLength[0] = 0.0;
                m_length *= (bdr - throatStart)/(m_length*scaleFact);
            }
            else                                                            // Pore 2 is only partially within box
            {
                m_poreLength[0] = 0.0;
                m_length = 0.0;
            }
        }
    }
    
    if(m_poreLength[0] > 1.1*oldPOneLen || m_poreLength[1] > 1.1*oldPTwoLen || m_length > 1.1*oldThrLen)
    {
        cout << endl
            << "==============================================="    << endl
            << "Warning: The new lengths for elements connected"    << endl
            << "to the pressure boundary are larger than the"       << endl
            << "original ones. The lengths should be smaller"       << endl
            << "since we do not want pressure drops occuring"       << endl
            << "outside the box across which we're calculating"     << endl
            << "relative permeability."                             << endl
            << "=============================================="     << endl
            << endl;
    }
    
    m_connections.push_back(first);         // Add pore connections
    m_connections.push_back(second);    

    double minRad(100), maxRad(0.0), radSum(0.0);
    for(size_t i = 0; i < m_connections.size(); ++i)
    {
        double rad = m_connections[i]->shape()->radius();
        minRad = min(minRad, rad);
        maxRad = max(minRad, rad);
        radSum += rad;
    }
    m_averageAspectRatio = m_elemShape->radius()*m_connections.size()/radSum;
    m_maxAspectRatio = m_elemShape->radius()/maxRad;
    m_minAspectRatio = m_elemShape->radius()/minRad;
}

void Pore::setRadiusFromAspectRatio(double aspectRatio)
{
    if(aspectRatio < 0.0) aspectRatio = m_averageAspectRatio;
    
    double connRadSum(0.0), maxEqRad(0.0), pi(acos(-1.0));
    for(size_t i = 0; i < m_connections.size(); ++i)
    {
        connRadSum += m_connections[i]->shape()->radius();
        maxEqRad = max((1.0+2.0*sqrt(pi*m_connections[i]->shape()->shapeFactor()))*
            m_connections[i]->shape()->radius()/(1.0+2.0*sqrt(pi*m_elemShape->shapeFactor())), maxEqRad);
    }
    double avrConnRad = connRadSum/m_connections.size();
    m_elemShape->setRadius(max(maxEqRad, aspectRatio*avrConnRad));
}

void Throat::setRadiusFromAspectRatio(double aspectRatio)
{
    if(aspectRatio < 0.0) aspectRatio = m_averageAspectRatio;
    
    double connRadSum(0.0), minEqRad(0.0), pi(acos(-1.0));
    for(size_t i = 0; i < m_connections.size(); ++i)
    {
        connRadSum += m_connections[i]->shape()->radius();
        minEqRad = min((1.0+2.0*sqrt(pi*m_connections[i]->shape()->shapeFactor()))*
            m_connections[i]->shape()->radius()/(1.0+2.0*sqrt(pi*m_elemShape->shapeFactor())), minEqRad);
    }
    double avrConnRad = connRadSum/m_connections.size();
    m_elemShape->setRadius(min(minEqRad, aspectRatio*avrConnRad));
}

//////////////////////////////////////////////////////////////////////////////
// All connecting throats for a pore should already have been set in the
// pore constructor.
/////////////////////////////////////////////////////////////////////////////
void Pore::addConnections(RockElem* first, RockElem* second, double inBdr, double outBdr, bool moveBdr) 
{
    cerr << "Illegal operation" << endl; exit(-1);
}

/////////////////////////////////////////////////////////////////////////////////
// Pore body filling during imbibition is dependent on the radii of connecting
// throats, with the effective radius being on the form Re = Rp + SUM(bxRt) where
// b is some defined weight and x is a random number between 0 and 1. Since the 
// shape class (responsible for calculating entry pressures) do not have a 
// notion of neighbouring radii, this class will have to be responsible for 
// calculating the sum part of the pore body filling expression. BTW, this is
// *WAY* into hand-waving territory. We're also including a weight for I0 event,
// which naturally is always zero. Throats will always pass back a 2 element 
// vector containing zeros.  
/////////////////////////////////////////////////////////////////////////////////
vector< double > RockElem::pbFillingRadiusSum() const 
{
    string poreFillAlg = m_commonData->poreFillAlg();
    vector< double > radiiSum(m_connectionNum, 0.0);

    if(dynamic_cast< const Pore* >(this) != NULL)
    {
        for(int i = 1; i < m_connectionNum; ++i) 
        {
            double radSum = 0.0;
            for(int j = 1; j <= i; ++j)
            {
                assert(m_connectionNum > 1);
                RockElem *nb = m_connections[j-1];
                double weight = (j >= 6) ? m_commonData->radiiWeights(5): m_commonData->radiiWeights(j-1);
                    
                double radContrib(1.0);             
                if(poreFillAlg[0] != 'b') radContrib = nb->shape()->radius();   // Both blunt models don't contain ri
                
                double randNum = static_cast<double>(rand()) / static_cast<double>(RAND_MAX);
                //double randNum = 0.5;     // delete me
                radSum += weight * radContrib * randNum;        
            }
            assert(radSum >= 0.0);
            radiiSum[i] = radSum;
        }
    }
    
    return radiiSum;
}


/////////////////////////////////////////////////////////////////////////////////
// Passes back the prevous pressure determined when solving the laplace eq. 
/////////////////////////////////////////////////////////////////////////////////
bool Pore::prevSolvrRes(const Fluid* fluid, bool resistSolve, double loc, double& res, double& flowRate) const
{
    if(dynamic_cast< const Water* >(fluid) != NULL)
    {
        if(resistSolve)
            res = m_watSolverVolt;
        else
            res = m_watSolverPrs;
    }
    else
    {
        if(resistSolve)
            res = m_oilSolverVolt;
        else
            res = m_oilSolverPrs;
    }

    return true;
}

/////////////////////////////////////////////////////////////////////////////////
// Passes back the prevous pressure determined when solving the laplace eq.
// The routine for throats interpolates between the connecting pore pressures and
// also calculates the flowrate between the pores 
/////////////////////////////////////////////////////////////////////////////////
bool Throat::prevSolvrRes(const Fluid* fluid, bool resistSolve, double loc, double& res, double& flowRate) const
{
    double resZero(0.0), resOne(0.0), tmp(0.0), gravCorrZero(0.0), gravCorrOne(0.0);

    m_connections[0]->prevSolvrRes(fluid, resistSolve, loc, resZero, tmp);
    m_connections[1]->prevSolvrRes(fluid, resistSolve, loc, resOne, tmp);
    
    if(USE_GRAV_IN_KR && !resistSolve)
    {
        gravCorrZero = gravityCorrection(fluid->density(), m_connections[0]->node()->xPos(), 
            m_connections[0]->node()->yPos(), m_connections[0]->node()->zPos());
        gravCorrOne = gravityCorrection(fluid->density(), m_connections[1]->node()->xPos(), 
            m_connections[1]->node()->yPos(), m_connections[1]->node()->zPos());
    }

    if(m_connections[0]->isExitOrEntryRes())                        // Datum correct in/outlet reservoirs. These are always assumed
    {                                                               // to be at the same lavel as connecting pore.
        resZero += gravCorrOne-gravCorrZero;
        gravCorrZero = gravCorrOne;
    }
    
    if(m_connections[1]->isExitOrEntryRes()) 
    {
        resOne += gravCorrZero-gravCorrOne;
        gravCorrOne = gravCorrZero;
    }
    
    double locOne(m_connections[0]->node()->xPos());
    double locTwo(m_connections[1]->node()->xPos());

    if(m_poreToPoreCond > 0.0)
    {
        flowRate = locOne < locTwo ? 
            (resZero-gravCorrZero-resOne+gravCorrOne)*m_poreToPoreCond: 
            (resOne-gravCorrOne-resZero+gravCorrZero)*m_poreToPoreCond;
    }
    else
        flowRate = 0.0;

    resZero -= gravCorrZero;                          // Pressures passed back are datum levelled
    resOne -= gravCorrOne;
    res = resZero + (resOne-resZero) * (loc-locOne) / (locTwo-locOne);
                
    assert((loc >= locOne && loc <= locTwo) || (loc >= locTwo && loc <= locOne));
    assert(res >= min(resZero, resOne) && res <= max(resZero, resOne));
   
    return m_poreToPoreCond > 0.0; 
}

////////////////////////////////////////////////////////////////////////////////////
// Base class iostream class
///////////////////////////////////////////////////////////////////////////////////
ostream& operator<< (ostream& out, RockElem& elem) 
{
    elem.printData(out);
    return out;
}

/////////////////////////////////////////////////////////////////////////////////////
// Pore data is written to output stream, containing following data:
//
// x pos, y pos, z pos, radius, area, shape factor, volume, clay volume, connection number
/////////////////////////////////////////////////////////////////////////////////////
void Pore::printData(ostream& out) const
{      
    m_node->printInfo(out);
    out << *m_elemShape;

    out.flags(ios::showpoint);
    out.flags(ios::scientific);

    double radAspectRatioSum(0.0);
    for(int i = 0; i < m_connectionNum; ++i)
        radAspectRatioSum += m_connections[i]->shape()->radius();
    double aspectRat(m_elemShape->radius()*m_connectionNum/radAspectRatioSum);
   
    out << setprecision(4)
        << setw(15) << m_netVolume*1.0E18
        << setw(15) << m_clayVolume*1.0E18
        << setw(4)  << m_isInsideSolverBox
        << setw(4)  << m_connectionNum
        << setw(4)  << m_isOnInletSlvrBdr
        << setw(4)  << m_isOnOutletSlvrBdr
        << setw(15) << aspectRat;
}

/////////////////////////////////////////////////////////////////////////////////////
// Throat data is written to output stream, containing following data:
//
// radius, area, shape factor, volume, clay volume, throat length
/////////////////////////////////////////////////////////////////////////////////////
void Throat::printData(ostream& out) const
{  
    bool slvrBdrIn(false), slvrBdrOut(false);
    if(m_isInsideSolverBox)
    {
        slvrBdrIn = (m_connections[0]->isOnInletSlvrBdr() || m_connections[1]->isOnInletSlvrBdr());
        slvrBdrOut = (m_connections[0]->isOnOutletSlvrBdr() || m_connections[1]->isOnOutletSlvrBdr());
    }

    out << setw(10) << m_connections[0]->node()->indexOren()
        << setw(10) << m_connections[1]->node()->indexOren()    
        << *m_elemShape;

    out.flags(ios::showpoint);
    out.flags(ios::scientific);
    
    out << setprecision(4)
        << setw(15) << m_netVolume*1.0E18
        << setw(15) << m_clayVolume*1.0E18
        << setw(4)  << m_isInsideSolverBox
        << setw(4)  << slvrBdrIn
        << setw(4)  << slvrBdrOut
        << setw(15) << m_length*1.0E6
        << setw(15) << m_poreLength[0]*1.0E6
        << setw(15) << m_poreLength[1]*1.0E6;
}


/////////////////////////////////////////////////////////////////////////////////////
// These functions checks the integrity on the network, ie that pores and throats 
// agree that they point to each other and that there are no NULL pointers lurking
// about. The total volume and number of connections contained within the network
// is also counted here. 
/////////////////////////////////////////////////////////////////////////////////////
void Pore::checkNetworkIntegrity(double& totNetVolume, double& totClayVolume, int& maxNonZeros, int& isolatedSum) const
{
    if(!m_connectedToNetwork) ++isolatedSum;
    if(m_isInsideSatBox)
    {
        totNetVolume += m_netVolume;
        totClayVolume += m_clayVolume;
    }
    maxNonZeros += m_connectionNum + 1;
    checkConnections();
}

void EndPore::checkNetworkIntegrity(double& totNetVolume, double& totClayVolume, int& maxNonZeros, int& isolatedSum) const
{
    checkConnections();
}

void Throat::checkNetworkIntegrity(double& totNetVolume, double& totClayVolume, int& maxNonZeros, int& isolatedSum) const
{
    if(!m_connectedToNetwork) ++isolatedSum;
    if(m_isInsideSatBox)
    {
        totNetVolume += m_netVolume;
        totClayVolume += m_clayVolume;
    }
    checkConnections();
}

void RockElem::checkConnections() const
{
    if(m_connectionNum != static_cast< int >(m_connections.size()))
    {
        cout << "=========================================" << endl
            << "Error: The connection number is incorrect" << endl
            << "=========================================" << endl;
        ERROR_STATE = true;
    }
    
    for(int conn = 0; conn < m_connectionNum; ++conn)
    {
        if(m_connections[conn] == NULL)
        {
            cout << "==================================="                   << endl
                << "Error: Missing network connecetions"                    << endl
                << "Interanl pointer remains NULL      "                    << endl
                << "Element index: " << orenIndex()                         << endl
                << "Element type: " << typeid(*this).name()                 << endl
                << "Connection: " << conn                                   << endl
                << "Total connection number: " << m_connectionNum           << endl
                << "==================================="                    << endl;
            ERROR_STATE = true;
        }
        
        RockElem* nextElem = m_connections[conn];
        
        bool connExist(false);
        for(int nxtConn = 0; nxtConn < nextElem->connectionNum(); ++nxtConn)
        {
            connExist = (nextElem->connection(nxtConn) == this);
            if (connExist) break;
        }

        if(!connExist)
        {
            cout << "==================================="                               << endl
                << "Error: Missing network connecetions"                                << endl
                << "Interanal links do not match up    "                                << endl
                << "Element index: " << orenIndex()                                     << endl
                << "Element type: " << typeid(*this).name()                             << endl
                << "Opposing index: " << nextElem->orenIndex()                          << endl
                << "Opposing type: " << typeid(*nextElem).name()                        << endl
                << "Connection: " << conn                                               << endl
                << "Total connection number: " << m_connectionNum                       << endl
                << "Opposing total connection number: " << nextElem->connectionNum()    << endl
                << "==================================="                                << endl;
            ERROR_STATE = true;
        }
    }
}

///////////////////////////////////////////////////////////////////////////////////
// Returns the neighbouring pore, going through the throat. Returns a NULL pointer
// if there is no fluid conductance between the pores or if the next pore does not
// exist (ie current pore is on a boundary). The resulting conductance between the
// pores is also calculated. Conductance is passed in as a reference.
///////////////////////////////////////////////////////////////////////////////////
RockElem* Pore::nextPore(int conn, double& conductance, double& deltaGrav, const Fluid *fluid, bool resistivitySolve)
{
    if(conn >= m_connectionNum)
    {
        cerr << "=========================================" << endl
            << "Error: Requesting network connection that" << endl
            << "does not exist.                          " << endl
            << "=========================================" << endl;
        exit(-1);
    }
   
    Throat *throat = dynamic_cast< Throat* >(m_connections[conn]);
    assert(throat);
        
    RockElem *nextPore;
    double nextLength, thisLength, throatLength(throat->length());
    if(throat->connection(0) == this)
    {
        nextPore = throat->connection(1);
        nextLength = throat->poreLength(1);
        thisLength = throat->poreLength(0);
    }
    else
    {   
        nextPore = throat->connection(0);
        nextLength = throat->poreLength(0);
        thisLength = throat->poreLength(1);
    }

    bool filmConnection(m_canBePassedToSolver.first && throat->canBePassedToSolver(filmBlob) && 
        (nextPore->canBePassedToSolver(filmBlob) || 
        (nextPore->isOnSlvrBdr() && nextPore->shape()->containsFluid(fluid))));
    
    bool bulkConnection(m_canBePassedToSolver.second && throat->canBePassedToSolver(bulkBlob) && 
        (nextPore->canBePassedToSolver(bulkBlob) || 
        (nextPore->isOnSlvrBdr() && nextPore->shape()->containsFluid(fluid))));

    bool paralell(filmConnection && bulkConnection && 
        m_elemShape->allOilLayers() && nextPore->shape()->allOilLayers() && throat->shape()->allOilLayers());
    
    if(filmConnection || bulkConnection)
    {        
        double flowResistance(0.0);
        bool singlePhase(nextPore->isExitOrEntryRes());

        double thisCond, throatCond, nextCond;
        if(resistivitySolve)
        {
            double resistivity = fluid->resistivity();
            thisCond = m_elemShape->crossSectionalArea(fluid)/resistivity;
            throatCond = throat->shape()->crossSectionalArea(fluid, singlePhase)/resistivity;
            nextCond = nextPore->shape()->crossSectionalArea(fluid, singlePhase)/resistivity;
            assert(thisCond != 0.0 &&  throatCond != 0.0 && nextCond != 0.0);
            
            flowResistance = (throatLength/throatCond + thisLength/thisCond + nextLength/nextCond);
        }
        else if(paralell)
        {
            assert(dynamic_cast< const Water* >(fluid));
            thisCond = m_elemShape->waterConductance(filmBlob);
            throatCond = throat->shape()->waterConductance(filmBlob, singlePhase);
            nextCond = nextPore->shape()->waterConductance(filmBlob, singlePhase);
            assert(thisCond*throatCond*nextCond != 0.0);
            
            flowResistance = (throatLength/throatCond + thisLength/thisCond + nextLength/nextCond);

            double thisCondBulk = m_elemShape->waterConductance(bulkBlob);
            double throatCondBulk = throat->shape()->waterConductance(bulkBlob, singlePhase);
            double nextCondBulk = nextPore->shape()->waterConductance(bulkBlob, singlePhase);
            assert(thisCondBulk*throatCondBulk*nextCondBulk != 0.0);
            
            double flowResistBulk = (throatLength/throatCondBulk+thisLength/thisCondBulk+nextLength/nextCondBulk);
            
            flowResistance = 1.0/(1.0/flowResistance + 1.0/flowResistBulk);
        }
        else
        {
            thisCond = m_elemShape->conductance(fluid, filmConnection, bulkConnection);
            throatCond = throat->shape()->conductance(fluid, filmConnection, bulkConnection, singlePhase);
            nextCond = nextPore->shape()->conductance(fluid, filmConnection, bulkConnection, singlePhase);
                        
            assert(thisCond*throatCond*nextCond > 0.0);
            
            flowResistance = (throatLength/throatCond + thisLength/thisCond + nextLength/nextCond);
        }
        
        assert(flowResistance > 0.0);

        conductance = 1.0 / flowResistance;

        if(nextPore->isExitOrEntryRes() || resistivitySolve)
            deltaGrav = 0.0;
        else
        {
            deltaGrav = gravityCorrection(fluid->density(), m_node->xPos()-nextPore->node()->xPos(),
                m_node->yPos()-nextPore->node()->yPos(), m_node->zPos()-nextPore->node()->zPos());
        }
        
        throat->poreToPoreCond(conductance);

		if(thisCond <= 0.0 || throatCond <= 0.0 || nextCond <= 0.0 || conductance <= 0.0)
		{
            cout << "Holy crapola. Negative conductances" << endl
                << "This Cond     " << thisCond << endl
				<< "Throat Cond   " << throatCond << endl
				<< "Next Cond     " << nextCond << endl
				<< "Conductance   " << conductance << endl;
			exit(-1);
		}
    }
    else
    {
        conductance = 0.0;
    }

    return nextPore;
}

const RockElem* Throat::nextPore(const RockElem* callingPore) const
{
    if(m_connections.size() != 2) 
    {
        cerr << endl 
            << "============================================" << endl
            << "For optimized network to be written to file " << endl
            << "the option to drain singlets must be enabled" << endl
            << "============================================" << endl;
        exit(-1);
    }
    if(m_connections[0] == callingPore)
        return m_connections[1];
    else
        return m_connections[0];
}



///////////////////////////////////////////////////////////////////////////////////////////
// This is the wrapper function for the trapping routine. In the trapping routine the 
// visited elements are kept track of by inserting them in a vector. If an exit is found
// all the elements in that storage is unmarked and the storage is cleared. If the element
// indeed is trapped the region is given an identifier and the storage is copied such that
// during secondary drainage when this region again might be connected they can all be 
// quickly unmarked.
///////////////////////////////////////////////////////////////////////////////////////////
void RockElem::checkForOilTrapping(double prs, vector< RockElem* >& trappingStorageOil, 
                                   double& elapsed, TrappingCriteria criteria) 
{
    if(!m_isExitRes && !m_isEntryRes && m_elemShape->containsOil())
    {
        clock_t startTrapRoutine(clock());
        if(criteria != escapeToBoth)
        {
            if(foundOutletDepthFirstOil(prs, trappingStorageOil, criteria))    
            {
                for(size_t i = 0; i < trappingStorageOil.size(); ++i)
                {
                    trappingStorageOil[i]->unTrapOil();
                }
                trappingStorageOil.clear();
            }
        }
        else
        {
            if(foundOutletDepthFirstOil(prs, trappingStorageOil, escapeToOutlet))  
            {
                for(size_t i = 0; i < trappingStorageOil.size(); ++i)
                    trappingStorageOil[i]->unTrapOil();
                trappingStorageOil.clear();
                
                if(foundOutletDepthFirstOil(prs, trappingStorageOil, escapeToInlet))
                {
                    for(size_t j = 0; j < trappingStorageOil.size(); ++j)
                    {
                        trappingStorageOil[j]->unTrapOil();
                    }
                    trappingStorageOil.clear();
                }
            }
        }   
        elapsed += (double)(clock() - startTrapRoutine) / CLOCKS_PER_SEC;
    }
}

void RockElem::checkForWatTrapping(double prs, FluidBlob startPt, vector< pair<RockElem*,FluidBlob> >& trappingStorageWat, 
                                   double& elapsed, TrappingCriteria criteria) 
{
    if(!m_isExitRes && !m_isEntryRes && m_elemShape->containsWat(startPt))
    {
        clock_t startTrapRoutine(clock());
        if(criteria != escapeToBoth)
        {
            if(foundOutletDepthFirstWat(prs, startPt, trappingStorageWat, criteria))    
            {
                for(size_t i = 0; i < trappingStorageWat.size(); ++i)
                    trappingStorageWat[i].first->unTrapWat(trappingStorageWat[i].second);
                trappingStorageWat.clear();
            }
        }
        else
        {
            if(foundOutletDepthFirstWat(prs, startPt, trappingStorageWat, escapeToOutlet))  
            {
                for(size_t i = 0; i < trappingStorageWat.size(); ++i)
                    trappingStorageWat[i].first->unTrapWat(trappingStorageWat[i].second);
                trappingStorageWat.clear();
                
                if(foundOutletDepthFirstWat(prs, startPt, trappingStorageWat, escapeToInlet))
                {
                    for(size_t j = 0; j < trappingStorageWat.size(); ++j)
                        trappingStorageWat[j].first->unTrapWat(trappingStorageWat[j].second);
                    trappingStorageWat.clear();
                }
            }
        }        
        elapsed += (double)(clock() - startTrapRoutine) / CLOCKS_PER_SEC;
    }
}

/////////////////////////////////////////////////////////////////////////////////////////////////
// This is the trapping routine. It is a depth first traversal of the network. To prevent stack 
// overflow this routine is implemented without recursion. It could have been made general with
// respect to fluid, however that would have reduced efficiency due to *MANY* checks for fluid.
// This consern was stong enough to warrant seperate implementations for each fluid
/////////////////////////////////////////////////////////////////////////////////////////////////
bool RockElem::foundOutletDepthFirstOil(double pc, vector< RockElem* >& trappingStorage, TrappingCriteria criteria)
{
    RockElem *elemPtr = this;
    stack< RockElem * > elemStack;                      // Simulate recursive behavior using a stack
    elemStack.push(this);
    double datumPc(pc+m_elemShape->gravityCorrection());
    
    while(elemPtr)
    {
        if((elemPtr->isTrappingEntry() && (criteria == escapeToInlet || criteria == escapeToEither)) ||
            (elemPtr->isTrappingExit() && (criteria == escapeToOutlet || criteria == escapeToEither)))
        {
            return true;
        }
        
        double localPc(datumPc - elemPtr->shape()->gravityCorrection());
        assert(!elemPtr->isExitOrEntryRes());
        elemPtr->trapOil(localPc);
        trappingStorage.push_back(elemPtr);
        elemPtr = elemPtr->nextSuccessorOil(criteria);       
        
        while(!elemPtr)
        {
            elemStack.pop();                            // Simulate recursive return. Will unwind the stack
            if(elemStack.empty()) break;                // until a new possible branch is found
            elemPtr = elemStack.top();
            elemPtr = elemPtr->nextSuccessorOil(criteria);
        }
        
        if(elemPtr) elemStack.push(elemPtr);            // Simulate recursive descent
    }
    return false;                                       // Stack is empty. The region is trapped.
}

bool RockElem::foundOutletDepthFirstWat(double pc, FluidBlob startPt, vector< pair<RockElem*,FluidBlob> >& trappingStorage, 
										TrappingCriteria criteria)
{
    pair<RockElem*, FluidBlob> elem(this, startPt);
    stack< pair<RockElem*, FluidBlob> > elemStack;
    elemStack.push(elem);
    double datumPc(pc+m_elemShape->gravityCorrection());
    
    while(elem.first)
    {
        if((elem.first->isTrappingEntry() && (criteria == escapeToInlet || criteria == escapeToEither)) ||
            (elem.first->isTrappingExit() && (criteria == escapeToOutlet || criteria == escapeToEither)))
            return true;            

        assert(!elem.first->isExitOrEntryRes());
        double localPc(datumPc-elem.first->shape()->gravityCorrection());
        elem.first->trapWat(localPc, elem.second);
        trappingStorage.push_back(elem);
        assert(elem.first);
        elem.first = elem.first->nextSuccessorWat(criteria, elem.second);
        
        while(!elem.first)
        {
            elemStack.pop();
            if(elemStack.empty()) break;
            elem = elemStack.top();
            assert(elem.first);
            elem.first = elem.first->nextSuccessorWat(criteria, elem.second);
        }
        
        if(elem.first) elemStack.push(elem);
    }
    return false;
}


///////////////////////////////////////////////////////////////////////////////////////////
// Does a throat cross any given interior plane
///////////////////////////////////////////////////////////////////////////////////////////
bool Throat::crossesPlaneAt(double location) const
{
    double ptOne(m_connections[0]->node()->xPos()), ptTwo(m_connections[1]->node()->xPos());

    if(ptTwo < ptOne)
    {
        double tmp = ptTwo;         // Ensure that that the two points we pick up acyually are within
        ptTwo = ptOne;              // the inteior of the model
        ptOne = tmp;
    }
        
    return ptOne < location && ptTwo >= location;
}

/////////////////////////////////////////////////////////////////////////////////////////
// It is possible that some parts of the network in fact is physically isolated from the 
// network due to diagenesis processes. These parts will remain water filled but are not 
// available for displacement. Identify these elements during initialization by doing a
// graph traversal from the inlet.  
/////////////////////////////////////////////////////////////////////////////////////////
void RockElem::identifyConnectedRockElems()
{
    m_connectedToNetwork = true;                                            // Do allow escapes through entry reservoir
    set< RockElem* > investigatedThroats;                                   

    for(int throat = 0; throat < m_connectionNum; ++throat)
    {
        if(investigatedThroats.count(m_connections[throat]) == 0 &&     // Already failed to find outlet for this
           !m_connections[throat]->connectedToNetwork())                // Might have been marked previously
        {                                                               
            set< RockElem * > frontier, oldFrontier;                     
            frontier.insert(m_connections[throat]);                
            
            while(!frontier.empty())
            {
                oldFrontier = frontier;
                frontier.clear();
                
                for(ItrSet itrElem = oldFrontier.begin(); itrElem != oldFrontier.end(); ++itrElem)
                {
                    if(!(*itrElem)->connectedToNetwork())                   // Flag is not set
                    {
                        (*itrElem)->connectedToNetwork(true);               // Set flag
                        for(int i = 0; i < (*itrElem)->connectionNum(); ++i)
                            frontier.insert((*itrElem)->connection(i));
                    }
                }
            }
            for(int j = 0; j < m_connectionNum; ++j)
            {
                if(m_connections[j]->connectedToNetwork())              // Make sure we don't traverse same region twice. If the
                    investigatedThroats.insert(m_connections[j]);       // elem was not connected it would have been unmarked
            }
        }
    }    
}

/////////////////////////////////////////////////////////////////////////////////////////////////
// Before the solver can solve the pressure field we need to ensure that there is pressure
// communication. Previous trapping routines are not sufficient for this as region connected
// through oil layers might have become trapped without having been identified. This is done 
// through a depth first traversal from the elements at the inlet boundary. All connected 
// elements (through the defined fluid) are marked. These marked elements can then be passed to 
// the solver. The routine is somewhat more involved than the trapping routine since we do not 
// want connections through in/outlet to be marked. What we want is only the region of nodes 
// that are connected both to the in and outlet.
/////////////////////////////////////////////////////////////////////////////////////////////////
bool RockElem::connectedToOutlet(const Fluid* fluid) const
{
    if(!m_elemShape->containsFluid(fluid) && !m_isEntryRes) return false;
    
    bool outletFound(false);
    for(int throat = 0; throat < m_connectionNum; ++throat)
    {
        if(dynamic_cast<const Water*>(fluid))
        {
            pair<const RockElem *, FluidBlob> elem(m_connections[throat], filmBlob);
            
            if(markWaterElemForSolver(elem)) outletFound = true;
            
            elem.second = bulkBlob;
            if(markWaterElemForSolver(elem)) outletFound = true;
        }
        else
        {
            if(markOilElemForSolver(m_connections[throat])) outletFound = true;
        }
    }
    return outletFound;
}

bool RockElem::markWaterElemForSolver(pair<const RockElem*, FluidBlob> elem) const
{
    if((!elem.first->shape()->containsWat(elem.second) && 
        !elem.first->connectedToEntryOrExit()) ||                                          
        elem.first->touchedInSearch(elem.second) ||
        !elem.first->isInsideSolverBox())
    {
        return false;
    }
    
    vector< pair<const RockElem*,FluidBlob> > trappingStorage;
    stack< pair<const RockElem*, FluidBlob> > elemStack;
    elemStack.push(elem);
    bool outletFound(false);
    
    while(elem.first)
    {
        elem.first->setSolverFlag(elem.second);
        trappingStorage.push_back(elem);
        elem.first = elem.first->nextSuccSolvrWat(outletFound, elem.second);
        
        while(!elem.first)
        {
            elemStack.pop();
            if(elemStack.empty()) break;
            elem = elemStack.top();
            elem.first = elem.first->nextSuccSolvrWat(outletFound, elem.second);
        }
        
        if(elem.first) elemStack.push(elem);
    }

    if(!outletFound)
    {
        for(size_t k = 0; k < trappingStorage.size(); ++k)         // If region was isolated it should not be passed 
        {                                                       // to solver
            trappingStorage[k].first->clearSolverFlag(trappingStorage[k].second);
        }
    }
    
    return outletFound;
}

bool RockElem::markOilElemForSolver(const RockElem* elem) const
{
    if((!elem->shape()->containsOil() && 
        !elem->connectedToEntryOrExit()) ||                                          
        elem->touchedInSearch() ||
        !elem->isInsideSolverBox())
    {
        return false;
    }
    
    vector<const RockElem*> trappingStorage;
    stack<const RockElem*> elemStack;
    elemStack.push(elem);
    bool outletFound(false);
    
    while(elem)
    {
        elem->setSolverFlag();
        trappingStorage.push_back(elem);
        elem = elem->nextSuccSolvrOil(outletFound);
        
        while(!elem)
        {
            elemStack.pop();
            if(elemStack.empty()) break;
            elem = elemStack.top();
            elem = elem->nextSuccSolvrOil(outletFound);
        }
        
        if(elem) elemStack.push(elem);
    }

    if(!outletFound)
    {
        for(size_t k = 0; k < trappingStorage.size(); ++k)     // If region was isolated it should not be passed 
        {                                                   // to solver
            trappingStorage[k]->clearSolverFlag();
        }
    }
    
    return outletFound;
}

///////////////////////////////////////////////////////////////////////////////////////////
// We have the option to not drain pores that are connected to the network through a 
// single throat. These pores (and throats) are removed completley from the network 
// initially. This is found to greatly improve solver convergence
///////////////////////////////////////////////////////////////////////////////////////////
int RockElem::removeFromNetwork()
{
    int numRemoved(1);
    m_connectedToNetwork = false;
    for(int i = 0; i < m_connectionNum; ++i)
    {
        if(m_connections[i]->connectedToNetwork())
        {
            if(m_connections[i]->connectionNum() > 2)
            { 
                m_connections[i]->severConnection(this);
                severConnection(m_connections[i]);
            }
            else
                numRemoved += m_connections[i]->removeFromNetwork();
        }
    }
    return numRemoved;
}

///////////////////////////////////////////////////////////////////////////////////////////
// Physically remove the single connetion to the network. This cluster is now isolated
///////////////////////////////////////////////////////////////////////////////////////////
void RockElem::severConnection(RockElem* connection)
{
    assert(m_connectionNum > 0);
    vector< RockElem* >::iterator delCandidate;
    delCandidate = find(m_connections.begin(), m_connections.end(), connection);
    if(*delCandidate == connection)
        m_connections.erase(delCandidate);
    else
    {
        cerr << endl << "oooops..." << endl;
        exit(-1);
    }
    --m_connectionNum;
    m_elemShape->reduceNeighbourCount();
}

//////////////////////////////////////////////////////////////////////////////////////////
// In order to reach the outlet quickly during the trapping routine, we sort the 
// connecting elements according to the distance to outlet
//////////////////////////////////////////////////////////////////////////////////////////
void Pore::sortConnectingElems() 
{
    sort(m_connections.begin(), m_connections.end(), DistCompareThroats());
}

//////////////////////////////////////////////////////////////////////////////////////////
// When sorting the connecting pores in the throats we have to make sure that we also sort
// the associated pore lengths ;-). 
//////////////////////////////////////////////////////////////////////////////////////////
void Throat::sortConnectingElems() 
{
    RockElem *oldFirst = m_connections[0];
    sort(m_connections.begin(), m_connections.end(), DistComparePores());

    assert(m_connections.size() == 2 && m_poreLength.size() == 2);
    if(oldFirst != m_connections[0])    // Pores got switched
    {
        double tmp(m_poreLength[0]);
        m_poreLength[0] = m_poreLength[1];
        m_poreLength[1] = tmp;

        if(m_originalPoreLengths)
        {
            double tmpLen(m_originalPoreLengths[0]);
            m_originalPoreLengths[0] = m_originalPoreLengths[2];
            m_originalPoreLengths[2] = tmpLen;
        }
    }
}

const Node* Throat::node() const 
{
    cerr << "No node associated with throats" << endl; 
    exit(-1);
    return m_connections[0]->node();
}

Node* Throat::node() 
{
    cerr << "No node associated with throats" << endl; 
    exit(-1);
    return m_connections[0]->node();
}

////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// The pore data is written to file in following format:
//
// *_node1.dat (outOne):
// index, x_pos, y_pos, z_pos, connection num, connecting nodes..., at inlet?, at outlet?, connecting links...
//
// *_node2.dat (outTwo):
// index, volume, radius, shape factor, clay volume
////////////////////////////////////////////////////////////////////////////////////////////////////////////////
void Pore::writeNetworkData(ostream& outOne, ostream& outTwo) const
{
    size_t i;
    bool connToIn(false), connToOut(false);
    outOne.flags(ios::showpoint);
    outOne.flags(ios::scientific);
    outTwo.flags(ios::showpoint);
    outTwo.flags(ios::scientific);

    outOne << setw(7)  << orenIndex()
        << *m_node 
        << setw(5) << m_connectionNum;

    for(i = 0; i < m_connections.size(); ++i)
    {
        const Throat* throat = dynamic_cast< Throat* >(m_connections[i]);
        assert(throat);
        const RockElem* nextPore = throat->nextPore(this);
        if(nextPore->isEntryRes()) connToIn = true;
        if(nextPore->isExitRes()) connToOut = true;
            
        outOne << setw(7) << nextPore->orenIndex();                                 // Connecting nodes
    }

    outOne << setw(7) << connToIn << setw(7) << connToOut;                          // In and outlet?

    for(i = 0; i < m_connections.size(); ++i)
        outOne << setw(7) << m_connections[i]->orenIndex();                     // Connecting throats

    outOne << endl;
    
    outTwo << setw(7) << orenIndex() 
        << setw(15) << m_netVolume 
        << setw(15) << m_elemShape->radius()
        << setw(15) << m_elemShape->shapeFactor()
        << setw(15) << m_clayVolume
        << endl;
}

void Pore::writeNetworkDataBinary(ostream& out) const
{
    PoreStruct poreProp;
    poreProp.index = orenIndex();
    poreProp.x = m_node->xPos();
    poreProp.y = m_node->yPos();
    poreProp.z = m_node->zPos();
    poreProp.connNum = m_connectionNum;
    poreProp.volume = m_netVolume;
    poreProp.radius = m_elemShape->radius();
    poreProp.shapeFact = m_elemShape->shapeFactor();
    poreProp.clayVol = m_clayVolume;
    out.write((char *)(&poreProp), sizeof(poreProp));
    
    for(size_t i = 0; i < m_connections.size(); ++i)
    {
        const Throat* throat = dynamic_cast< Throat* >(m_connections[i]);
        assert(throat);
        int idx = throat->nextPore(this)->orenIndex();
        out.write((char *)(&idx), sizeof(int));
    }

    for(size_t j = 0; j < m_connections.size(); ++j)
    {
        int idx = m_connections[j]->orenIndex();
        out.write((char *)(&idx), sizeof(int));
    }
}

/////////////////////////////////////////////////////////////////////////////////////////////////////
// The throat data is written to file in following format:
//
// *_link1.dat (outOne):
// index, pore 1 index, pore 2 index, radius, shape factor, total length (pore center to pore center)
//
// *_link2.dat (outTwo):
// index, pore 1 index, pore 2 index, length pore 1, length pore 2, length throat, volume, clay volume
/////////////////////////////////////////////////////////////////////////////////////////////////////
void Throat::writeNetworkData(ostream& outOne, ostream& outTwo) const
{
    outOne.flags(ios::showpoint);
    outOne.flags(ios::scientific);
    outTwo.flags(ios::showpoint);
    outTwo.flags(ios::scientific);
    double lenPoreOne(m_poreLength[0]), lenPoreTwo(m_poreLength[1]), lenThroat(m_length);
    double lenTotal(m_poreLength[0]+m_poreLength[1]+m_length);

    if(m_originalPoreLengths)
    {
        lenPoreOne = m_originalPoreLengths[0];  // The pore lengths were modified when moving
        lenThroat = m_originalPoreLengths[1];   // pressure boundaries
        lenPoreTwo = m_originalPoreLengths[2];
        lenTotal = lenPoreOne+lenThroat+lenPoreTwo;
    }

    if(m_connections.size() != 2) 
    {
        cerr << endl 
            << "============================================" << endl
            << "For optimized network to be written to file " << endl
            << "the option to drain singlets must be enabled" << endl
            << "============================================" << endl;
        exit(-1);
    }    
    
    outOne << setw(7)   << orenIndex() 
        << setw(7)      << m_connections[0]->orenIndex()
        << setw(7)      << m_connections[1]->orenIndex()
        << setw(15)     << m_elemShape->radius()
        << setw(15)     << m_elemShape->shapeFactor()
        << setw(15)     << lenTotal
        << endl;

    outTwo << setw(7)   << orenIndex()
        << setw(7)      << m_connections[0]->orenIndex()
        << setw(7)      << m_connections[1]->orenIndex()
        << setw(15)     << lenPoreOne
        << setw(15)     << lenPoreTwo
        << setw(15)     << lenThroat
        << setw(15)     << m_netVolume
        << setw(15)     << m_clayVolume
        << endl;
}

void Throat::writeNetworkDataBinary(ostream& out) const
{
    double lenPoreOne(m_poreLength[0]), lenPoreTwo(m_poreLength[1]), lenThroat(m_length);
    double lenTotal(m_poreLength[0]+m_poreLength[1]+m_length);
    if(m_originalPoreLengths)
    {
        lenPoreOne = m_originalPoreLengths[0];  // The pore lengths were modified when moving
        lenThroat = m_originalPoreLengths[1];   // pressure boundaries
        lenPoreTwo = m_originalPoreLengths[2];
        lenTotal = lenPoreOne+lenThroat+lenPoreTwo;
    }

    ThroatStruct throatProp;
    throatProp.index = orenIndex();
    throatProp.poreOne = m_connections[0]->orenIndex();
    throatProp.poreTwo = m_connections[1]->orenIndex();
    throatProp.radius = m_elemShape->radius();
    throatProp.shapeFact = m_elemShape->shapeFactor();
    throatProp.lenPoreOne = lenPoreOne;
    throatProp.lenPoreTwo = lenPoreTwo;
    throatProp.lenThroat = lenThroat;
    throatProp.lenTot = lenTotal;
    throatProp.volume = m_netVolume;
    throatProp.clayVol = m_clayVolume;
    out.write((char *)(&throatProp), sizeof(throatProp));
}

void Throat::modifyLength(double scaleFactor)
{
    m_poreLength[0] *= scaleFactor;
    m_poreLength[1] *= scaleFactor;
    m_length *= scaleFactor;

    if(m_originalPoreLengths)
    {
        m_originalPoreLengths[0] *= scaleFactor;
        m_originalPoreLengths[1] *= scaleFactor;
        m_originalPoreLengths[2] *= scaleFactor;
    }
}

double Throat::lenToRadRatio() const 
{
    assert(m_poreLength.size() == 2);
    return (m_poreLength[0]+m_poreLength[1]+m_length)/m_elemShape->radius();
}



