#ifndef ROCKELEM_H
#define ROCKELEM_H

#include "node.h"
#include "commonData.h"
#include "shape.h"

class Fluid;
class Node;
class commonData;

///////////////////////////////// Base Class //////////////////////////////////////

class RockElem   
{
    
    friend ostream& operator<< (ostream&, RockElem&);
    
public:
    
    RockElem(CommonData*, const Fluid*, const Fluid*, double, double, double, double, double, int, bool);
    RockElem(CommonData*, const Fluid*, const Fluid*, const RockElem&);
    virtual ~RockElem(){delete m_elemShape;}

    virtual double lenToRadRatio() const = 0;
    virtual void modifyLength(double scaleFactor) = 0;
    virtual const Node* node() const = 0;
    virtual Node* node() = 0;
    virtual bool crossesPlaneAt(double location) const = 0;
    virtual int latticeIndex() const = 0;
    virtual int orenIndex() const = 0;
    virtual void addConnections(RockElem* first, RockElem* second, double inBdr, double outBdr, 
        bool moveBdr) = 0; 
    virtual bool prevSolvrRes(const Fluid* fluid, bool resistSolve, double loc, double& res, 
        double& flowRate) const = 0;
    virtual void checkNetworkIntegrity(double& vTot, double& vcTot, int& maxNonO, int& isoSum) const = 0;
    virtual void sortConnectingElems() = 0; 
    virtual void writeNetworkData(ostream& outOne, ostream& outTwo) const = 0;
    virtual void writeNetworkDataBinary(ostream& out) const = 0;
    virtual void updateLatticeIndex(int newIdx) = 0;
    virtual void setRadiusFromAspectRatio(double aspectRadius) = 0;
           
    static void useGravInKr(bool soAreWe) {USE_GRAV_IN_KR = soAreWe;}
    static void conductanceCutOff(double cutOffVal) {COND_CUT_OFF = cutOffVal;}
    RockElem* connection(int conn) const {return m_connections[conn];}
    Shape* shape() {return m_elemShape;}    
    const Shape* shape() const {return m_elemShape;}
    int numOilNeighbours() const {return m_numOilNeighbours;}
    int connectionNum() const {return m_connectionNum;}
    int fillingEvent() const {return m_fillingEvent;}
    double netVolume() const {return m_netVolume;}
    double clayVolume() const {return m_clayVolume;}
	double waterSaturation() const {return m_waterSaturation;}   
    static bool errorState() {return ERROR_STATE;}
    bool isOnInletSlvrBdr() const {return m_isOnInletSlvrBdr;}
    bool isOnOutletSlvrBdr() const {return m_isOnOutletSlvrBdr;}
    bool isOnSlvrBdr() const {return m_isOnOutletSlvrBdr || m_isOnInletSlvrBdr;}
    bool isExitRes() const {return m_isExitRes;}
    bool isEntryRes() const {return m_isEntryRes;}
    bool isExitOrEntryRes() const {return m_isExitRes || m_isEntryRes;}
    bool connectedToEntryOrExit() const {return m_connectedToEntryOrExit;}
    bool isInsideSolverBox() const {return m_isInsideSolverBox;}
    bool isInsideSatBox() const {return m_isInsideSatBox;}
    bool connectedToNetwork() const {return m_connectedToNetwork;}
    bool isTrappingExit() const {return m_isTrappingExit;}
    bool isTrappingEntry() const {return m_isTrappingEntry;}    
    bool isInWatFloodVec() const  {return m_isInWatFloodVec;}
    bool isInOilFloodVec() const  {return m_isInOilFloodVec;}
    void isTrappingExit(bool isIt) {m_isTrappingExit = isIt;}
    void isTrappingEntry(bool isIt) {m_isTrappingEntry = isIt;}    
    void finalizeCopyConstruct(vector< RockElem* >& conn) {m_connections = conn;}
    void isOnInletSlvrBdr(bool isIt) {m_isOnInletSlvrBdr = isIt;}
    void isOnOutletSlvrBdr(bool isIt) {m_isOnOutletSlvrBdr = isIt;}
    void increaseOilNeighbourCount() {++m_numOilNeighbours;}
    void reduceOilNeighbourCount() {--m_numOilNeighbours;}
    void isInWatFloodVec(bool isIt) {m_isInWatFloodVec = isIt;}
    void isInOilFloodVec(bool isIt) {m_isInOilFloodVec = isIt;}
    void connectedToNetwork(bool isIt){m_connectedToNetwork = isIt;}
    void resetFillingEvent() {m_fillingEvent = -2;}
    bool trappedOil() const {return m_trappingIndexOil.first > -1;}
    int trappingIndexOil() const {return m_trappingIndexOil.first;}
    void unTrapOil() {m_trappingIndexOil.first = -1;}
    const pair<int, double>& trappingOil() const {return m_trappingIndexOil;}
    inline const pair<int, double>& trappingWat(FluidBlob blob) const;
    const pair<int, double>& trappingWatBulk() const {return m_trappingIndexWatBulk;}
    const pair<int, double>& trappingWatFilm() const {return m_trappingIndexWatFilm;}
    double averageAspectRatio() const {return m_averageAspectRatio;}
    double minAspectRatio() const {return m_minAspectRatio;}
    double maxAspectRatio() const {return m_maxAspectRatio;}
    bool iAmAPore() const {return m_iAmAPore;}

    vector< double > pbFillingRadiusSum() const; 
    void checkForOilTrapping(double prs, vector< RockElem* >& stor, 
        double& elap, TrappingCriteria crit); 
    void checkForWatTrapping(double prs, FluidBlob startPt, vector< pair<RockElem*,FluidBlob> >& stor, 
        double& elap, TrappingCriteria crit); 
    void identifyConnectedRockElems();
    bool connectedToOutlet(const Fluid* fluid) const;
    int removeFromNetwork();
    unsigned int randomAssInt() const {return m_randomNum;}
    void severConnection(RockElem* connection);
    double oilVolume() const {return (1.0-m_waterSaturation) * m_netVolume + m_clayVolume;}
    double waterVolume() const {return m_waterSaturation * m_netVolume + m_clayVolume;}
   
    inline double gravityCorrection(double density, double xLen, double yLen, double zLen) const;
    inline bool canBePassedToSolver(FluidBlob blob) const;
    inline bool canBePassedToSolver() const;
    inline void clearAllSolverFlag() const;
    inline bool trappedWat(FluidBlob blob) const; 
    inline int trappingIndexWat(FluidBlob blob) const;
    inline bool trapped(FluidBlob blob, const Fluid* fluid) const;
    inline double updateSatAndConductance(double cappPress);
    inline void unTrapWat(FluidBlob blob);
    inline void drain(); 
    inline void imbide(bool snapOffOverRide = false); 
    inline bool addToEventVec(const Fluid* injectant) const;
    inline bool nonTrappedOilNeighbour(const Fluid* injectant) const;
    inline bool nonTrappedWatNeighbour(const Fluid* injectant) const;
    inline bool addToLayerVec(const Fluid* injectant, Polygon** polyShape, vector<int>& addCrn, double pc) const;
    inline void adjustVolume(double newNetVol, double newClayVol, double& netVolSum, double& clayVolSum);
    inline void copyWatFilmTrappingToBulk();
   
protected:

    virtual void printData(ostream& out) const = 0;

    inline RockElem* nextSuccessorOil(TrappingCriteria criteria); 
    inline RockElem* nextSuccessorWat(TrappingCriteria criteria, FluidBlob& blob); 
    inline const RockElem* nextSuccSolvrWat(bool& outletFound, FluidBlob& blob) const; 
    inline const RockElem* nextSuccSolvrOil(bool& outletFound) const; 
    inline void trapOil(double prs);
    inline void clearSolverFlag(FluidBlob blob = bulkBlob) const;
    inline void trapWat(double prs, FluidBlob blob); 
    inline void setSolverFlag(FluidBlob blob = bulkBlob) const; 
    inline bool touchedInSearch(FluidBlob blob = bulkBlob) const;

    bool foundOutletDepthFirstOil(double pc, vector<RockElem*>& stor, TrappingCriteria crit);
    bool foundOutletDepthFirstWat(double pc, FluidBlob startPt, vector< pair<RockElem*, FluidBlob> >& stor, 
        TrappingCriteria criteria);
    bool markWaterElemForSolver(pair<const RockElem*, FluidBlob> elem) const;
    bool markOilElemForSolver(const RockElem* elem) const;
    void checkConnections() const;
    
    typedef set< RockElem* >::iterator ItrSet;   
            
    static bool                             ERROR_STATE;
    static bool                             USE_GRAV_IN_KR;
    static double                           COND_CUT_OFF;
    static const double                     PI;

    const unsigned int                      m_randomNum;
    const bool                              m_iAmAPore;
    CommonData*                             m_commonData;
    double                                  m_netVolume;
    double                                  m_clayVolume;
    int                                     m_connectionNum;

    vector< RockElem* >                     m_connections; 
    Shape*                                  m_elemShape;
    double                                  m_volumeWater; 
	double									m_waterSaturation;
    double                                  m_averageAspectRatio;
    double                                  m_minAspectRatio;
    double                                  m_maxAspectRatio;
    bool                                    m_isInsideSolverBox;
    bool                                    m_isInsideSatBox;
    bool                                    m_isOnInletSlvrBdr;
    bool                                    m_isOnOutletSlvrBdr;
    bool                                    m_isExitRes;
    bool                                    m_isEntryRes;
    bool                                    m_isInWatFloodVec;
    bool                                    m_isInOilFloodVec;
    bool                                    m_connectedToNetwork;
    bool                                    m_isTrappingExit;
    bool                                    m_isTrappingEntry;
    bool                                    m_connectedToEntryOrExit;
    mutable pair<bool,bool>                 m_canBePassedToSolver;
    mutable pair<bool,bool>                 m_touchedInSearch;
    mutable double                          m_poreToPoreCond;
    pair<int, double>                       m_trappingIndexOil;
    pair<int, double>                       m_trappingIndexWatBulk;
    pair<int, double>                       m_trappingIndexWatFilm;
    int                                     m_numOilNeighbours;
    int                                     m_fillingEvent;
};

///////////////////////////////// Pore Class //////////////////////////////////////

class Pore : public RockElem
{
public:
    
    Pore(CommonData*, Node*, const Fluid*, const Fluid*, double, double, double, double,
        double, bool, bool, double, vector< RockElem* >&);
    Pore(CommonData*, const Fluid*, const Fluid*, const Pore&);
    virtual ~Pore(){delete m_node;}
    
    virtual double lenToRadRatio() const {return 0.0;}
    virtual void modifyLength(double scaleFactor) {}
    virtual const Node* node() const {return m_node;}
    virtual Node* node() {return m_node;}
    virtual bool crossesPlaneAt(double location) const {return false;}
    virtual int latticeIndex() const {return m_node->index();}
    virtual int orenIndex() const {return m_node->indexOren();}
    virtual void addConnections(RockElem* first, RockElem* second, double inBdr, 
        double outBdr, bool moveBdr); 
    virtual bool prevSolvrRes(const Fluid* fluid, bool resistSolve, double loc, double& res, 
        double& flowRate) const;
    virtual void checkNetworkIntegrity(double& vTot, double& vcTot, int& maxNonO, int& isoSum) const;
    virtual void sortConnectingElems(); 
    virtual void writeNetworkData(ostream& outOne, ostream& outTwo) const;
    virtual void writeNetworkDataBinary(ostream& out) const;
    virtual void updateLatticeIndex(int newIdx) {}
    virtual void setRadiusFromAspectRatio(double aspectRadius);

    RockElem* nextPore(int conn, double& conductance, double& deltaGrav, const Fluid *fluid, bool resistivitySolve);

    inline void solverResults(const Fluid* fluid, bool resistSolve, double res);
    
protected:

    virtual void printData(ostream& out) const;
    
    Node                            *m_node; 
    double                          m_oilSolverPrs;
    double                          m_watSolverPrs;
    double                          m_oilSolverVolt;
    double                          m_watSolverVolt;
};

///////////////////////////////// Inlet & Outlet Pore /////////////////////////////////////

class EndPore : public Pore
{
public:
    
    EndPore(CommonData*, Node*, const Fluid*, const Fluid*, vector< RockElem* >&);
    EndPore(CommonData* common, const Fluid* oil, const Fluid* water, const EndPore& endP) : Pore(common, oil, water, endP) {}
    virtual ~EndPore(){}

    virtual void checkNetworkIntegrity(double& vTot, double& vcTot, int& maxNonO, int& isoSum) const;
    
};

//////////////////////////////////////// Throat //////////////////////////////////////////

class Throat : public RockElem
{
public:
    
    Throat(CommonData*, const Fluid*, const Fluid*, double, double, double, double, double, 
        double, double, double, int);
    Throat(CommonData*, const Fluid*, const Fluid*, const Throat&);
    virtual ~Throat();
    
    virtual double lenToRadRatio() const; 
    virtual void modifyLength(double scaleFactor);
    virtual const Node* node() const;
    virtual Node* node();
    virtual bool crossesPlaneAt(double location) const;
    virtual int latticeIndex() const {return m_latticeIndex;}
    virtual int orenIndex() const {return m_latticeIndex-m_commonData->numPores()-1;}
    virtual void addConnections(RockElem* first, RockElem* second, double inletBdr, 
        double outletBdr, bool moveBoundary); 
    virtual bool prevSolvrRes(const Fluid* fluid, bool resistSolve, double loc, double& res, 
        double& flowRate) const;
    virtual void checkNetworkIntegrity(double& vTot, double& vcTot, int& maxNonO, int& isoSum) const;
    virtual void sortConnectingElems(); 
    virtual void writeNetworkData(ostream& outOne, ostream& outTwo) const;
    virtual void writeNetworkDataBinary(ostream& out) const;
    virtual void updateLatticeIndex(int newIdx) {m_latticeIndex = newIdx;}
    virtual void setRadiusFromAspectRatio(double aspectRadius);

    const RockElem* nextPore(const RockElem* callingPore) const;

    double poreLength(int conn) const {return m_poreLength[conn];}
    double length() const {return m_length;}
    void poreToPoreCond(double cond) {m_poreToPoreCond = cond;}

private:

    virtual void printData(ostream& out) const;
    
    int                                 m_latticeIndex;
    double*                             m_originalPoreLengths;
    vector< double >                    m_poreLength;
    double                              m_length;
};
///////////////////////////////////////////////////////////////////////////////////////
//////////////////////////// Inline Func Definitions //////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////

inline void RockElem::adjustVolume(double newNetVol, double newClayVol, 
                                   double& netVolSum, double& clayVolSum)
{
    m_netVolume = newNetVol;
    m_clayVolume = newClayVol;
    m_volumeWater = m_netVolume + m_clayVolume;

    if(m_isInsideSatBox)
    {
        netVolSum += m_netVolume;
        clayVolSum += m_clayVolume;
    }
}

inline const pair<int, double>& RockElem::trappingWat(FluidBlob blob) const
{
    if(blob == filmBlob)
        return m_trappingIndexWatFilm;
    else
        return m_trappingIndexWatBulk;
}

////////////////////////////////////////////////////////////////////////////////
// During depth first graph traversal we need to retrive the next sucsessor
// after a elemnt. This will be an element that is not marked and contains
// the fluid in question (oil in this case)
////////////////////////////////////////////////////////////////////////////////
inline RockElem* RockElem::nextSuccessorOil(TrappingCriteria criteria) 
{
    if(criteria == escapeToInlet)
    {
        for(short i = m_connectionNum; i > 0; --i) 
        {
            if(!m_connections[i-1]->isExitOrEntryRes() &&
                !m_connections[i-1]->trappedOil() && 
                m_connections[i-1]->shape()->containsOil())
                return m_connections[i-1];
        }
    }
    else
    {
        for(short i = 0; i < m_connectionNum; ++i) 
        {
            if(!m_connections[i]->isExitOrEntryRes() &&
                !m_connections[i]->trappedOil() && 
                m_connections[i]->shape()->containsOil())
                return m_connections[i];
        }
    }
    return NULL;
}

inline RockElem* RockElem::nextSuccessorWat(TrappingCriteria criteria, FluidBlob& blob) 
{
    if(criteria == escapeToInlet)
    {
        for(short i = m_connectionNum; i > 0; --i) 
        {
            if(!m_connections[i-1]->isExitOrEntryRes() &&
                !m_connections[i-1]->trappedWat(blob) && 
                m_connections[i-1]->shape()->containsWat(blob))
                //(m_connections[i-1]->shape()->containsWat(blob) || m_connections[i-1]->connectedToEntryOrExit()))
                return m_connections[i-1];
        }
    }
    else
    {
        for(short i = 0; i < m_connectionNum; ++i) 
        {
            if(!m_connections[i]->isExitOrEntryRes() &&
                !m_connections[i]->trappedWat(blob) && 
                m_connections[i]->shape()->containsWat(blob))
                //(m_connections[i]->shape()->containsWat(blob) || m_connections[i]->connectedToEntryOrExit()))
                return m_connections[i];
        }
    }

    FluidBlob blobbo(blob == filmBlob ? bulkBlob: filmBlob);
    if(!m_elemShape->allOilLayers() && m_elemShape->containsWat(blobbo) && !trappedWat(blobbo))
    {
        blob = blobbo;
        return this;
    }

    return NULL;
}

inline const RockElem* RockElem::nextSuccSolvrWat(bool& outletFound, FluidBlob& blob) const 
{
    for(short i = 0; i < m_connectionNum; ++i) 
    {
        if(!m_connections[i]->touchedInSearch(blob) && m_connections[i]->isInsideSolverBox() &&
            (m_connections[i]->shape()->containsWat(blob) || m_connections[i]->connectedToEntryOrExit()))
        {
            return m_connections[i];
        }
        else if(m_connections[i]->isOnOutletSlvrBdr() && 
            (m_connections[i]->shape()->containsWat() || m_connections[i]->isExitOrEntryRes()))
        {
            outletFound = true;
        }
    }
    
    FluidBlob blobbo(blob == filmBlob ? bulkBlob: filmBlob);
    if(!m_elemShape->allOilLayers() && m_elemShape->containsWat(blobbo) && !touchedInSearch(blobbo))
    {
        blob = blobbo;
        return this;
    }
    
    return NULL;
}

inline const RockElem* RockElem::nextSuccSolvrOil(bool& outletFound) const 
{
    for(short i = 0; i < m_connectionNum; ++i) 
    {
        if(!m_connections[i]->touchedInSearch(bulkBlob) && 
            m_connections[i]->isInsideSolverBox() && 
            (m_connections[i]->shape()->containsOil() || m_connections[i]->connectedToEntryOrExit()) && 
            (m_connections[i]->shape()->oilConductance() > COND_CUT_OFF || m_connections[i]->connectedToEntryOrExit())) 
        {
            return m_connections[i];
        }
        else if(m_connections[i]->isOnOutletSlvrBdr() && 
            (m_connections[i]->shape()->containsOil() || m_connections[i]->isExitOrEntryRes()))
        {
            outletFound = true;
        }
    }
    
    return NULL;
}


inline bool RockElem::touchedInSearch(FluidBlob blob) const
{
	if(blob == bulkBlob)
        return m_touchedInSearch.second;
	else
        return m_touchedInSearch.first;
}

inline bool RockElem::canBePassedToSolver(FluidBlob blob) const
{
	if(blob == bulkBlob)
        return m_canBePassedToSolver.second;
	else
        return m_canBePassedToSolver.first;
}

inline bool RockElem::canBePassedToSolver() const
{
	return m_canBePassedToSolver.second || m_canBePassedToSolver.first;
}

inline void RockElem::setSolverFlag(FluidBlob blob) const 
{
    if(blob == bulkBlob)
    {
		m_canBePassedToSolver.second = true;
        m_touchedInSearch.second = true;
    }
	else
    {
		m_canBePassedToSolver.first = true;
        m_touchedInSearch.first = true;
    }
}

inline void RockElem::clearAllSolverFlag() const
{
    m_canBePassedToSolver.first = false;
    m_touchedInSearch.first = false;
    m_canBePassedToSolver.second = false;
    m_touchedInSearch.second = false;
    m_poreToPoreCond = 0.0;
}

inline void RockElem::clearSolverFlag(FluidBlob blob) const
{
    if(blob == bulkBlob)
		m_canBePassedToSolver.second = false;
	else
		m_canBePassedToSolver.first = false;
}

inline int RockElem::trappingIndexWat(FluidBlob blob) const
{
    if(blob == filmBlob)
        return m_trappingIndexWatFilm.first;
    else
        return m_trappingIndexWatBulk.first;
}


inline bool RockElem::trappedWat(FluidBlob blob) const 
{
	if(blob == filmBlob)
		return m_trappingIndexWatFilm.first > -1;
	else
		return m_trappingIndexWatBulk.first > -1;
}

inline bool RockElem::trapped(FluidBlob blob, const Fluid* fluid) const
{
    if(dynamic_cast< const Oil* >(fluid))
        return m_trappingIndexOil.first > -1;
    else if(blob == filmBlob)
        return m_trappingIndexWatFilm.first > -1;
    else
        return m_trappingIndexWatBulk.first > -1;
}

///////////////////////////////////////////////////////////////////////////////////////////
// The corner conductancees are computed using the suggested procedure by Patzek SPE59312
// Elements that are trapped have their capillary pressure frozen at the time they're
// identified as being trapped.
///////////////////////////////////////////////////////////////////////////////////////////
inline double RockElem::updateSatAndConductance(double cappPress)
{
    if(m_isEntryRes || m_isExitRes) return 0.0;

    m_waterSaturation = m_elemShape->updateState(cappPress);  
    m_volumeWater = m_waterSaturation * m_netVolume + m_clayVolume;

    return m_isInsideSatBox ? m_volumeWater: 0.0;
}


////////////////////////////////////////////////////////////////////////////////
// Marking during the trapping routine is done by increasing the trapping index.
// If the outlet is subsequently found, it is reset down to 0. 
////////////////////////////////////////////////////////////////////////////////
inline void RockElem::trapOil(double prs) 
{
    m_trappingIndexOil.first = static_cast< int >(m_commonData->newOilTrappingIndex());
    m_trappingIndexOil.second = prs;
}

inline void RockElem::trapWat(double prs, FluidBlob blob) 
{
    if(blob == bulkBlob) 
	{
		m_trappingIndexWatBulk.first = static_cast< int >(m_commonData->newWatTrappingIndex());
		m_trappingIndexWatBulk.second = prs;
	}
	else
	{
		m_trappingIndexWatFilm.first = static_cast< int >(m_commonData->newWatTrappingIndex());
		m_trappingIndexWatFilm.second = prs;
	}
}

inline void RockElem::unTrapWat(FluidBlob blob)
{
    if(blob == bulkBlob) 
		m_trappingIndexWatBulk.first = -1;
	else
		m_trappingIndexWatFilm.first = -1;
}


///////////////////////////////////////////////////////////////////////////////
// As an elemnt is drained, its neighbours have an increase in number of elems
// containing oil.
//////////////////////////////////////////////////////////////////////////////
inline void RockElem::drain() 
{
    assert(dynamic_cast< const Water* >(m_elemShape->bulkFluid()));

    bool snappedOff = m_elemShape->fillWithOil();
    for(int i = 0; i < m_connectionNum; ++i)
        m_connections[i]->increaseOilNeighbourCount();
    
    if(snappedOff)
    {
        m_elemShape->addFillingEventToHistory(2);
        m_fillingEvent = -1;
    }
    else
    {
        m_elemShape->addFillingEventToHistory(1);
        m_fillingEvent = m_connectionNum - m_numOilNeighbours;
    }
}

/////////////////////////////////////////////////////////////////////////////////
// As an elemt is imbided we need to update the imbibition entry pressures of
// neighbouring elements.
////////////////////////////////////////////////////////////////////////////////
inline void RockElem::imbide(bool snapOffOverRide) 
{
    assert(dynamic_cast< const Oil* >(m_elemShape->bulkFluid()));
    bool snappedOff = m_elemShape->fillWithWater(snapOffOverRide);
    for(int i = 0; i < m_connectionNum; ++i) 
        m_connections[i]->reduceOilNeighbourCount();

    if(snappedOff)
    {
        m_elemShape->addFillingEventToHistory(4);
        m_fillingEvent = -1;
    }
    else
    {
        m_elemShape->addFillingEventToHistory(3);
        m_fillingEvent = m_numOilNeighbours;
    }
}

inline void Pore::solverResults(const Fluid* fluid, bool resistSolve, double res)
{
    assert(canBePassedToSolver() || m_isOnInletSlvrBdr || m_isOnOutletSlvrBdr);

    if(dynamic_cast< const Water* >(fluid) != NULL)
    {
        if(resistSolve)
            m_watSolverVolt = res;
        else
            m_watSolverPrs = res;
    }
    else
    {
        if(resistSolve)
            m_oilSolverVolt = res;
        else
            m_oilSolverPrs = res;
    }
}

inline bool RockElem::addToEventVec(const Fluid* injectant) const
{
    if(m_elemShape->bulkFluid() == injectant || 
        m_trappingIndexOil.first > -1 ||
        !m_connectedToNetwork) 
    {
        return false;
    }

    if(dynamic_cast< const Water* >(injectant))
        return !m_isInWatFloodVec && m_trappingIndexWatFilm.first == -1 && (m_elemShape->containsWat() || nonTrappedWatNeighbour(injectant));
    else
        return !m_isInOilFloodVec && m_trappingIndexWatBulk.first == -1 && (m_elemShape->containsOil() || nonTrappedOilNeighbour(injectant));
}

inline bool RockElem::nonTrappedOilNeighbour(const Fluid* injectant) const
{
    if(m_numOilNeighbours == 0) return false;

    for(int i = 0; i < m_connectionNum; ++i)
    {
        if(m_connections[i]->shape()->bulkFluid() == injectant && !m_connections[i]->trappedOil())
            return true;
    }
    return false;
}

inline bool RockElem::nonTrappedWatNeighbour(const Fluid* injectant) const
{
    if(m_numOilNeighbours == m_connectionNum) return false;

    for(int i = 0; i < m_connectionNum; ++i)
    {
        if(m_connections[i]->shape()->bulkFluid() == injectant && !m_connections[i]->trappedWat(bulkBlob))
            return true;
    }
    return false;
}


inline bool RockElem::addToLayerVec(const Fluid* injectant, Polygon** polyShape, vector< int >& cornToAdd, double pc) const
{
    *polyShape = dynamic_cast< Polygon* >(m_elemShape);
    if(!(*polyShape)) return false;
    
    const Oil* oil = dynamic_cast< const Oil* >(injectant);
    
    if((*polyShape)->bulkFluid() == oil || !(*polyShape)->oilInCorner(0)->exists()) return false;

    if(oil)     // Layer reformation
    {
        for(int i = 0; i < (*polyShape)->numCorners(); ++i)
        {
            if(!(*polyShape)->oilInCorner(i)->isInReformVec() &&
                (*polyShape)->oilInCorner(i)->exists() &&
                !(*polyShape)->oilInCorner(i)->stable())
            {
                cornToAdd.push_back(i);
            }
        }

        if(!cornToAdd.empty() && m_trappingIndexOil.first == -1)
        {
            for(size_t j = 0; j < m_connections.size(); ++j)
            {
                if(m_connections[j]->shape()->containsOil() && m_connections[j]->trappingIndexOil() == -1)
                {
                    return true;
                }
            }
        }
        return false;
    }
    else        // Layer collapse
    {
        for(int i = 0; i < (*polyShape)->numCorners(); ++i)
        {
            if(!(*polyShape)->oilInCorner(i)->isInCollapseVec() &&
                (*polyShape)->oilInCorner(i)->exists() &&
                (*polyShape)->oilInCorner(i)->stable())
            {
                cornToAdd.push_back(i);
            }
        }
              
        if(!cornToAdd.empty())
        {
            if(m_trappingIndexOil.first == -1 && (m_trappingIndexWatFilm.first == -1 || m_trappingIndexWatBulk.first == -1))
                return true;
        }
        return false;
    }
}

inline void RockElem::copyWatFilmTrappingToBulk() 
{
    assert(m_trappingIndexWatBulk.first == -1); 
    m_trappingIndexWatFilm = m_trappingIndexWatBulk;
}

inline double RockElem::gravityCorrection(double density, double xLen, double yLen, double zLen) const
{
    return density*(m_commonData->gravConstX()*xLen+m_commonData->gravConstY()*yLen+m_commonData->gravConstZ()*zLen);
}
    
#endif
