#ifdef WIN32        
#pragma warning(disable:4786)        
#endif

#include <iostream>
#include <fstream>
#include <sstream>
#include <iomanip>
#include <cmath>
#include <algorithm>
#include <vector>
#include <cassert>
#include <utility>
#include <set>
#include <map>
using namespace std;

#include "f2c.h"
#include "sortedEvents.h"
#include "threeSome.h"
#include "fluid.h"
#include "apex.h"
#include "rockElem.h"
#include "commonData.h"

#include "shape.h"

const int       Shape::MAX_NEWT_ITR =       1000;
const double    Shape::INF_NEG_NUM =        -1.0E21;
const double    Shape::EPSILON =            1.0E-6;
const double    Shape::PI =                 acos(-1.0);

///////////////////////////////////////////////////////////////////////////////////////////
// Base Class Constructors
///////////////////////////////////////////////////////////////////////////////////////////
Shape::Shape(RockElem* parent, CommonData* common, const Fluid* oil, const Fluid* water, double radius, 
             double shapeFactor, double initConAng, int connNum): m_parent(parent), m_oil(oil), 
	     m_water(water), m_radius(radius), m_shapeFactor(shapeFactor), m_conAngleInit(initConAng), 
             m_bulkFluid(water), m_numNeighbours(connNum) 
{
    m_commonData = common;
    m_watCondMultFactWhenOilFilled = m_commonData->circWatCondMultFact();
    m_containsWater = true; 
    m_containsOil = false;
    m_anyOilLayers = false;
    m_allOilLayers = false;
    m_virginState = true;
    m_wettingCluster = 0;
    m_conAngleAdv = 0.0;
    m_conAngleRec = 0.0;
    m_conAngEquil = 0.0;

    m_area = pow(m_radius, 2.0) / (4.0 * m_shapeFactor);     //From Oren; This is ~correct for all shape
    m_areaWater = m_area;
    assure(m_area > 0.0, "1");
}

Shape::Shape(RockElem* parent, CommonData* common, const Fluid* oil, const Fluid* water, 
             const Shape& shapeCp) : m_parent(parent), m_oil(oil), m_water(water), 
             m_radius(shapeCp.m_radius), m_shapeFactor(shapeCp.m_shapeFactor), 
             m_conAngleInit(shapeCp.m_conAngleInit), m_bulkFluid(water), 
	     m_numNeighbours(shapeCp.m_numNeighbours)  
	      
{
    m_commonData = common;    
    m_conAngleRec = shapeCp.m_conAngleRec;
    m_conAngleAdv = shapeCp.m_conAngleAdv;
    m_containsWater = shapeCp.m_containsWater;
    m_containsOil = shapeCp.m_containsOil;
    m_area = shapeCp.m_area;
    m_areaWater = shapeCp.m_areaWater;    
    m_conductanceOil = shapeCp.m_conductanceOil;
    m_conductanceWater = shapeCp.m_conductanceWater;
    m_entryPress = shapeCp.m_entryPress;
    m_maxConAngSpont = shapeCp.m_maxConAngSpont;
    m_pistonTypeCurveRad = shapeCp.m_pistonTypeCurveRad;    
    m_waterSatHistory = shapeCp.m_waterSatHistory;
    m_anyOilLayers = shapeCp.m_anyOilLayers;
    m_allOilLayers = shapeCp.m_allOilLayers;
    m_watCondMultFactWhenOilFilled = shapeCp.m_watCondMultFactWhenOilFilled;
}

///////////////////////////////////////////////////////////////////////////////////////////
// Polygon Class Constructors
///////////////////////////////////////////////////////////////////////////////////////////
Polygon::Polygon(RockElem* parent, CommonData* common, const Fluid* oil, const Fluid* water, 
                 double radius, double shapeFactor, double initConAng, int numCorners, 
                 int connNum) : Shape(parent, common, oil, water, radius, shapeFactor, 
                 initConAng, connNum) 
{
    m_numCorners = numCorners;
    m_numLayers = 0;
    m_maxConAngSpont = PI;  

    for(int i = 0; i < m_numCorners; ++i)
    {
        CornerApex *corner = new CornerApex(initConAng, this);
        LayerApex *layer = new LayerApex(corner, this);
        m_waterInCorner.push_back(corner);
        m_oilInCorner.push_back(layer);
    }
}

Polygon::Polygon(RockElem* parent, CommonData* common, const Fluid* oil, const Fluid* water, 
                 const Polygon& shapeCp) : Shape(parent, common, oil, water, shapeCp)
{
    m_cornerHalfAngles = shapeCp.m_cornerHalfAngles;
    m_snapOffPrs = shapeCp.m_snapOffPrs;
    m_numCorners = shapeCp.m_numCorners;
    for(int i = 0; i < m_numCorners; ++i)
    {
        CornerApex *corner = new CornerApex(*shapeCp.m_waterInCorner[i], this);
        LayerApex *layer = new LayerApex(corner, *shapeCp.m_oilInCorner[i], this);
        m_waterInCorner.push_back(corner);
        m_oilInCorner.push_back(layer);
    }
}

///////////////////////////////////////////////////////////////////////////////////////////
// Polygon Class Destructor
///////////////////////////////////////////////////////////////////////////////////////////
Polygon::~Polygon()
{
    for(int i = 0; i < m_numCorners; ++i)
    {
        delete m_waterInCorner[i];
        delete m_oilInCorner[i];
    }
}
///////////////////////////////////////////////////////////////////////////////////////////
// Triangle Class Constructors
///////////////////////////////////////////////////////////////////////////////////////////
Triangle::Triangle(RockElem* parent, CommonData* common, const Fluid* oil, const Fluid* water, double radius, 
                   double shapeFactor, double initConAng, int connNum) : Polygon(parent, common, oil, water, 
                   radius, shapeFactor, initConAng, 3, connNum)
{
    m_commonData->addTriangle();    
    m_cornerHalfAngles.resize(m_numCorners);
    getHalfAngles();

    m_conductanceOil = 0.0;
    m_conductanceWater.first = 0.0;
    m_conductanceWater.second = centerConductance(m_areaWater, m_water->viscosity());
}

Triangle::Triangle(RockElem* parent, CommonData* common, const Fluid* oil, const Fluid* water, const Triangle& shapeCp) : 
                   Polygon(parent, common, oil, water, shapeCp)
{
}

///////////////////////////////////////////////////////////////////////////////////////////
// Square Class Constructors
///////////////////////////////////////////////////////////////////////////////////////////
Square::Square(RockElem* parent, CommonData* common, const Fluid* oil, const Fluid* water, 
               double radius, double initConAng, int connNum) : Polygon(parent, common, oil, 
               water, radius, 0.0625, initConAng, 4, connNum)
{
    m_commonData->addSquare();
    m_cornerHalfAngles.resize(m_numCorners, PI/4.0);
    m_conductanceOil = 0.0;
    m_conductanceWater.first = 0.0;
    m_conductanceWater.second = centerConductance(m_areaWater, m_water->viscosity());
}

Square::Square(RockElem* parent, CommonData* common, const Fluid* oil, const Fluid* water, const Square& shapeCp) : 
               Polygon(parent, common, oil, water, shapeCp)
{
}

///////////////////////////////////////////////////////////////////////////////////////////
// Circle Class Constructors
///////////////////////////////////////////////////////////////////////////////////////////
Circle::Circle(RockElem* parent, CommonData* common, const Fluid* oil, const Fluid* water, 
               double radius, double initConAng, int connNum) : Shape(parent, common, oil, 
               water, radius, 1.0/(4.0*PI), initConAng, connNum)
{
    m_commonData->addCircle();

    m_conductanceOil = 0.0;
    m_conductanceWater.first = 0.0;
    m_conductanceWater.second = centerConductance(m_areaWater, m_water->viscosity());
}

Circle::Circle(RockElem* parent, CommonData* common, const Fluid* oil, const Fluid* water, const Circle& shapeCp) : 
               Shape(parent, common, oil, water, shapeCp)
{
}

///////////////////////////////////////////////////////////////////////////////////////////
// Friend output functions
///////////////////////////////////////////////////////////////////////////////////////////
ostream& operator<< (ostream& out, Shape& shape) 
{
    out.flags(ios::showpoint);
    out.flags(ios::scientific);
    
    out << setprecision(4)
        << setw(15) << shape.m_radius*1.0E6
        << setw(15) << shape.m_area*1.0E12
        << setw(15) << shape.m_shapeFactor;

    out.flags(ios::fixed);

    out << setw(12) << shape.m_conAngleRec * 180.0 / shape.PI
        << setw(12) << shape.m_conAngleAdv * 180.0 / shape.PI;
    
    return out;  
}

void Triangle::setShapeFactor(double shapeFact)
{
    assure(shapeFact <= sqrt(3.0)/36.0, "2");
    m_shapeFactor = shapeFact;
    m_area = pow(m_radius, 2.0) / (4.0 * m_shapeFactor);
    assure(m_area > 0.0, "3");
    m_areaWater = m_area;
    getHalfAngles();
    m_conductanceWater.second = centerConductance(m_areaWater, m_water->viscosity());
}

///////////////////////////////////////////////////////////////////////////////////////////////////
// This function evaluates the three half angles that make up the triangular pore. The routine
// follows the outline that was described by Patzek. 
///////////////////////////////////////////////////////////////////////////////////////////////////
void Triangle::getHalfAngles()
{
    double beta_2_min = atan((2.0/sqrt(3.0))*cos(acos(-12.0*sqrt(3.0)*m_shapeFactor)/3.0+4.0*PI/3.0));        
    double beta_2_max = atan((2.0/sqrt(3.0))*cos(acos(-12.0*sqrt(3.0)*m_shapeFactor)/3.0));
    double randNum = static_cast<double>(rand()) / static_cast<double>(RAND_MAX);
    //double randNum = 0.5;     // delete me

    m_cornerHalfAngles[1] = beta_2_min + (beta_2_max - beta_2_min)*randNum;
    m_cornerHalfAngles[0] = -0.5*m_cornerHalfAngles[1] + 0.5*asin((tan(m_cornerHalfAngles[1])+4.0*m_shapeFactor) 
        * sin(m_cornerHalfAngles[1]) / (tan(m_cornerHalfAngles[1])-4.0*m_shapeFactor));
    m_cornerHalfAngles[2] = PI/2.0 - m_cornerHalfAngles[1] - m_cornerHalfAngles[0];    
}

/////////////////////////////////////////////////////////////////////////////////////////
// Contact angles are assigned based on the principle of equilibrium contacr angles
/////////////////////////////////////////////////////////////////////////////////////////
void Shape::setContactAngles(double equilConAng, int wettClass, double modelTwoSepAng)
{
    m_conAngEquil = equilConAng;
	if(wettClass == 1)
	{
		m_conAngleRec = m_conAngEquil;
		m_conAngleAdv = m_conAngEquil;
	}
	else if(wettClass == 2)
	{
        double growthExp((PI+modelTwoSepAng)/PI);
        
        m_conAngleRec = max(0.0, growthExp*m_conAngEquil-modelTwoSepAng);
        m_conAngleAdv = min(PI, growthExp*m_conAngEquil);
	}
	else
	{
        if(m_conAngEquil < 0.38349) m_conAngleRec = 0.0;
        else if(m_conAngEquil < 1.5289) m_conAngleRec = (0.5*exp(0.05*m_conAngEquil*180.0/PI)-1.5)*PI/180.0;
        else if(m_conAngEquil < 2.7646) m_conAngleRec = 2.0*(m_conAngEquil-1.19680);
        else m_conAngleRec = PI;

		if(m_conAngEquil < 0.38349) m_conAngleAdv = 0.0;
        else if(m_conAngEquil < 1.61268) m_conAngleAdv = 2.0*(m_conAngEquil-0.38349);
        else if(m_conAngEquil < 2.75805) m_conAngleAdv = (181.5 - 4051.0*exp(-0.05*m_conAngEquil*180.0/PI))*PI/180.0;
        else m_conAngleAdv = PI;
	}

    Polygon *polyShape = dynamic_cast< Polygon* >(this);
    if(polyShape)
    {
        for(int i = 0; i < polyShape->numCorners(); ++i)
        {
            polyShape->oilInCorner(i)->advConAng(m_conAngleAdv);
        }
    }
}
 
void Polygon::checkStateOfInitConAng()
{
    if(m_conAngleRec < m_conAngleInit)
    {
        m_conAngleInit = m_conAngleRec;
        for(int i = 0; i < m_numCorners; ++i)
        {
            m_waterInCorner[i]->setInitConAng(m_conAngleInit);
        }
    }
}

void Circle::checkStateOfInitConAng()
{
    if(m_conAngleRec < m_conAngleInit) m_conAngleInit = m_conAngleRec;
}

void Polygon::satReversal(double pc, bool didInjOil)
{
    for(int i = 0; i < m_numCorners; ++i)
    {
        m_waterInCorner[i]->pinnCorner(pc, m_conAngleRec, m_conAngleAdv, m_cornerHalfAngles[i], 
            m_water->interPhaseTen(), didInjOil);

        if(m_bulkFluid == m_water)
        {
            m_oilInCorner[i]->pinnLayer(pc, m_conAngleRec, m_conAngleAdv, m_cornerHalfAngles[i], 
                m_water->interPhaseTen(), didInjOil);
        }
    }
}

double Polygon::reformOilLayer(int corner)
{
    assure(!m_oilInCorner[corner]->stable(), "4");
    m_containsOil = true;
    ++m_numLayers;
    assure(m_numLayers > 0 && m_numLayers <= m_numCorners, "Wrong num Layers");
    m_oilInCorner[corner]->isInReformVec(false);
    m_oilInCorner[corner]->stable(true);
    m_anyOilLayers = true;
    if(m_numLayers == m_numCorners) m_allOilLayers = true;

    return m_oilInCorner[corner]->collapsePc();
}

void Polygon::updateFilmsForTrapping(double pc)
{
    
    const pair<int, double>& oilTrp = m_parent->trappingOil(); 
    const pair<int, double>& watTrpFilm = m_parent->trappingWatFilm(); 
    const pair<int, double>& watTrpBulk = m_parent->trappingWatBulk();

    for(int i = 0; i < m_numCorners; ++i)
    {
        if(oilTrp.first > -1)
        {
            m_waterInCorner[i]->updateTrapping(oilTrp, pc, m_conAngleRec, m_conAngleAdv, 
                m_cornerHalfAngles[i], m_water->interPhaseTen(), m_commonData->injectant() == m_oil);
            m_oilInCorner[i]->updateTrapping(oilTrp, pc, m_conAngleRec, m_conAngleAdv, m_cornerHalfAngles[i], 
                m_water->interPhaseTen(), m_commonData->injectant() == m_oil);
        }
        else
        {
            m_waterInCorner[i]->updateTrapping(watTrpFilm, pc, m_conAngleRec, m_conAngleAdv, 
                m_cornerHalfAngles[i], m_water->interPhaseTen(), m_commonData->injectant() == m_oil);
            m_oilInCorner[i]->updateTrapping(watTrpBulk, pc, m_conAngleRec, m_conAngleAdv, m_cornerHalfAngles[i], 
                m_water->interPhaseTen(), m_commonData->injectant() == m_oil);
        }
    }
}

double Polygon::collapseOilLayer(int corner)    // move me back to inline func... pleeeeeese...
{    
    assure(m_containsOil && m_anyOilLayers, "5");
    double collapsePc(m_oilInCorner[corner]->collapsePc());
    m_oilInCorner[corner]->isInCollapseVec(false);
    m_oilInCorner[corner]->stable(false);
    --m_numLayers;
    assure(m_numLayers >= 0 && m_numLayers < m_numCorners, "Layer counting problems");
    m_waterInCorner[corner]->pinnBeforeCollapse(collapsePc, m_conAngleRec, m_conAngleAdv, m_cornerHalfAngles[corner], m_water->interPhaseTen()); 
    m_oilInCorner[corner]->pinnBeforeCollapse(collapsePc, m_conAngleRec, m_conAngleAdv, m_cornerHalfAngles[corner], m_water->interPhaseTen()); 
    m_allOilLayers = false;
    
    if(!m_numLayers)
    {
        assure(!m_allOilLayers, "6");
        m_anyOilLayers = false;
        m_containsOil = false;
    }
    return collapsePc;
}


void Polygon::checkForUnstableWaterConfigs(SortedEvents< ThreeSome<Polygon*, int, double>, CoalesceWatFillCmp >& watEvents, 
                                           double globalPc)
{
    if(m_bulkFluid == m_oil && m_waterInCorner[0]->exists() && !m_waterInCorner[0]->trappedInside().first)  // Water snap off when the bulk is filled by
    {                                                                                                       // oil. Make sure there is no trapping in 
        double snapOffPrs = calcSnapOffPressureImb();                                                       // either phase
        if(snapOffPrs > globalPc)
        {
            ThreeSome<Polygon*, int, double> newEvent(this, -1, snapOffPrs+m_gravityCorrection);
            watEvents.quickInsert(newEvent);
        }
    }
    else if(m_bulkFluid == m_water && 
        m_oilInCorner[0]->exists() && 
        !(m_waterInCorner[0]->trappedInside().first && m_oilInCorner[0]->trappedOutside().first))          // Collapsing oil layers. Again make sure that 
    {                                                                                                       // trapping allows us to collapse it            
        for(size_t i = 0; i < m_oilInCorner.size(); ++i)
        {
            if(m_oilInCorner[i]->stable())
            {
                double collPc = m_oilInCorner[i]->collapsePc();
                if(collPc > globalPc)
                {
                    ThreeSome<Polygon*, int, double> newEvent(this, static_cast< int >(i), collPc+m_gravityCorrection);
                    watEvents.quickInsert(newEvent);
                }
            }
        }
    }
}

void Polygon::checkForUnstableOilConfigs(SortedEvents< ThreeSome<Polygon*, int, double>, CoalesceOilFillCmp >& oilEvents, 
                                         double globalPc)
{
    if(m_bulkFluid == m_water && m_oilInCorner[0]->stable() && !m_oilInCorner[0]->trappedOutside().first) 
    {
        assure(m_anyOilLayers, "7");
        double snapOffPrs = calcSnapOffPressureDrain();

        if(snapOffPrs < globalPc)
        {
            ThreeSome<Polygon*, int, double> newEvent(this, -1, m_snapOffPrs+m_gravityCorrection);
            oilEvents.quickInsert(newEvent);
        }
    }
}

/////////////////////////////////////////////////////////////////////////////////
// When the number of oil filled neighbours change the imbibition entry pressure
// will need to be updated. E.g. piston type displacement might now be possible
// or we move from an I3 to I2 pore body filling event. If water injection is
// forced, coopertive pore body filling will not occur. 
/////////////////////////////////////////////////////////////////////////////////
bool Polygon::updateImbEntryPrs(double& newEntryPrs) const
{
    int numOilNeighbours(m_parent->numOilNeighbours());
    double pistonEntryPrs(m_oil->interPhaseTen()/m_pistonTypeCurveRad);
    
    if(m_parent->iAmAPore() && m_conAngleAdv < PI/2.0 && numOilNeighbours != m_numNeighbours)
    {
        double radSum(0.0);
        int iEvent(0);
        string poreBodyFillAlg(m_commonData->poreFillAlg());
        for(int i = 0; i < m_parent->connectionNum(); ++i)
        {
            if(m_parent->connection(i)->shape()->bulkOil())
            {
                if(poreBodyFillAlg[0] == 'o' || poreBodyFillAlg[0] == 'O')
                {
                    radSum += m_commonData->radiiWeights(min(iEvent, 5))*
                        m_parent->connection(i)->shape()->radius()*
                        static_cast<double>(rand())/static_cast<double>(RAND_MAX);
                }
                else
                {
                    radSum += m_commonData->radiiWeights(min(iEvent, 5))*
                        static_cast<double>(rand())/static_cast<double>(RAND_MAX);
                }
                ++iEvent;
            }
        }
        assure(iEvent == numOilNeighbours, "Failed on poly imb I Events");
        if(poreBodyFillAlg == "blunt2")
            pistonEntryPrs = m_oil->interPhaseTen()*(2.0*cos(m_conAngleAdv)/m_radius - radSum);
        else if(poreBodyFillAlg == "blunt1" || poreBodyFillAlg == "oren1") 
            pistonEntryPrs = 2.0*m_oil->interPhaseTen()*cos(m_conAngleAdv)/(m_radius+radSum);
        else
            pistonEntryPrs = m_oil->interPhaseTen()*(1.0+2.0*sqrt(PI*m_shapeFactor))*cos(m_conAngleAdv)/(m_radius+radSum);
    }
        
    if(numOilNeighbours == m_numNeighbours)
    {
        newEntryPrs = m_snapOffPrs;
    }
    else
    {
        newEntryPrs = max(pistonEntryPrs, m_snapOffPrs);
    }
    return (newEntryPrs != m_entryPress);
}

bool Circle::updateImbEntryPrs(double& newEntryPrs) const 
{
    int numOilNeighbours(m_parent->numOilNeighbours());
    newEntryPrs = m_oil->interPhaseTen()/m_pistonTypeCurveRad;
    
    if(m_parent->iAmAPore() && m_conAngleAdv < PI/2.0 && numOilNeighbours != m_numNeighbours)
    {
        double radSum(0.0);
        int iEvent(0);
        string poreBodyFillAlg(m_commonData->poreFillAlg());
        for(int i = 0; i < m_parent->connectionNum(); ++i)
        {
            if(m_parent->connection(i)->shape()->bulkOil())
            {
                if(poreBodyFillAlg[0] == 'o' || poreBodyFillAlg[0] == 'O')
                {
                    radSum += m_commonData->radiiWeights(min(iEvent, 5))*
                        m_parent->connection(i)->shape()->radius()*
                        static_cast<double>(rand())/static_cast<double>(RAND_MAX);
                }
                else
                {
                    radSum += m_commonData->radiiWeights(min(iEvent, 5))*
                        static_cast<double>(rand())/static_cast<double>(RAND_MAX);
                }
                ++iEvent;
            }
        }
        assure(iEvent == numOilNeighbours, "Failed on circle imb I Events");
        if(poreBodyFillAlg == "blunt2")
            newEntryPrs = m_oil->interPhaseTen()*(2.0*cos(m_conAngleAdv)/m_radius - radSum);
        else
            newEntryPrs = 2.0*m_oil->interPhaseTen()*cos(m_conAngleAdv)/(m_radius+radSum);
    }

    return (newEntryPrs != m_entryPress);
}

bool Polygon::updateDrainEntryPrs(double& newEntryPrs) const 
{
    int numOilNeighbours(m_parent->numOilNeighbours());
    double conAng(m_virginState ? m_conAngleInit: m_conAngleRec);
    double pistonEntryPrs(m_oil->interPhaseTen()/m_pistonTypeCurveRad);

    if(m_parent->iAmAPore() && conAng > PI/2.0 && numOilNeighbours != 0)
    {
        double radSum(0.0);
        int iEvent(0);
        string poreBodyFillAlg(m_commonData->poreFillAlg());
        for(int i = 0; i < m_parent->connectionNum(); ++i)
        {
            if(!m_parent->connection(i)->shape()->bulkOil())
            {
                if(poreBodyFillAlg[0] == 'o' || poreBodyFillAlg[0] == 'O')
                {
                    radSum += m_commonData->radiiWeights(min(iEvent, 5))*
                        m_parent->connection(i)->shape()->radius()*
                        static_cast<double>(rand())/static_cast<double>(RAND_MAX);
                }
                else
                {
                    radSum += m_commonData->radiiWeights(min(iEvent, 5))*
                        static_cast<double>(rand())/static_cast<double>(RAND_MAX);
                }
                ++iEvent;
            }
        }
        assure(iEvent == m_numNeighbours-numOilNeighbours, "Failed on poly drain I Events");
        if(poreBodyFillAlg == "blunt2")
            pistonEntryPrs = m_oil->interPhaseTen()*(2.0*cos(conAng)/m_radius - radSum);
        else if(poreBodyFillAlg == "blunt1" || poreBodyFillAlg == "oren1") 
            pistonEntryPrs = 2.0*m_oil->interPhaseTen()*cos(conAng)/(m_radius+radSum);
        else
            pistonEntryPrs = m_oil->interPhaseTen()*(1.0+2.0*sqrt(PI*m_shapeFactor))*cos(conAng)/(m_radius+radSum);
    }
 
    if(m_anyOilLayers && numOilNeighbours == 0 && m_oilInCorner[0]->stableAtPrs(m_snapOffPrs))
    {
        newEntryPrs = m_snapOffPrs;
    }
    else if(m_anyOilLayers && m_oilInCorner[0]->stableAtPrs(m_snapOffPrs))
    {
        newEntryPrs = min(pistonEntryPrs, m_snapOffPrs);
    }
    else
    {
        newEntryPrs = pistonEntryPrs;
    }
    
    return (newEntryPrs != m_entryPress);
}

bool Circle::updateDrainEntryPrs(double& newEntryPrs) const 
{
    int numOilNeighbours(m_parent->numOilNeighbours());
    double conAng(m_virginState ? m_conAngleInit: m_conAngleRec);
    newEntryPrs = m_oil->interPhaseTen()/m_pistonTypeCurveRad;

    if(m_parent->iAmAPore() && conAng > PI/2.0 && numOilNeighbours != 0)
    {
        double radSum(0.0);
        int iEvent(0);
        string poreBodyFillAlg(m_commonData->poreFillAlg());
        for(int i = 0; i < m_parent->connectionNum(); ++i)
        {
            if(!m_parent->connection(i)->shape()->bulkOil())
            {
                if(poreBodyFillAlg[0] == 'o' || poreBodyFillAlg[0] == 'O')
                {
                    radSum += m_commonData->radiiWeights(min(iEvent, 5))*
                        m_parent->connection(i)->shape()->radius()*
                        static_cast<double>(rand())/static_cast<double>(RAND_MAX);
                }
                else
                {
                    radSum += m_commonData->radiiWeights(min(iEvent, 5))*
                        static_cast<double>(rand())/static_cast<double>(RAND_MAX);
                }
                ++iEvent;
            }
        }
        assure(iEvent == m_numNeighbours-numOilNeighbours, "Failed on circle drain I Events");
        if(poreBodyFillAlg == "blunt2")
            newEntryPrs = m_oil->interPhaseTen()*(2.0*cos(conAng)/m_radius - radSum);
        else
            newEntryPrs = 2.0*m_oil->interPhaseTen()*cos(conAng)/(m_radius+radSum);
    }
    
    return (newEntryPrs != m_entryPress);
}
        
/////////////////////////////////////////////////////////////////////////////////////////////////
// Before invoking the solver and calculating saturations we must update the state of the
// element, ie calculate saturation and conductance. Since the configuartion in imbibition can
// be more complex than in drainage, the calculations are sepaarte for each event
/////////////////////////////////////////////////////////////////////////////////////////////////
double Polygon::updateState(double pc)
{    
    if(m_bulkFluid == m_water && !m_anyOilLayers)
	{
        m_areaWater = m_area;    
        m_conductanceOil = 0.0;
        m_conductanceWater.first = 0.0;
        m_conductanceWater.second = centerConductance(m_area, m_water->viscosity());                                
	}
	else if(m_bulkFluid == m_oil && !m_waterInCorner[0]->exists())
	{
        m_areaWater = 0.0;    
        m_conductanceOil = centerConductance(m_area, m_oil->viscosity());
        m_conductanceWater.first = m_watCondMultFactWhenOilFilled * m_conductanceOil;
        m_conductanceWater.second = 0.0;
	}
	else if(m_bulkFluid == m_oil && m_waterInCorner[0]->exists())
    {
		oilWithWaterInCorners(pc);
    }
	else 
    {
		waterWithOilLayers(pc);
    }

    assure(m_conductanceOil >= 0.0 && m_conductanceWater.first >= 0.0 && m_conductanceWater.second >= 0.0, "8");
    return m_areaWater/m_area;
}

//////////////////////////////////////////////////
// Circular elements can only hold a single fluid.
//////////////////////////////////////////////////
double Circle::updateState(double pc)
{
    if(m_bulkFluid == m_water)                                                              
    {
        m_areaWater = m_area;    
        m_conductanceOil = 0.0;
        m_conductanceWater.first = 0.0;
        m_conductanceWater.second = centerConductance(m_area, m_water->viscosity());                                
    }
    else          
    {
        m_areaWater = 0.0;    
        m_conductanceOil = centerConductance(m_area, m_oil->viscosity());
        m_conductanceWater.first = m_watCondMultFactWhenOilFilled * m_conductanceOil;
        m_conductanceWater.second = 0.0;
	}
    
    return m_areaWater/m_area;
}

////////////////////////////////////////////////////////////////////////////////////////////////
// Water will remain in some of the corners of the polygon, Conductance and area is calculated
// for each corner.
////////////////////////////////////////////////////////////////////////////////////////////////
void Polygon::oilWithWaterInCorners(double cappPressure)
{
    double conAng(0.0), cornerArea(0.0), cornerCond(0.0);
    bool overRideTrapping(false);

	if(m_commonData->injectant() == m_water)			// Water flooding  
    {
		if(cappPressure < m_snapOffPrs) 
        {
            cappPressure = m_snapOffPrs+EPSILON;   
            if(m_waterInCorner[0]->trappedInside().first && m_waterInCorner[0]->trappedInside().second < m_snapOffPrs)
                overRideTrapping = true;
        }
        conAng = m_conAngleAdv;
    }
	else												// Oil flooding (secondary drainage etc..)
		conAng = m_conAngleRec;
		        
    for(int i = 0; i < m_numCorners; ++i)
    {
        if(m_waterInCorner[i]->exists())
        {                
            double conAngCurr(conAng), apexDist(0.0);
            m_waterInCorner[i]->currentState(conAngCurr, apexDist, cappPressure, m_cornerHalfAngles[i], 
                m_water->interPhaseTen(), m_conAngleAdv, overRideTrapping);
            
            double dlCornerArea = dimLessCornerArea(m_cornerHalfAngles[i], conAngCurr);
			double conductance = cornerConductance(dlCornerArea, apexDist, m_cornerHalfAngles[i], 
				conAngCurr, m_water->viscosity());
						
            assure(conductance > 0.0, "9");
            cornerCond += conductance;
			cornerArea += apexDist * apexDist * dlCornerArea;
        }
    }        
    m_areaWater = cornerArea;
    assure(m_areaWater < m_area && cornerCond > 0.0, "10");

    double oilSat((m_area-m_areaWater) / m_area); 
    m_conductanceOil = centerConductance(m_area, m_oil->viscosity()) * oilSat;         
    m_conductanceWater.first = cornerCond;
    m_conductanceWater.second = 0.0;
}

////////////////////////////////////////////////////////////////////////////////////////////////
// If oil is present at the same time that water occupies the center there must be 
// oil layer sandwisched between water in corners and center.
////////////////////////////////////////////////////////////////////////////////////////////////
void Polygon::waterWithOilLayers(double cappPressure)
{
    double conAng(0.0);
    double cornerAreaWat(0.0), cornerCondWat(0.0);
    double cornerAreaOil(0.0), cornerCondOil(0.0);
    double layerAreaOil(0.0), layerCondOil(0.0);
    bool overRideTrapping(false);

    if(m_commonData->injectant() == m_water)			// Water flooding  
        conAng = m_conAngleAdv;
	else												// Oil flooding (secondary drainage etc..)
    {
        if(cappPressure > m_snapOffPrs && !m_oilInCorner[0]->trappedOutside().first) 
            cappPressure = m_snapOffPrs-EPSILON;
        
        if(m_oilInCorner[0]->trappedOutside().first && m_oilInCorner[0]->trappedOutside().second > m_snapOffPrs)
        {
            overRideTrapping = true;
            cappPressure = m_snapOffPrs-EPSILON;
        }
        
        conAng = m_conAngleRec;
    }

    double baseLength(m_radius*(1.0/tan(m_cornerHalfAngles[0])+1.0/tan(m_cornerHalfAngles[1])));
    double oilApexBaseLengths(0.0);

    for(int i = 0; i < m_numCorners; ++i)
    {
        if(m_waterInCorner[i]->exists() && m_oilInCorner[i]->stable())               
        {                                                                                           
            double currPc(cappPressure);
            if(!m_oilInCorner[i]->stableAtPrs(currPc)) 
                currPc = m_oilInCorner[i]->collapsePc()+EPSILON;
                                   
            double outerConAng(conAng), outerApexDist(0.0);
            m_oilInCorner[i]->currentState(outerConAng, outerApexDist, currPc, m_cornerHalfAngles[i], 
                m_water->interPhaseTen(), m_conAngleAdv, overRideTrapping);

            double dlTotCornerArea = dimLessCornerArea(m_cornerHalfAngles[i], PI-outerConAng);          
            double areaCornerTot = outerApexDist * outerApexDist * dlTotCornerArea;

            if(i < 2) oilApexBaseLengths += outerApexDist;    // A permanaent check for not overstepping oil volumes
            assure(oilApexBaseLengths <= baseLength, "11");

            double innerConAng(conAng), innerApexDist(0.0);
            m_waterInCorner[i]->currentState(innerConAng, innerApexDist, currPc, m_cornerHalfAngles[i], 
                m_water->interPhaseTen(), m_conAngleAdv, overRideTrapping);
            double dlWatCornerArea = dimLessCornerArea(m_cornerHalfAngles[i], innerConAng);
            double areaWater = innerApexDist * innerApexDist * dlWatCornerArea;
            assure(outerApexDist > innerApexDist, "12");
                        
            double areaOil = areaCornerTot - areaWater;            
            double waterCond = cornerConductance(dlWatCornerArea, innerApexDist, m_cornerHalfAngles[i], 
                innerConAng, m_water->viscosity());
            double oilCond = layerConductance(1.0, 1.0, areaOil, areaCornerTot, outerApexDist, innerApexDist, 
                innerConAng, outerConAng, m_cornerHalfAngles[i], m_oil->viscosity());

            assure(areaWater*waterCond*areaOil*oilCond > 0.0, "Negative area or conductance in layer calcs");

            cornerAreaWat += areaWater;
            cornerCondWat += waterCond;
            layerAreaOil += areaOil;
            layerCondOil += oilCond;
        }
        else if(m_oilInCorner[i]->stable())
        {
            double currentConAng(conAng), apexDist(0.0);
            m_oilInCorner[i]->currentState(currentConAng, apexDist, cappPressure, m_cornerHalfAngles[i], 
                m_water->interPhaseTen(), m_conAngleAdv, overRideTrapping);            
            
            double dlCornerArea = dimLessCornerArea(m_cornerHalfAngles[i], PI-currentConAng);
            double conductance = cornerConductance(dlCornerArea, apexDist, m_cornerHalfAngles[i], 
                PI-currentConAng, m_oil->viscosity());
            
            cornerCondOil += conductance;
            cornerAreaOil += apexDist * apexDist * dlCornerArea;
        }
    }    
    
    assure(layerAreaOil > 0.0 || cornerAreaOil > 0.0, "14");
    m_areaWater = m_area - cornerAreaOil - layerAreaOil;   
    m_conductanceOil = layerCondOil + cornerCondOil;

    double watSat((m_area-cornerAreaOil-layerAreaOil-cornerAreaWat) / m_area);
    
    double centerWatCond = centerConductance(m_area, m_water->viscosity()) * watSat;    
    assure(watSat > 0.0 && watSat < 1.0, "15");

    if(m_allOilLayers)
    {
        assure(cornerCondWat > 0.0, "16");
        m_conductanceWater.first = cornerCondWat;
        m_conductanceWater.second = centerWatCond;
    }
    else
    {
        m_conductanceWater.first = 0.0;
        m_conductanceWater.second = centerWatCond+cornerCondWat;
    }
}

///////////////////////////////////////////////////////////////////////////
// Computed the conductance of a fluid occupying the center of the element
//
// There are some choices when it comes to compute the conducance of the 
// centre of an element. 
//
// g_p(A) = 3r^2A / 20mju   (1)     or      g_p(A) = 3A^2G / 5mju   (2)
//
// g = g_p(A_t) * S_p       (A)     or      g = g_p(A_p)            (B)
//
// Most combinations seem to give similar results (with the exception of
// 2B). Lets run with 2A....
//////////////////////////////////////////////////////////////////////////
double Square::centerConductance(double area, double visc) const 
{
    // return (m_radius * m_radius * area) / (7.1136 * visc);   // 1
    return (0.5623 * area * area * m_shapeFactor) / visc;       // 2
}

///////////////////////////////////////////////////////////////////////////
// Computed the conductance of a fluid occupying the center of the element
//////////////////////////////////////////////////////////////////////////
double Triangle::centerConductance(double area, double visc) const 
{
    // return (3.0 * m_radius * m_radius * area) / (20.0 * visc);   // 1
    return (3.0 * area * area * m_shapeFactor) / (5.0 * visc);      // 2
}

///////////////////////////////////////////////////////////////////////////
// Computed the conductance of a fluid occupying the center of the element
//////////////////////////////////////////////////////////////////////////
double Circle::centerConductance(double area, double visc) const 
{
    return (m_radius * m_radius * area) / (8.0 * visc);
}

/*double Polygon::layerConductance(double insideBC, double outsideBC, double layerArea, double cornerArea, double outerApexDist, 
                                 double innerApexDist, double innerConAng, double outerConAng, double halfAng, double viscosity) const
{
    double insideArea(cornerArea-layerArea);
    double phi((PI/2.0 - halfAng)*tan(halfAng));
    double numerator(pow(layerArea, 3.0) * pow(1.0-sin(halfAng), 2.0) * tan(halfAng) * phi * phi);
    double denominator(12.0 * viscosity * cornerArea * sin(halfAng) * sin(halfAng) * (1.0-phi) * 
        pow(1.0 + outsideBC*phi - (1.0-insideBC*phi) * sqrt(insideArea/cornerArea), 2.0));
    
    return numerator/denominator;
}*/


//////////////////////////////////////////////////////////////////////////////////////////////
// My very own layer conductance routine. Wohoo :-)
//////////////////////////////////////////////////////////////////////////////////////////////
double Polygon::layerConductance(double insideBC, double outsideBC, double layerArea, double cornerArea, double outerApexDist, 
                                 double innerApexDist, double innerConAng, double outerConAng, double halfAng, double viscosity) const
{
    double coeff[3];
    coeff[0] = -2.4010e-002;
    coeff[1] = 2.8402e-001;
    coeff[2] = -2.9531;

    if(halfAng < 10.0*PI/180.0)
    {
        coeff[0] = -1.0610E-002;   
        coeff[1] = 5.1608E-001;  
        coeff[2] = -2.0645;
    }
    else if(halfAng < 20.0*PI/180.0)
    {
        coeff[0] = -2.6810E-002;      
        coeff[1] = 1.8672E-001;  
        coeff[2] = -3.5977;
    }
    else if(halfAng < 30.0*PI/180.0)
    {
        coeff[0] = -4.4021E-002;     
        coeff[1] = -6.3195E-002;  
        coeff[2] = -4.3749;
    }
    else if(halfAng < 40.0*PI/180.0)
    {
        coeff[0] = -3.1523E-002;     
        coeff[1] = 1.6948E-001;  
        coeff[2] = -3.3600;
    }
    else if(halfAng < 50.0*PI/180.0)
    {
        coeff[0] = -3.1367E-002;     
        coeff[1] = 1.9327E-001;  
        coeff[2] = -3.2673;
    }
    else if(halfAng < 60.0*PI/180.0)
    {
        coeff[0] = -2.3201E-002;     
        coeff[1] = 3.1178E-001;  
        coeff[2] = -2.9251;
    }
    else
    {
        coeff[0] = -3.5760E-002;     
        coeff[1] = -6.5370E-004;  
        coeff[2] = -4.7019;
    }

    double dimlessAo(layerArea / (outerApexDist*outerApexDist));
    double dimlessBi(innerApexDist/outerApexDist);
    
    double outerOWlen = - (2.0*sin(halfAng)*(halfAng-outerConAng+PI/2.0))
        / cos(PI-outerConAng+halfAng);
    
    double innerOWLen = - (2.0*dimlessBi*sin(halfAng)*(halfAng+innerConAng-PI/2.0))
        / cos(innerConAng+halfAng);
    
    double perimeterLen = outerOWlen + innerOWLen + 2.0*(1.0-dimlessBi);
    double shapeFactorOil = dimlessAo / (perimeterLen*perimeterLen);
    double xGroup = log(pow(dimlessAo, 3.0)*shapeFactorOil);

    double dimlessOilCond = exp(coeff[0]*xGroup*xGroup + coeff[1]*xGroup + coeff[2]);
    double dimCond = pow(outerApexDist, 4.0) * dimlessOilCond / viscosity;
    return dimCond;
}

//////////////////////////////////////////////////////////////////////////
// The corner conductance version by Paal-Eric
//////////////////////////////////////////////////////////////////////////
double Polygon::cornerConductance(double dimLessCornerA, double meniscusApexDist, double halfAng, 
                                  double contactAngle, double viscosity) const
{
    double cornerGstar((cos(halfAng)*sin(halfAng)) / (4.0*pow(1+sin(halfAng), 2.0)));
    double cornerG(cornerGstar);

    if(fabs(contactAngle + halfAng - PI/2.0) > 0.01)
    {
        cornerG = dimLessCornerA / 
        (4.0 * pow(1.0 - (sin(halfAng)/cos(halfAng+contactAngle)) * (halfAng+contactAngle-PI/2.0), 2.0));
    }


    double cFactor = 0.364 + 0.28 * cornerGstar / cornerG;
    double dimLessCornerConduct = cFactor * dimLessCornerA * dimLessCornerA * cornerG;

    assure(dimLessCornerA != 0.0 && halfAng !=0.0 && cornerG != 0.0, "17");
    
    return dimLessCornerConduct * pow(meniscusApexDist, 4.0) / viscosity;
}
/*
double Polygon::cornerConductance(double dimLessCornerA, double meniscusApexDist, double halfAng, 
                                  double contactAngle, double viscosity) const
{
    double cornerG;
    if(contactAngle + halfAng == PI/2.0)              
        cornerG = dimLessCornerA / (4.0*pow(1+sin(halfAng), 2.0));
    else
        cornerG = dimLessCornerA / 
            (4.0 * pow(1.0-(sin(halfAng)/cos(halfAng+contactAngle))*(halfAng + contactAngle - PI/2.0), 2.0));
    
    double a1, a2, a3;                  // From Patzek: Initially find a scaled conductance

    if(halfAng <= PI/18.0)
    {
        a1 = 224.885;
        a2 = -6.80214;
        a3 = -0.205123;
    }
    else if(halfAng <= PI/12.0)
    {
        a1 = 68.356;
        a2 = 0.15;
        a3 = -0.264133;
    }
    else if(halfAng <= PI/6.0)
    {
        a1 = -1.52067;
        a2 = 4.13661;
        a3 = -0.304756;
    }
    else if(halfAng <= PI/5.0)
    {
        a1 = -6.07127;
        a2 = 4.43528;
        a3 = -0.307906;
    }
    else if(halfAng <= PI/4.0)
    {
        a1 = -10.7395;
        a2 = 4.79784;
        a3 = -0.314243;
    }
    else if(halfAng <= PI/3.0)
    {
        a1 = -19.9801;
        a2 = 5.69843;
        a3 = -0.336266;
    }
    else 
    {
        a1 = -31.5714;
        a2 = 6.82525;
        a3 = -0.363203;
    }

    double scaledDLCond = a1 * cornerG * cornerG + a2 * cornerG + a3;   
    
    double e1(7.0/8.0), e2(0.5);                                        // No slip case
    double dimLessCornerConduct = exp((scaledDLCond + 0.02*sin(halfAng - PI/6.0)) 
        / (pow(1.0/(4.0*PI) - cornerG, e1)*pow(cos(halfAng - PI/6.0), e2))) * dimLessCornerA * dimLessCornerA;
      
    return dimLessCornerConduct * pow(meniscusApexDist, 4.0) / viscosity;
}
*/
//////////////////////////////////////////////////////////////////////////////////////////
// Calculate the various entry pressures for drainage, This function is used for all
// displacement cycles
//////////////////////////////////////////////////////////////////////////////////////////
void Polygon::initDrainage(double cappPrs)
{
    assure(m_numNeighbours > 0, "18");
    m_parent->resetFillingEvent();
    m_parent->isInWatFloodVec(false);
    m_parent->isInOilFloodVec(false);
    for(int j = 0; j < m_numCorners; ++j) 
    {
        m_oilInCorner[j]->isInCollapseVec(false);
        m_oilInCorner[j]->isInReformVec(false);
    }

    satReversal(cappPrs, false);

    if(m_bulkFluid == m_oil) return;

    double conAng(m_virginState ? m_conAngleInit: m_conAngleRec);

    double minLocalPcLastCycle(m_commonData->minDatumPcLastCycle()-m_gravityCorrection);
    double minLocalPc(m_commonData->minDatumPc()-m_gravityCorrection);    
    double minPc = m_oilInCorner[0]->pinnedInLastCycle(minLocalPcLastCycle) ? minLocalPcLastCycle: minLocalPc;
    double angSum(0.0), normThresPress((m_radius*minPc)/m_oil->interPhaseTen());   
    for(int i = 0; i < m_numCorners; ++i)
    {
        if(m_oilInCorner[i]->stable()) angSum += cos(m_conAngleAdv - m_cornerHalfAngles[i]);
    }
       
    double rhsMaxRecConAng = max((4.0*m_shapeFactor*angSum)                                 // Prevent falling out of range [1, -1]. 
        / (normThresPress-cos(m_conAngleAdv)-12.0*m_shapeFactor*sin(m_conAngleAdv)), -1.0); // This is only applicable when r is very small
    rhsMaxRecConAng = min(rhsMaxRecConAng, 1.0);                                            // and these elems are probably not drained.                                                                
    m_maxConAngSpont = acos(rhsMaxRecConAng);
    
    if(m_oilInCorner[0]->exists()) m_snapOffPrs = calcSnapOffPressureDrain();

    if(conAng > m_maxConAngSpont && m_oilInCorner[0]->exists())
        m_pistonTypeCurveRad = mspCurveRadHingDrain();
    else if(conAng >= PI/2.0 - m_cornerHalfAngles[0])         
        m_pistonTypeCurveRad = m_radius / (2.0*cos(conAng));              
    else     
        m_pistonTypeCurveRad = mspCurveRadDrain(conAng);

    updateDrainEntryPrs(m_entryPress);
}

void Circle::initDrainage(double cappPrs)
{
    assure(m_numNeighbours > 0, "20");
    m_parent->resetFillingEvent();
    m_parent->isInWatFloodVec(false);
    m_parent->isInOilFloodVec(false);

    if(m_bulkFluid == m_oil) return;
      
    double conAng(m_virginState ? m_conAngleInit: m_conAngleRec);
    
    m_maxConAngSpont = PI/2.0;
    m_pistonTypeCurveRad = m_radius / (2.0*cos(conAng));
        
    updateDrainEntryPrs(m_entryPress);
}

//////////////////////////////////////////////////////////////////////////////////////////
// Calculate the various entry pressures for imbibition, This function is used for all
// displacement cycles
//////////////////////////////////////////////////////////////////////////////////////////
void Polygon::initImbibition(double cappPress)
{
    assure(m_numNeighbours > 0, "21");
    m_parent->resetFillingEvent();
    m_parent->isInWatFloodVec(false);
    m_parent->isInOilFloodVec(false);
    for(int j = 0; j < m_numCorners; ++j) 
    {
        m_oilInCorner[j]->isInCollapseVec(false);
        m_oilInCorner[j]->isInReformVec(false);
    }

    satReversal(cappPress, true);

    if(m_bulkFluid == m_water) return;

    m_virginState = false;

    double recConAng(m_waterInCorner[0]->hingingConAng(cappPress, m_conAngleAdv,
        m_cornerHalfAngles[0], m_oil->interPhaseTen()));
    
    double maxLocalPcLastCycle(m_commonData->maxDatumPcLastCycle()-m_gravityCorrection);
    double maxLocalPc(m_commonData->maxDatumPc()-m_gravityCorrection);    
    double maxPc = m_waterInCorner[0]->pinnedInInitState() ? maxLocalPc: maxLocalPcLastCycle;
    double angSum(0.0), normThresPress((m_radius*maxPc)/m_oil->interPhaseTen()), curvRad(0.0);   
    for(int i = 0; i < m_numCorners; ++i)
    {
        if(m_waterInCorner[i]->exists())
            angSum += cos(recConAng + m_cornerHalfAngles[i]);
    }
       
    double rhsMaxAdvConAng = max((-4.0*m_shapeFactor*angSum)                         // Prevent falling out of range [1, -1]. 
        / (normThresPress-cos(recConAng)+12.0*m_shapeFactor*sin(recConAng)), -1.0);  // This is only applicable when r is very small
    rhsMaxAdvConAng = min(rhsMaxAdvConAng, 1.0);                                     // and these elems are probably not drained.                                                                
    m_maxConAngSpont = acos(rhsMaxAdvConAng);
                                                                            
    if(m_waterInCorner[0]->exists()) m_snapOffPrs = calcSnapOffPressureImb();
    
    if(m_conAngleAdv < m_maxConAngSpont)
        m_pistonTypeCurveRad = mspCurveRadHingImb();
    else if(m_conAngleAdv <= PI/2.0 + m_cornerHalfAngles[0])            // If oil layers are not possible in all 
        m_pistonTypeCurveRad = m_radius / (2.0*cos(m_conAngleAdv));     // corners, we do not use MSP procedure
    else                                                                // => do the check with max(beta)
        m_pistonTypeCurveRad = mspCurveRadImb();
        
    updateImbEntryPrs(m_entryPress);
}

void Circle::initImbibition(double cappPress)
{
    assure(m_numNeighbours > 0, "23");
    m_parent->resetFillingEvent();
    m_parent->isInWatFloodVec(false);
    m_parent->isInOilFloodVec(false);

    if(m_bulkFluid == m_water) return;
  
    m_maxConAngSpont = PI/2.0;
    m_pistonTypeCurveRad = m_radius / (2.0*cos(m_conAngleAdv)); 
    
    updateImbEntryPrs(m_entryPress);
}

/////////////////////////////////////////////////////////////////////////////////////////////////////
// The threshold pressures are based on the Mayer-Stowe-Princen method for calculating entry pressure.
// The contact angle used in the calculations is PI-contactAngle when the contactangle is > PI/2. 
// Initally we assume that oil layers are stable in all coners. For the then given entry pressure we
// make sure that the collapsing pressure in the most oblique has not been reached. If it has we 
// compare that pressure to the entry pressure for two layers. If the entry pressure for oil in two 
// coners 
/////////////////////////////////////////////////////////////////////////////////////////////////////
double Polygon::mspCurveRadImb() const
{
    double contactAngle = PI - m_conAngleAdv;
    vector< double > potentialCurveRad;
    
    double sOne(0.0), sTwo(0.0), sThree(0.0);
    for(int i = 0; i < m_numCorners; ++i)
    {
        if(m_conAngleAdv > PI/2.0 + m_cornerHalfAngles[i])  // Only cases where oil layers/oil in corner might exist
        {
            sOne += cos(contactAngle)*cos(contactAngle+m_cornerHalfAngles[i])/sin(m_cornerHalfAngles[i]) 
                - (PI/2.0-contactAngle-m_cornerHalfAngles[i]);
            sTwo += cos(contactAngle+m_cornerHalfAngles[i])/sin(m_cornerHalfAngles[i]);
            sThree += 2.0 * (PI/2.0 - contactAngle - m_cornerHalfAngles[i]);

            double dFact = sOne - 2.0*sTwo*cos(contactAngle) + sThree;
            double rootFact = 1.0 + 4.0*m_shapeFactor*dFact/(cos(contactAngle)*cos(contactAngle));
            
            double radOne = m_radius*cos(contactAngle)*(-1+sqrt(rootFact))/(4.0*m_shapeFactor*dFact);
            double radTwo = m_radius*cos(contactAngle)*(-1-sqrt(rootFact))/(4.0*m_shapeFactor*dFact);            
            potentialCurveRad.push_back(-min(radOne, radTwo));
        }
    }

    for(int j = static_cast< int >(potentialCurveRad.size())-1; j >= 0; --j)
    {
        double tension(m_oil->interPhaseTen());
        double pc(tension / potentialCurveRad[j]);
        double collapsePc(INF_NEG_NUM);
        if(m_waterInCorner[j]->exists())
            collapsePc = m_oilInCorner[j]->collapsePc_noTrap(pc, m_conAngleAdv, m_cornerHalfAngles[j], tension);

        if(pc > collapsePc) return potentialCurveRad[j];
    }

    return m_radius/(2.0*cos(m_conAngleAdv));
}

double Polygon::mspCurveRadDrain(double conAng) const
{
    double cFactor(0.0);

    for(int i = 0; i < m_numCorners; ++i)
    {
        if(conAng < PI/2.0 - m_cornerHalfAngles[i]) 
        {
            cFactor += cos(conAng) * cos(conAng+m_cornerHalfAngles[i])/sin(m_cornerHalfAngles[i]) 
                - (PI/2.0 - conAng - m_cornerHalfAngles[i]);
        }
    }

    double funcF = (1.0 + sqrt(1.0 - 4.0*m_shapeFactor*cFactor/(cos(conAng)*cos(conAng))))
        / (1.0 + 2.0*sqrt(PI * m_shapeFactor));

    return m_radius / ((1.0 + 2.0*sqrt(PI*m_shapeFactor)) * cos(conAng) * funcF);
}

/////////////////////////////////////////////////////////////////////////////////////////////////////////
// Piston type displacement (also referred to as event 1 displacement) for advancing contact angles less
// than the critical is obtained by solving a set of non-linear equations using the Newton-Raphson tech.
// The hinging contact angle will vary between the receding and advancing, fixed at a position bi. This 
// procedure is identical to that used by Patzek and Oren.
////////////////////////////////////////////////////////////////////////////////////////////////////////
double Polygon::mspCurveRadHingImb() const 
{    
    const double curvRadDrain(m_oil->interPhaseTen()/(m_commonData->maxDatumPcLastCycle()-m_gravityCorrection)); 
    double oldRad(curvRadDrain), tension(m_oil->interPhaseTen()), err;

    if(m_waterInCorner[0]->trappedInside().first)
        oldRad = tension/m_waterInCorner[0]->trappedInside().second;

    int itr;
    double oldFirstGuess(oldRad);

    for(itr = 0; itr < MAX_NEWT_ITR; ++itr)
    {            
        double sumOne(0.0), sumTwo(0.0), sumThree(0.0), sumFour(0.0);
        double hingConAng, meniscusApexDist, alpha;

        for(int j = 0; j < m_numCorners; ++j)
        {                                                
            hingConAng = m_waterInCorner[j]->hingingConAng(tension/oldRad, m_conAngleAdv, m_cornerHalfAngles[j], 
                tension, true);
            meniscusApexDist = m_waterInCorner[j]->apexDistance(tension/oldRad, m_conAngleAdv, m_cornerHalfAngles[j], 
                tension);
            double partus(meniscusApexDist * sin(m_cornerHalfAngles[j]) / oldRad);
            assure(partus >= -1.0 && partus <= 1.0, "24");
            alpha = asin(partus);

            if(m_waterInCorner[j]->exists())
            {
                sumOne += meniscusApexDist*cos(hingConAng);
                sumTwo += PI/2.0-hingConAng-m_cornerHalfAngles[j];
                sumThree += alpha;
                sumFour += meniscusApexDist;
            }
        }

        double newRad = (m_radius*m_radius/(4.0*m_shapeFactor) - oldRad*sumOne + oldRad*oldRad*sumTwo)
            / (2.0*oldRad*sumThree + cos(m_conAngleAdv)*(m_radius/(2.0*m_shapeFactor) - 2.0*sumFour));
                
        err = fabs((newRad - oldRad)/oldRad);        
        if(err < EPSILON) return newRad;

        oldRad = newRad;
    }

    if(err < 0.1) 
    {
        cout << endl << "Problem in Polygon::mspCurveRadHingImb" << endl;
        return oldRad;
    }
    //if(mspCurveRadHingImbNR(oldRad, err)) return oldRad;

    cerr << endl
        << "=================================================" << endl
        << "Error: Failed to obtain valid value for threshold" << endl
        << "radius of curvature in piston type displacement  " << endl
        << "during imbibition."                                << endl
        << "Trapped water: " << m_waterInCorner[0]->trappedInside().first << endl
        << "Err  " << err << endl
        << "Con ang " << m_conAngleAdv*180.0/PI << endl
        << "Radius " << m_radius << endl
        << "G " << m_shapeFactor << endl
        << "Iteration " << itr << endl
        << "=================================================" << endl;
    exit(-1);

    return 0.0;
}

bool Polygon::mspCurveRadHingImbNR(double& firstGuess, double& err) const
{
    const double curvRadDrain = firstGuess; 
    double oldRad(curvRadDrain), tension(m_oil->interPhaseTen());                     

    for(int i = 0; i < MAX_NEWT_ITR; ++i)
    {
        double sumAeff(0.0), sumLeff(0.0), sumAeffDr(0.0), sumLeffDr(0.0);

        for(int j = 0; j < m_numCorners; ++j)
        {                        
            double hingContactAng, hingContactAngDr, meniscusApexDist, meniscusApexDistDr, alpha, alphaDr;
                        
            hingContactAng = m_waterInCorner[j]->hingingConAng(tension/oldRad, m_conAngleAdv, m_cornerHalfAngles[j], 
                tension, true);
            hingContactAngDr = (curvRadDrain/(oldRad*oldRad)) * 
                (cos(m_conAngleRec+m_cornerHalfAngles[j]) / sin(hingContactAng+m_cornerHalfAngles[j]));

            meniscusApexDist = m_waterInCorner[j]->apexDistance(tension/oldRad, m_conAngleAdv, m_cornerHalfAngles[j], 
                tension);
            meniscusApexDistDr = 0.0;

            double partus(meniscusApexDist * sin(m_cornerHalfAngles[j]) / oldRad);
            assure(partus >= -1.0 && partus <= 1.0, "24");
            alpha = asin(partus);
            alphaDr = -(meniscusApexDist/(oldRad*oldRad)) * (sin(m_cornerHalfAngles[j])/cos(alpha));
            
            if(hingContactAng >= m_conAngleAdv)
            {
                hingContactAngDr = 0.0;               
                meniscusApexDistDr = cos(m_conAngleAdv+m_cornerHalfAngles[j]) / sin(m_cornerHalfAngles[j]);               
                alpha = PI/2.0 - m_conAngleAdv - m_cornerHalfAngles[j];
                alphaDr = 0.0;
            }

            if(m_waterInCorner[j]->exists())
            {
                sumAeff += meniscusApexDist*cos(hingContactAng)/oldRad + hingContactAng + m_cornerHalfAngles[j] - PI/2.0;
                sumAeffDr += meniscusApexDist*cos(hingContactAng)/oldRad + meniscusApexDistDr*cos(hingContactAng) -
                    meniscusApexDist*hingContactAngDr*sin(hingContactAng) - 
                    2.0*(PI/2.0-hingContactAng-m_cornerHalfAngles[j]) + oldRad*hingContactAngDr;

                sumLeff += oldRad*alpha - meniscusApexDist*cos(m_conAngleAdv);
                sumLeffDr += alpha + oldRad*alphaDr - cos(m_conAngleAdv)*meniscusApexDistDr;
            }
        }

        double areaEff = (m_radius*m_radius)/(4.0*m_shapeFactor) - oldRad * oldRad * sumAeff;
        double areaEffDr = -oldRad * sumAeffDr;
        double lengthEff = m_radius*cos(m_conAngleAdv)/(2.0*m_shapeFactor) + 2.0*sumLeff;
        double lengthEffDr = 2.0 * sumLeffDr;

        double func = oldRad - areaEff/lengthEff;
        double funcDr = 1 - (areaEffDr*lengthEff - areaEff*lengthEffDr)/(lengthEff*lengthEff);

        double newRad = areaEff/lengthEff;
        err = fabs((newRad - oldRad)/oldRad);

        if(err < EPSILON) 
        {
            firstGuess = newRad;
            return true;
        }

        oldRad = newRad;
    }

    return false;
}


double Polygon::mspCurveRadHingDrain() const 
{    
    double tension(m_oil->interPhaseTen());
    double minLocalPcLastCycle(m_commonData->minDatumPcLastCycle()-m_gravityCorrection);
    double minLocalPc(m_commonData->minDatumPc()-m_gravityCorrection);
    
    double minPc = m_oilInCorner[0]->pinnedInLastCycle(minLocalPcLastCycle) ? minLocalPcLastCycle: minLocalPc;
    const double curvRadImb(tension/minPc); 
    double oldRad(curvRadImb), err(1.0);
    int itr(0);

    if(m_oilInCorner[0]->trappedOutside().first)
        oldRad = tension/m_oilInCorner[0]->trappedOutside().second;

    int numCorn(m_numCorners);
    while(numCorn >= 0)
    {
        for(itr = 0; itr < MAX_NEWT_ITR; ++itr)
        {            
            double sumOne(0.0), sumTwo(0.0), sumThree(0.0), sumFour(0.0);
            double hingConAng, meniscusApexDist, alpha;
            
            for(int j = 0; j < numCorn; ++j)
            {
                if(m_oilInCorner[j]->stableAtPrs(tension/oldRad) && !m_oilInCorner[j]->forcedSnapOff(tension/oldRad))
                {
                    hingConAng = m_oilInCorner[j]->hingingConAng(tension/oldRad, m_conAngleRec, m_cornerHalfAngles[j], 
                        tension, true);
                    meniscusApexDist = m_oilInCorner[j]->apexDistance(tension/oldRad, m_conAngleRec, m_cornerHalfAngles[j], 
                        tension, true);
                    double partus(-meniscusApexDist*sin(m_cornerHalfAngles[j])/oldRad);
                    assure(partus >= -1.0 && partus <= 1.0, "25");
                    alpha = asin(partus);
                    sumOne += meniscusApexDist*cos(hingConAng);
                    sumTwo += hingConAng-m_cornerHalfAngles[j]-PI/2.0;
                    sumThree += alpha;
                    sumFour += meniscusApexDist;
                }
            }
            
            double newRad = (m_radius*m_radius/(4.0*m_shapeFactor) - oldRad*sumOne + oldRad*oldRad*sumTwo)
                / (2.0*oldRad*sumThree + cos(m_conAngleRec)*(m_radius/(2.0*m_shapeFactor) - 2.0*sumFour));
            
            err = fabs((newRad - oldRad)/oldRad);
            
            if(err < EPSILON) return newRad;
            oldRad = newRad;
        }
        --numCorn;
    }

    cerr << endl
        << "=================================================" << endl
        << "Error: Failed to obtain valid value for threshold" << endl
        << "radius of curvature in piston type displacement  " << endl
        << "during drainage."                                  << endl
        << "Err  " << err << endl
        << "Con ang " << m_conAngleRec*180.0/PI << endl
        << "Radius " << m_radius << endl
        << "G " << m_shapeFactor << endl
        << "Iteration " << itr << endl
        << "=================================================" << endl;
    exit(-1);

    return 0.0;
}

///////////////////////////////////////////////////////////////////////////////////
// Spontaneous snap off (Contact ang < PI/2 - beta_min): 
// Snap off ocuurs when the base of the menisci in the two sharpest corners meet. 
// 
// Forced snap off:
// Snap off occurs when the contact angle in the sharpest corner reach advancing
// contact angle. 
///////////////////////////////////////////////////////////////////////////////////
double Polygon::calcSnapOffPressureImb() const
{
    double snapOffPrs;
    if(m_conAngleAdv < PI/2.0 - m_cornerHalfAngles[0])              // Spontaneous 
        snapOffPrs = snapOffPrsImbNR();
    else
    {                                                               // Forced
        snapOffPrs = m_waterInCorner[0]->minHingingPc();
    }
    
    assure(snapOffPrs != 0.0, "26");
    return snapOffPrs;
}

double Polygon::calcSnapOffPressureDrain() const
{
    double snapOffPrs;
    if(m_conAngleRec > PI/2.0 + m_cornerHalfAngles[0])      // Spontaneous 
    {
        snapOffPrs = snapOffPrsDrainNR();
    }
    else
    {                                                       // Forced
        snapOffPrs = m_oilInCorner[0]->maxHingingPc();
        assure(snapOffPrs > 0.0, "27");
    }

    assure(snapOffPrs != 0.0, "28");
    return snapOffPrs;
}

///////////////////////////////////////////////////////////////////////////////////
// Spontaneous snap off can occur in two ways. If advancing contact angle is small
// all the menisci will move and snap off occurs when the menicii in the two 
// sharpest corners meet. For larger contact angles one or two of the meniscii may
// remain pinned. Snap off occurs when the sharpest coner meets any of the two 
// others. The event actually occuring is the one at the highest pressure.
///////////////////////////////////////////////////////////////////////////////////
double Triangle::snapOffPrsImbNR() const
{
    double tension(m_oil->interPhaseTen());
    double snapOffPrsOne = (tension/m_radius) * (cos(m_conAngleAdv) - 
            (2.0*sin(m_conAngleAdv))/(1.0/tan(m_cornerHalfAngles[0]) + 1.0/tan(m_cornerHalfAngles[1])));    // Small Contact Ang
    
    double oldPc, errorVal(1.0);
    if(m_waterInCorner[0]->trappedInside().first)
        oldPc = m_waterInCorner[0]->trappedInside().second;
    else if(m_commonData->drainagePhase())
        oldPc = m_commonData->datumCappPress()-m_gravityCorrection;
    else
        oldPc = m_commonData->maxDatumPcLastCycle()-m_gravityCorrection;

    for(int i = 0; i < MAX_NEWT_ITR; ++i)               // Larger contact angles => Need to use a NR technique
    {       
        double hingAng = m_waterInCorner[2]->hingingConAng(oldPc, m_conAngleAdv, m_cornerHalfAngles[2], tension, true);
        double pinnedApexDistance = m_waterInCorner[2]->pinnedApexDist();        
        double hingAngDpc = -pinnedApexDistance*sin(m_cornerHalfAngles[2])/(tension*sin(hingAng+m_cornerHalfAngles[2]));
        
        double func = oldPc - (m_oil->interPhaseTen()/m_radius)*((cos(m_conAngleAdv)/tan(m_cornerHalfAngles[0]) -
            sin(m_conAngleAdv) + cos(hingAng)/tan(m_cornerHalfAngles[2]) - sin(hingAng)) /
            (1.0/tan(m_cornerHalfAngles[0]) + 1.0/tan(m_cornerHalfAngles[2])));
        
        double funcDpc = 1 + (m_oil->interPhaseTen()/m_radius)*((hingAngDpc*sin(hingAng)/tan(m_cornerHalfAngles[2]) + 
            hingAngDpc*cos(hingAng)) / (1.0/tan(m_cornerHalfAngles[0]) + 1.0/tan(m_cornerHalfAngles[2])));

        double newPc = oldPc - func / funcDpc;

        errorVal = fabs((newPc-oldPc)/oldPc);
        if(errorVal < EPSILON)
        {
            if(hingAng < m_conAngleAdv)
                return max(newPc, snapOffPrsOne);     // Return the pressure occuring at the highest Pc
            else
                return snapOffPrsOne;
        }

        oldPc = newPc;
    }


    cerr <<"=================================================" << endl
        << "Error: Failed to obtain valid value for threshold" << endl
        << "capillary pressure during snap off.              " << endl
        << "Error: " << errorVal                               << endl
        << "=================================================" << endl;
    exit(-1);

    return 0.0;
}

double Square::snapOffPrsImbNR() const
{
        return (m_oil->interPhaseTen()/m_radius) * (cos(m_conAngleAdv) - sin(m_conAngleAdv));
}

/////////////////////////////////////////////////////////////////////////////////
// The problem with these NR approaches are that they're a bit sensitive
// to the first guess. 
/////////////////////////////////////////////////////////////////////////////////
double Triangle::oilApexMeetCorner(bool& casePossible, int cor) const
{
    if(!m_oilInCorner[cor]->exists()) return 0.0;
    double minLocalPcLastCycle(m_commonData->minDatumPcLastCycle()-m_gravityCorrection);
    double minLocalPc(m_commonData->minDatumPc()-m_gravityCorrection);
    
    double minPc = m_oilInCorner[0]->pinnedInLastCycle(minLocalPcLastCycle) ? minLocalPcLastCycle: minLocalPc;   
    double tension(m_oil->interPhaseTen()), errorVal(1.0);
    vector< double > initGuesses(3);

    initGuesses[2] = minPc;                                                 // First attempt
    initGuesses[1] = m_commonData->datumCappPress()-m_gravityCorrection;    // Second attempt
    if(m_oilInCorner[cor]->trappedOutside().first)                          // Getting desparete now       
        initGuesses[0] = m_oilInCorner[cor]->trappedOutside().second;
    else
        initGuesses[0] = m_parent->trappingOil().second;      

    while(!initGuesses.empty())
    {
        double oldPc = initGuesses.back();
        initGuesses.pop_back();
        
        for(int i = 0; i < MAX_NEWT_ITR; ++i)               // Larger contact angles => Need to use a NR technique
        {       
            double pinnApexThree = m_oilInCorner[cor]->pinnedApexDist();
            double part(pinnApexThree*oldPc*sin(m_cornerHalfAngles[cor])/tension);
            part = min(max(-1.0, part), 1.0);
            double hingAng = acos(part)+m_cornerHalfAngles[cor];
            double hingAngDpc = -pinnApexThree*sin(m_cornerHalfAngles[cor])/(tension*sin(hingAng-m_cornerHalfAngles[cor]));

            double func = oldPc - (tension/m_radius)*((cos(m_conAngleRec)/tan(m_cornerHalfAngles[0]) +
                sin(m_conAngleRec) + cos(hingAng)/tan(m_cornerHalfAngles[2]) + sin(hingAng)) /
                (1.0/tan(m_cornerHalfAngles[0]) + 1.0/tan(m_cornerHalfAngles[cor])));

            double funcDpc = 1 - (tension/m_radius)*((hingAngDpc*sin(hingAng)/tan(m_cornerHalfAngles[cor]) - 
                hingAngDpc*cos(hingAng)) / (1.0/tan(m_cornerHalfAngles[0]) + 1.0/tan(m_cornerHalfAngles[cor])));

            double newPc = oldPc - func / funcDpc;
            errorVal = fabs((newPc - oldPc)/oldPc);

            if(errorVal < EPSILON)
            {
                casePossible = hingAng >= m_conAngleRec && m_oilInCorner[cor]->stableAtPrs(newPc);
                return newPc;
            }

            oldPc = newPc;
        }
    }
    
    cerr <<"=================================================" << endl
        << "Error: Failed to obtain valid value for threshold" << endl
        << "capillary pressure during snap off.              " << endl
        << "Error: " << errorVal                               << endl
        << "=================================================" << endl;
    exit(-1);
    
    return 0.0;   
}

/////////////////////////////////////////////////////////////////////////////
// 4 Different scenarios exists for oil snap off:
//
//  Case 1: There is only oil layer in sharpest corner. Snap off occurs when 
//          interface meets opposing wall.
//  Case 2: Interfaces in the two sharpest corners slip. Snap off occurs when
//          they meet at the base.
//  Case 3: Interfaces exist in all corners. Only the sharpest corner slip. 
//          Snap off occurs when it meets the pinned interface in the most
//          oblique corner.
//  Case 4: Interfaces exist only in the two sharpest corners. Only the 
//          sharpest corner slip. Snap off occurs when it meets the pinned
//          interface. 
/////////////////////////////////////////////////////////////////////////////
double Triangle::snapOffPrsDrainNR() const
{
    double tension(m_oil->interPhaseTen());
    double caseOneSnapOff(tension*cos(m_conAngleRec-m_cornerHalfAngles[0])/(sin(m_cornerHalfAngles[0])*
        m_radius*(1.0/tan(m_cornerHalfAngles[0])+1.0/tan(m_cornerHalfAngles[2]))));

    double caseTwoSnapOff = (tension/m_radius) * (cos(m_conAngleRec) + 
            (2.0*sin(m_conAngleRec))/(1.0/tan(m_cornerHalfAngles[0]) + 1.0/tan(m_cornerHalfAngles[1])));

    bool caseThreePossible(false), caseFourPossible(false);
    double caseThreeSnapOff = oilApexMeetCorner(caseThreePossible, 2);
    double caseFourSnapOff = oilApexMeetCorner(caseFourPossible, 1);

    if(caseThreePossible)
        return min(caseThreeSnapOff, caseTwoSnapOff);
    else if(caseFourPossible)
        return min(caseFourSnapOff, min(caseTwoSnapOff, caseOneSnapOff));
    else if(m_oilInCorner[1]->stableAtPrs(caseTwoSnapOff))
        return min(caseTwoSnapOff, caseOneSnapOff);
    else
        return caseOneSnapOff;    
}

double Square::snapOffPrsDrainNR() const
{
        return (m_oil->interPhaseTen()/m_radius) * (cos(m_conAngleRec) + sin(m_conAngleRec));
}

bool Polygon::fillWithWater(bool snapOffOverRide)
{
    m_bulkFluid = m_water;
    m_containsWater = true;
    m_containsOil = false;
    double pc = m_commonData->datumCappPress()-m_gravityCorrection;

    for(int i = 0; i < m_numCorners; ++i)
    {
        bool film(false);
        if(m_entryPress != m_snapOffPrs && !snapOffOverRide)
        {
            film = m_oilInCorner[i]->createFilm(pc, m_conAngleAdv, m_maxConAngSpont, m_cornerHalfAngles[i], 
                m_oil->interPhaseTen());
            if(m_oilInCorner[i]->stable()) 
            {
                ++m_numLayers;
                m_eventHistory.push_back(5);
            }
        }
        
        if(!film) m_waterInCorner[i]->removeCorner();
    }

    if(m_numLayers)
    {
        m_containsOil = true;
        m_anyOilLayers = true;
    }
    m_allOilLayers = m_numLayers == m_numCorners;
    
    if(snapOffOverRide) return true;
    
    return m_entryPress == m_snapOffPrs;
}

bool Circle::fillWithWater(bool snapOffOverRide) 
{
    m_bulkFluid = m_water;
    m_containsOil = false;
    m_containsWater = true;
    return false;
}

bool Polygon::fillWithOil() 
{
    m_bulkFluid = m_oil;
    m_containsOil = true;
    double pc = m_commonData->datumCappPress()-m_gravityCorrection;
    double conAng = m_virginState ? m_conAngleInit: m_conAngleRec;
    
    for(int i = 0; i < m_numCorners; ++i)
    {
        m_waterInCorner[i]->createFilm(pc, conAng, m_maxConAngSpont, m_cornerHalfAngles[i], m_oil->interPhaseTen());
        m_oilInCorner[i]->stable(false);
        m_oilInCorner[i]->removeLayer();
    }
    m_numLayers = 0;
    m_allOilLayers = false;
    m_anyOilLayers = false;

    m_containsWater = m_waterInCorner[0]->exists();
    return m_entryPress == m_snapOffPrs;
}

bool Circle::fillWithOil() 
{
    m_bulkFluid = m_oil;
    m_containsOil = true;
    m_containsWater = false;
    return false;
}

bool Shape::containsFluid(const Fluid* fluid) const
{
    if(m_parent->isExitOrEntryRes())
        return true;
    else if(fluid == m_oil)
        return m_containsOil;
    else
        return m_containsWater;
}

bool Polygon::unstableWaterConfig(double cappPress) const
{
    bool watFlood(m_commonData->injectant() == m_water);
    bool bulkOil(m_bulkFluid == m_oil);
    const pair<bool, double>& watTrp = m_waterInCorner[0]->trappedInside();

    return watFlood && bulkOil && watTrp.first && watTrp.second < m_snapOffPrs && cappPress < m_snapOffPrs;
}

bool Polygon::unstableOilConfig(double cappPress) const
{
    bool oilFlood(m_commonData->injectant() == m_oil);
    bool bulkWater(m_bulkFluid == m_water);
    const pair<bool, double>& oilTrp = m_oilInCorner[0]->trappedOutside();

    return oilFlood && bulkWater && m_anyOilLayers && oilTrp.first && oilTrp.second > m_snapOffPrs && cappPress > m_snapOffPrs;
}
