#ifndef SHAPE_H
#define SHAPE_H

class Fluid;
class RockElem;
class CoalesceWatFillCmp;
class CoalesceOilFillCmp;

#include "commonData.h"
#include "apex.h"

////////////////////////////// BASE CLASS FOR PORE SHAPES ////////////////////////////////
class Shape
{

    friend ostream& operator<< (ostream&, Shape&);

public:

    Shape(RockElem*, CommonData*, const Fluid*, const Fluid*, double, double, double, int);
    Shape(RockElem*, CommonData*, const Fluid*, const Fluid*, const Shape&);
    virtual ~Shape(){}

    virtual double updateState(double pc) = 0;
    virtual void initDrainage(double pc) = 0;
    virtual void initImbibition(double pc) = 0;
    virtual bool fillWithWater(bool snapOffOverRide = false) = 0;
    virtual bool fillWithOil() = 0; 
    virtual bool updateImbEntryPrs(double& newEntryPrs) const = 0;
    virtual bool updateDrainEntryPrs(double& newEntryPrs) const = 0;
    virtual void setShapeFactor(double shapeFact) = 0;
    virtual bool unstableWaterConfig(double cappPress) const = 0;
    virtual bool unstableOilConfig(double cappPress) const = 0;
    virtual void checkStateOfInitConAng() = 0;

    double gravCorrectedEntryPress() const {return m_entryPress+m_gravityCorrection;}        
    void entryPress(double prs) {m_entryPress = prs;}
    double radius() const {return m_radius;}
    double area() const {return m_area;}
    double conAngEquil() const {return m_conAngEquil;}
	double shapeFactor() const {return m_shapeFactor;}
    double gravityCorrection() const {return m_gravityCorrection;}
    double oilConductance() const {return m_conductanceOil;}
    bool containsOil() const {return m_containsOil;}
    const CommonData* commonData() const {return m_commonData;}
    bool containsWat() const {return m_containsWater;}
    bool anyOilLayers() const {return m_anyOilLayers;}
    bool allOilLayers() const {return m_allOilLayers;}
    bool bulkOil() const {return m_bulkFluid == m_oil;}
    bool fractionalWetPopl(int polulation) const {return m_wettingCluster == polulation;}
    const Fluid* bulkFluid() const {return m_bulkFluid;} 
    const vector< double >& waterSatHistory() const {return m_waterSatHistory;}
    void recordSw() {m_waterSatHistory.push_back(m_areaWater/m_area);}
    void reduceNeighbourCount() {--m_numNeighbours;}
    void setFracWetPopl(int polulation) {m_wettingCluster = polulation;}
    int numNeighbours() const {return m_numNeighbours;}
    RockElem* parent() const {return m_parent;}
    void addFillingEventToHistory(short event) {m_eventHistory.push_back(event);}
    double waterSat() const {return m_areaWater/m_area;}
    inline void setGravityCorrection(const Node* node); 
    inline void setGravityCorrection(const Node* nodeOne, const Node* nodeTwo); 


    void setContactAngles(double equilConAng, int wettClass, double modelTwoSepAng);
    bool containsFluid(const Fluid* fluid) const;

    inline double conductance(const Fluid* fluid, bool film, bool bulk, bool singlePhase = false) const;
    inline double waterConductance(FluidBlob blob, bool singlePhase = false) const;
    inline double crossSectionalArea(const Fluid* fluid, bool connectedToExit = false) const;
    inline bool containsWat(FluidBlob blob) const;
    inline void setRadius(double newRad);
    inline double equivalentRadius() const;
    inline void assure(bool cond, const string errorMsg = "Shit") const;
    inline bool waterWet() const {return m_conAngleAdv < acos(-1.0)/2.0;}

protected:

    virtual double centerConductance(double area, double visc) const = 0; 

    static const int                MAX_NEWT_ITR;           
    static const double             EPSILON;               
    static const double             PI;
    static const double             INF_NEG_NUM;
        
    RockElem*                       m_parent;
    CommonData*                     m_commonData;
    const Fluid*                    m_oil;
    const Fluid*                    m_water;
    double                          m_radius;
    double                          m_shapeFactor;
    const Fluid*                    m_bulkFluid;
    int                             m_numNeighbours;
    
    bool                            m_anyOilLayers;
    bool                            m_allOilLayers;
    bool                            m_containsWater;
    bool                            m_containsOil;
    bool                            m_virginState;
    int                             m_wettingCluster;
    double                          m_conAngleInit;    
    double                          m_conAngEquil;
    double                          m_conAngleAdv;
    double                          m_conAngleRec;
    double                          m_maxConAngSpont;
    double                          m_area;
    double                          m_areaWater;    
    double                          m_entryPress;
    double                          m_gravityCorrection;
    double                          m_watCondMultFactWhenOilFilled;
    double                          m_conductanceOil;
    pair<double, double>            m_conductanceWater;   
    double                          m_pistonTypeCurveRad;
    vector< double >                m_waterSatHistory;
    
    // Oil piston type = 1, oil snap off = 2, water piston type = 3, water snap off = 4
    // oil layer creation = 5, oil layer collapse = 6, oil layer reformation = 7
    vector< short >                 m_eventHistory;

};

/////////////////////////  BASE CLASS FOR POLYGON SHAPES  /////////////////////////////////

class Polygon : public Shape
{
public:
    
    Polygon(RockElem*, CommonData*, const Fluid*, const Fluid*, double, double, double, int, int);
    Polygon(RockElem*, CommonData*, const Fluid*, const Fluid*, const Polygon&);
    virtual ~Polygon();

    virtual double updateState(double pc);
    virtual void initDrainage(double pc);
    virtual void initImbibition(double pc);
    virtual bool fillWithWater(bool snapOffOverRide = false);
    virtual bool fillWithOil(); 
    virtual bool updateImbEntryPrs(double& newEntryPrs) const;
    virtual bool updateDrainEntryPrs(double& newEntryPrs) const;
    virtual bool unstableWaterConfig(double cappPress) const;
    virtual bool unstableOilConfig(double cappPress) const;
    virtual void checkStateOfInitConAng();

    LayerApex* oilInCorner(int crn) {return m_oilInCorner[crn];}
    CornerApex* waterInCorner(int crn) {return m_waterInCorner[crn];}
    
    int numCorners() const {return m_numCorners;}

    void satReversal(double pc, bool didInjOil);
    void checkForUnstableWaterConfigs(SortedEvents< ThreeSome< Polygon*, int, double >, CoalesceWatFillCmp >& watEvents, 
        double globalPc);
    void checkForUnstableOilConfigs(SortedEvents< ThreeSome< Polygon*, int, double >, CoalesceOilFillCmp >& oilEvents, 
        double globalPc);
    void updateFilmsForTrapping(double pc);
    double calcSnapOffPressureDrain() const;
    void snapOffPrs(double prs) {m_snapOffPrs = prs;}
    double snapOffPrs() const {return m_snapOffPrs;}

    double reformOilLayer(int corner);
    double collapseOilLayer(int corner);

protected:


    virtual double snapOffPrsDrainNR() const = 0;
    virtual double snapOffPrsImbNR() const = 0;
    
    void oilWithWaterInCorners(double cappPressure);
    void waterWithOilLayers(double cappPressure);
    double calcSnapOffPressureImb() const;
    double layerConductance(double bci, double bco, double areaL, double areaC, double bo, 
        double bi, double conAngI, double conAngO, double halfAng, double mju) const;
    double cornerConductance(double areaDl, double b, double halfAng, double conAng, double mju) const;
    double mspCurveRadImb() const;
    double mspCurveRadHingImb() const; 
    bool mspCurveRadHingImbNR(double& firstGuess, double& err) const;
    double mspCurveRadDrain(double conAng) const;
    double mspCurveRadHingDrain() const; 

    inline double dimLessCornerArea(double halfAng, double contactAng) const; 

    vector< double >                m_cornerHalfAngles;
	vector< CornerApex* >	        m_waterInCorner;
	vector< LayerApex* >			m_oilInCorner;
    double                          m_snapOffPrs;
    short                           m_numLayers;
    int                             m_numCorners;
};

//////////////////////////////  SQUARE PORE SHAPES  //////////////////////////////////////
class Square : public Polygon
{
public:

    Square(RockElem*, CommonData*, const Fluid*, const Fluid*, double, double, int);
    Square(RockElem*, CommonData*, const Fluid*, const Fluid*, const Square&);
    virtual ~Square(){}

    virtual void setShapeFactor(double shapeFact) {};
    
private:

    double oilApexMeetCorner(bool& casePossible, int cor) const;
    virtual double snapOffPrsDrainNR() const;
    virtual double snapOffPrsImbNR() const;
    virtual double centerConductance(double area, double visc) const; 
    
};

/////////////////////////////  TRIANGULAR PORE SHAPES  ////////////////////////////////////
class Triangle : public Polygon
{
public:

    Triangle(RockElem*, CommonData*, const Fluid*, const Fluid*, double, double, double, int);
    Triangle(RockElem*, CommonData*, const Fluid*, const Fluid*, const Triangle&);
    virtual ~Triangle(){}
   
    virtual void setShapeFactor(double shapeFact);

private:

    virtual double snapOffPrsDrainNR() const;
    virtual double snapOffPrsImbNR() const;
    virtual double centerConductance(double area, double visc) const; 
    void getHalfAngles();
    double oilApexMeetCorner(bool& casePossible, int cor) const;

};

/////////////////////////////  CIRCULAR PORE SHAPES  ////////////////////////////////////

class Circle : public Shape
{
public:

    Circle(RockElem*, CommonData*, const Fluid*, const Fluid*, double, double, int);
    Circle(RockElem*, CommonData*, const Fluid*, const Fluid*, const Circle&);
    virtual ~Circle(){}

    virtual double updateState(double pc);
    virtual void initDrainage(double pc);
    virtual void initImbibition(double pc);
    virtual bool fillWithWater(bool snapOffOverRide = false);
    virtual bool fillWithOil(); 
    virtual bool updateImbEntryPrs(double& newEntryPrs) const;
    virtual bool updateDrainEntryPrs(double& newEntryPrs) const;
    virtual void setShapeFactor(double shapeFact) {};
    virtual bool unstableWaterConfig(double cappPress) const {return false;}
    virtual bool unstableOilConfig(double cappPress) const {return false;}
    virtual void checkStateOfInitConAng();

private:

    virtual double centerConductance(double area, double visc) const;     
};

class CoalesceWatFillCmp
{
public:
    bool operator() (const ThreeSome<Polygon*, int, double> elem1, const ThreeSome<Polygon*, int, double> elem2) const
    {
        if(elem1.third() == elem2.third())
            return elem1.second() < elem2.second();
        else
            return elem1.third() < elem2.third();
    }
};

class CoalesceOilFillCmp
{
public:
    bool operator() (const ThreeSome<Polygon*, int, double> elem1, const ThreeSome<Polygon*, int, double> elem2) const
    {
        if(elem1.third() == elem2.third())
            return elem1.second() > elem2.second();
        else
            return elem1.third() > elem2.third();
    }
};


///////////////////////////////////////////////////////////////////////////////////////
//////////////////////////// INLINE FUNCTION DEFINITIONS //////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////
inline void Shape::setRadius(double newRad)
{
    m_radius = newRad;
    m_area = pow(m_radius, 2.0) / (4.0 * m_shapeFactor); 
    assure(m_area > 0.0);
    m_areaWater = m_area;
    m_conductanceWater.second = centerConductance(m_areaWater, m_water->viscosity());
}

inline double Shape::equivalentRadius() const 
{
    return 2.0*m_radius/(1.0+2.0*sqrt(PI*m_shapeFactor));
}

inline double Shape::crossSectionalArea(const Fluid* fluid, bool connectedToExit) const
{
    if(connectedToExit)
        return m_area;
    else if(fluid == m_water)
        return m_areaWater;
    else
        return m_area - m_areaWater;
}

inline double Polygon::dimLessCornerArea(double halfAng, double contactAng) const 
{
    if(fabs(contactAng + halfAng - PI/2.0) < 0.01) 
    {              
        return sin(halfAng)*cos(halfAng);
    }
    else 
    {
        return pow(sin(halfAng)/cos(halfAng+contactAng), 2.0) 
        * (cos(contactAng)*cos(halfAng+contactAng)/sin(halfAng) + halfAng + contactAng - PI/2.0);
    }
}

inline bool Shape::containsWat(FluidBlob blob) const
{
    if(blob == bulkBlob)
        return (m_bulkFluid == m_water);
    else
        return m_containsWater;
}

inline double Shape::waterConductance(FluidBlob blob, bool singlePhase) const
{
    if(singlePhase)
        return centerConductance(m_area, m_water->viscosity());
    else if(blob == filmBlob)
        return m_conductanceWater.first;
    else
        return m_conductanceWater.second;
}

inline double Shape::conductance(const Fluid* fluid, bool filmFlow, bool bulkFlow, bool singlePhase) const
{
    double flowConductance(0.0);
    if(singlePhase)
        flowConductance = centerConductance(m_area, fluid->viscosity());
    else if(fluid == m_oil) 
        flowConductance = m_conductanceOil;
    else if(!m_allOilLayers || (filmFlow && bulkFlow))
        flowConductance = m_conductanceWater.first + m_conductanceWater.second;
    else if(filmFlow)
        flowConductance = m_conductanceWater.first;
    else
        flowConductance = m_conductanceWater.second;
    
    assure(flowConductance > 0.0);
    return flowConductance;
}

inline void Shape::setGravityCorrection(const Node* node) 
{
    m_gravityCorrection = (m_water->density()-m_oil->density())*
        (m_commonData->gravConstX()*node->xPos() + 
        m_commonData->gravConstY()*node->yPos() + 
        m_commonData->gravConstZ()*node->zPos());
}

inline void Shape::setGravityCorrection(const Node* nodeOne, const Node* nodeTwo) 
{
    double xPos((nodeOne->xPos()+nodeTwo->xPos())/2.0);
    double yPos((nodeOne->yPos()+nodeTwo->yPos())/2.0);
    double zPos((nodeOne->zPos()+nodeTwo->zPos())/2.0);

    m_gravityCorrection = (m_water->density()-m_oil->density())*
        (m_commonData->gravConstX()*xPos + 
        m_commonData->gravConstY()*yPos + 
        m_commonData->gravConstZ()*zPos);
}

inline void Shape::assure(bool cond, const string errorMsg) const
{
    return;     // comment for checking
    if(!cond)
    {
        cerr << "Shape::" << errorMsg << endl;
        //exit(-1);
    }
}
#endif

