#ifdef WIN32
#pragma warning(disable:4786)        
#endif

#include <fstream>
#include <iostream>
#include <sstream>
#include <iomanip>
#include <cmath>
#include <string>
#include <cassert>
#include <algorithm>
#include <functional>
#include <set>
#include <utility>
#include <vector>
#include <ctime>
#include <typeinfo>
#include <map>
using namespace std;

#include "f2c.h"
#include "AMG_prototypes.h"

#include "sortedEvents.h"
#include "threeSome.h"
#include "inputData.h"
#include "fluid.h"
#include "node.h"
#include "shape.h"
#include "rockElem.h"
#include "compareFuncs.h"
#include "solver.h"

const double            Solver::SCALE_FACTOR = 1.0;
int                     Solver::MAT_MEM_SCALE = 3;
bool                    Solver::INITIALISED = false;
bool                    Solver::USE_GRAVITY = false;
long                    Solver::SYMMETRIC_MAT = 22;
long                    Solver::SLVR_OUTPUT = 10;
long                    Solver::VERBOSE_SLVR = 11111;
double                  Solver::TOLERANCE = 1.0E-15;


//////////////////////////////////////////////////////////////////////////////////////////
// This is the constructor for the solver class. Given the network it allocates memory 
// for various C type arrays. These are set to the maximum length possible which is when 
// all pores contain the fluid for which the flowrate is going to be obtained.
//////////////////////////////////////////////////////////////////////////////////////////
Solver::Solver(const vector< RockElem* >& network, const vector< RockElem* >& inlet, const vector< RockElem* >& outlet, 
               int outletIdx, int maxNonZeros, const string& matFileName, bool matlabFormat) : m_network(network), 
               m_inletPores(inlet), m_outletPores(outlet), m_outletIdx(outletIdx), m_maxNonZeros(maxNonZeros), 
               m_matrixFileName(matFileName), m_matlabFormat(matlabFormat), m_probSize(outletIdx-1), m_matElemFactor(3),
               m_matSizeFactor(5)              
{
    long maxNdaFactor = static_cast< long >(m_matElemFactor*m_maxNonZeros) + 
        m_probSize*static_cast< long >(m_matSizeFactor*MAT_MEM_SCALE);

    m_matElems = new double[maxNdaFactor];         // These are all set to maximum number of non zero elements, ie         
    m_colIndex = new long[maxNdaFactor];            // the number it would be if all rock elements were invaded with 
    m_rowPtr = new long[3*m_probSize+1];

    m_solVecBuffer = new double[3*m_probSize];
    m_rhsVecBuffer = new double[3*m_probSize];        // fluid to be solved for. This will be the case for water but not
    m_someIdxBuffer = new long[12*m_probSize];
    
    m_colHash = new int[m_probSize];                // oil which never invades the smallest throats/pores.
    m_poreHash = new Pore*[m_probSize];
}

Solver::Solver(const Solver& solver) : m_network(solver.m_network), m_inletPores(solver.m_inletPores), 
               m_outletPores(solver.m_outletPores), m_outletIdx(solver.m_outletIdx), m_maxNonZeros(solver.m_maxNonZeros), 
               m_matrixFileName(solver.m_matrixFileName), m_matlabFormat(solver.m_matlabFormat), m_probSize(solver.m_probSize), 
               m_matElemFactor(solver.m_matElemFactor), m_matSizeFactor(solver.m_matSizeFactor)
{
    long maxNdaFactor = static_cast< long >(m_matElemFactor*m_maxNonZeros) + 
        m_probSize*static_cast< long >(m_matSizeFactor*MAT_MEM_SCALE);

    m_matElems = new double[maxNdaFactor];                 
    m_colIndex = new long[maxNdaFactor];            
    m_rowPtr = new long[3*m_probSize+1];
    m_solVecBuffer = new double[3*m_probSize];
    m_rhsVecBuffer = new double[3*m_probSize];
    m_someIdxBuffer = new long[12*m_probSize];    
    m_colHash = new int[m_probSize]; 
    m_poreHash = new Pore*[m_probSize];
}

Solver::~Solver()
{
    delete[] m_matElems;
    delete[] m_rhsVecBuffer;
    delete[] m_solVecBuffer;
    delete[] m_colHash;
    delete[] m_poreHash;
    delete[] m_rowPtr;
    delete[] m_colIndex;
    delete[] m_someIdxBuffer;
}

void Solver::initSolver(double eps, int scaleFact, int slvrOutput, bool verboseSlvr, bool useGrav)
{    
    if(!INITIALISED)
    {        
        ofstream tmp("fort.11");    // Clen the solver dump....
        tmp.close();

        INITIALISED = true;
        USE_GRAVITY = useGrav;
        SYMMETRIC_MAT = 12;     // Set to 22 if matrix is not symmetric 
        if(verboseSlvr)
            VERBOSE_SLVR = 10606;
        else
            VERBOSE_SLVR = 11111;

        MAT_MEM_SCALE = scaleFact;
        TOLERANCE = eps;
        SLVR_OUTPUT = 10 + slvrOutput;
    }
}


///////////////////////////////////////////////////////////////////////////////////////////
// In case we want to solve a matrix that we have as file. Only used for deugging
// purposes and is NEVER called.
// NB! The matrix that is read is zero based.
///////////////////////////////////////////////////////////////////////////////////////////
void Solver::readMatrixStd(const string& fileName)
{
    string fileMat = fileName + "_mat.out";
    string fileRhs = fileName + "_rhs.out";
    string fileSol = fileName + "_sol.out";
    
    ifstream inMat(fileMat.c_str());
    ifstream inRhs(fileRhs.c_str());
    ifstream inSol(fileSol.c_str());
    
    inMat.ignore(512, '\n');
    inMat.ignore(512, '\n');
    inRhs.ignore(512, '\n');
    inSol.ignore(512, '\n');
    
    int workRow(-1), workElem(-1);
    int rowPtr;
    while(inMat >> rowPtr) {
        ++workElem;
        if(rowPtr > workRow) {
            workRow = rowPtr;
            m_rowPtr[workRow] = workElem+1;     // Change from zero to one based matrix
        }
        inMat >> m_colIndex[workElem];
        ++m_colIndex[workElem];                 // Change from zero to one based matrix
        inMat >> m_matElems[workElem];
    }
    m_rowPtr[++workRow] = ++workElem+1;
    inMat.close();
    
    workRow = -1;
    while(inRhs >> m_rhsVecBuffer[++workRow]) {}
    inRhs.close();
    
    workRow = -1;
    while(inRhs >> m_solVecBuffer[++workRow]) {}
    inSol.close();
}

///////////////////////////////////////////////////////////////////////////////////////////
// Writes the matrix to file in matlab format. Again, used for debugging purposes where 
// we want to check the solution with a different solver eg matlab
///////////////////////////////////////////////////////////////////////////////////////////
void Solver::writeVelocityEntry(ostream& out, pair<const RockElem*, double> entry, const Fluid* fluid, bool resistivitySolve) const
{
	const RockElem* throat = entry.first;
	const RockElem* pore_i = throat->connection(0);
	const RockElem* pore_j = throat->connection(1);
    double deltaHeight = pore_i->node()->zPos()-pore_j->node()->zPos();
    if(!USE_GRAVITY || pore_i->isExitOrEntryRes() || pore_j->isExitOrEntryRes()) deltaHeight = 0.0;

	double presure_i(-9.9), presure_j(-9.9), tmp(0.0);           
	pore_i->prevSolvrRes(fluid, resistivitySolve, tmp, presure_i, tmp);
	pore_j->prevSolvrRes(fluid, resistivitySolve, tmp, presure_j, tmp);
	double gravCorr(pore_i->gravityCorrection(fluid->density(), pore_i->node()->xPos()-pore_j->node()->xPos(),
        pore_i->node()->yPos()-pore_j->node()->yPos(), pore_i->node()->zPos()-pore_j->node()->zPos()));
    double flowrate(entry.second * (presure_i - presure_j - gravCorr));
	
	double sat_i(pore_i->waterSaturation()), sat_j(pore_j->waterSaturation()), sat_t(throat->waterSaturation());
	if(dynamic_cast<const Oil* >(fluid) != 0) {
		sat_i = 1.0 - sat_i;
		sat_j = 1.0 - sat_j;
		sat_t = 1.0 - sat_t;
	}

	if(pore_i->isExitOrEntryRes()){			// Reservoirs and throats connected to 
		sat_i = 1.0;						// reservoirs are always assumed single
		sat_t = 1.0;						// phase
	}
	else if(pore_j->isExitOrEntryRes()){
		sat_j = 1.0;
		sat_t = 1.0;
	}

	
	out << setw(8) << throat->orenIndex() 
		<< setw(8) << pore_i->orenIndex()
		<< setw(8) << pore_j->orenIndex()
		<< setw(16) << presure_i
		<< setw(16) << presure_j
		<< setw(16) << flowrate
		<< setw(16) << flowrate / (pore_i->shape()->area() * sat_i)
		<< setw(16) << flowrate / (pore_j->shape()->area() * sat_j)
		<< setw(16) << flowrate / (throat->shape()->area() * sat_t);	
}

void Solver::dumpVelocityMatlab(const string& fileName, const Fluid* fluid, bool resistivitySolve) const 
{
    string fileVelocity = fileName + "_velocity.m";
    
    ofstream out(fileVelocity.c_str());
    out.flags(ios::showpoint);
    out.flags(ios::scientific);

    string header("{'Throat index'}");
	out << "%=== " << (int)m_throatConductances.size() << " ===" << endl;
	out << "leg = ["  
		<< "{'Throat index'} "  
		<< "{'Pore i index'} " 
		<< "{'Pore j index'} " 
		<< "{'Pore i pressure (Pa)'} " 
		<< "{'Pore j pressure (Pa)'} " 
		<< "{'Flowrate (m3/s)'} " 
		<< "{'Pore i velocity (m/s)'} " 
		<< "{'Pore j velocity (m/s)'} " 
		<< "{'Throat velocity (m/s)'} " 
		<< "]; "
		<< endl;

    out << "Velocity = [... " << endl;
    for(size_t i = 0; i < m_throatConductances.size(); ++i) 
    {
		writeVelocityEntry(out, m_throatConductances[i], fluid, resistivitySolve);
		out << "; ..." << endl;
    }
    
    out << "]; "  << endl;        
    out.close();   
}

void Solver::dumpVelocityStd(const string& fileName, const Fluid* fluid, bool resistivitySolve) const 
{
    string fileVelocity = fileName + "_velocity.out";
    
    ofstream out(fileVelocity.c_str());
    out.flags(ios::showpoint);
    out.flags(ios::scientific);

    out << "Velocity output for " << (int)m_throatConductances.size() << " throats" << endl;
	out << 9 << endl
		<< "Throat index" << endl 
		<< "Pore i index" << endl
		<< "Pore j index" << endl
		<< "Pore i pressure (Pa)" << endl
		<< "Pore j pressure (Pa)" << endl
		<< "Flowrate (m3/s)" << endl
		<< "Pore i velocity (m/s)" << endl
		<< "Pore j velocity (m/s)" << endl
		<< "Throat velocity (m/s)" << endl;

    for(size_t i = 0; i < m_throatConductances.size(); ++i) {
		writeVelocityEntry(out, m_throatConductances[i], fluid, resistivitySolve);
		out << endl;
    }
    out.close();   
}


///////////////////////////////////////////////////////////////////////////////////////////
// Writes the matrix to file in matlab format. Again, used for debugging purposes where 
// we want to check the solution with a different solver eg matlab
///////////////////////////////////////////////////////////////////////////////////////////
void Solver::dumpMatrixMatlab(const string& fileName) const 
{
    string fileMat = fileName + "_mat.m";
    string fileRhs = fileName + "_rhs.m";
    string fileSol = fileName + "_sol.m";
    
    ofstream outMat(fileMat.c_str());
    ofstream outRhs(fileRhs.c_str());
    ofstream outSol(fileSol.c_str());

    outMat.flags(ios::scientific);
    outRhs.flags(ios::scientific);
    outSol.flags(ios::scientific);

    m_rowPtr[m_matrixSize] = m_finalRowPointer;     //AMG seems to overwrite the final row pointer for some reason
                                                    //Since AMG is finished, let's just write it back....
    outMat << "%=== " << m_matrixSize << " ===" << endl;
    int workE(0);
    outMat << "rowIdx = [... " << endl;
    for(int e = 0; e < m_matrixSize; ++e) {
        while(workE < m_rowPtr[e+1]-1) {
            outMat << e+1 << "; ";
            if(!(++workE%10)) 
                outMat << endl;
        }
    }
    outMat << "];" << endl << endl;


    outMat << "colIdx = [ ... " << endl;
    for(int f = 0; f < workE; ++f) {
        outMat << m_colIndex[f] << "; ";
        if(!(f%10)) 
            outMat << endl;
    }
    outMat << "];" << endl << endl;
    
    outMat << "matVals = [ ... " << endl;
    for(int g = 0; g < workE; ++g) {
        outMat << setprecision(12) << m_matElems[g] << "; ";
        if(!(g%5)) 
            outMat << endl;
    }
    outMat << "];" << endl << endl;
   
    outMat << "Mat = sparse(rowIdx', colIdx', matVals', " << m_matrixSize << ", " << m_matrixSize << ");" << endl;        
    outMat.close();
    
    outRhs << "% === " << m_matrixSize << " ===" << endl;
    outRhs << "rhsVec = [ ..." << endl;
    for(int h = 0; h < m_matrixSize; ++h) {
        outRhs << setprecision(12) << m_rhsVecBuffer[h] << "; ";
        if(!((h+1)%5)) 
            outRhs << endl;
    }
    outRhs << "];" << endl;
    outRhs.close();
    
    outSol << "% === " << m_matrixSize << " ===" << endl;
    outSol << "Sol = [ ..." << endl;
    for(int o = 0; o < m_matrixSize; ++o) {
        outSol << setprecision(12) << m_solVecBuffer[o] << "; ";
        if(!((o+1)%5)) 
            outSol << endl;
    }
    outSol << "];" << endl;
    outSol.close();
}

///////////////////////////////////////////////////////////////////////////////////////////
// Dumps the matrix in column oriented format. 
///////////////////////////////////////////////////////////////////////////////////////////
void Solver::dumpMatrixStd(const string& fileName) const 
{
    string fileMat = fileName + "_mat.m";
    string fileRhs = fileName + "_rhs.m";
    string fileSol = fileName + "_sol.m";
    
    ofstream outMat(fileMat.c_str());
    ofstream outRhs(fileRhs.c_str());
    ofstream outSol(fileSol.c_str());

    m_rowPtr[m_matrixSize] = m_finalRowPointer;     //AMG seems to overwrite the final row pointer for some reason
                                                    //Since AMG is finished, let's just write it back....

    outMat << "=== " << m_matrixSize << " ===" << endl;
    outMat << setw(10) << "row" << setw(10) << "col" << setw(18) << "value" << endl; 
    int workE(0);
    for(int e = 0; e < m_matrixSize; ++e) {
        while(workE < m_rowPtr[e+1]-1) {
            outMat << setw(10) << e 
                << setw(10) << m_colIndex[workE]-1 
                << setw(18) << m_matElems[workE]
                << endl;
            ++workE;
        }
    }
    outMat.close();
    
    outRhs << "=== " << m_matrixSize << " ===" << endl;
    for(int f = 0; f < m_matrixSize; ++f)
        outRhs << m_rhsVecBuffer[f] << endl;
    outRhs.close();
    
    outSol << "=== " << m_matrixSize << " ===" << endl;
    for(int g = 0; g < m_matrixSize; ++g)
        outSol << m_solVecBuffer[g] << endl;
    outSol.close();        
}

///////////////////////////////////////////////////////////////////////////////////////////
// This is the only public function of the solver class. It computes the outlet flow of a 
// given fluid from the network given the inlet and outlet pressures. The solved pressure
// field is not retained. The matrix is solved using a sparse matrix solver (BiCG) for
// quick solution times. No assumptions are being made about the structure of the matrix.
///////////////////////////////////////////////////////////////////////////////////////////
double Solver::flowrate(double inletPrs, double outletPrs, const Fluid *fluid, double& flowError, double& elapsed, 
                        double waterSat, bool writeVelocity, bool writeMatrix, bool resistivitySolve)
{            
    for(size_t i = 0; i < m_network.size(); ++i)
        m_network[i]->clearAllSolverFlag();
    
    bool outletConnectionExists(false);                     // For each pore at inlet boundary a path to the outlet is tried found.
    for(size_t bdrP = 0; bdrP < m_inletPores.size(); ++bdrP)   // The pores existing within this path is marked in the process
    {
        if(m_inletPores[bdrP]->connectedToOutlet(fluid)) 
        {
            outletConnectionExists = true;
        }
    }
    
    if(!outletConnectionExists) return 0.0;                 // No fluid connection exists between inlet and outlet
            
    fillMatrix(inletPrs, outletPrs, fluid, writeVelocity, resistivitySolve);

    clock_t  startSolve(clock());
        
    // parameters suggested in the description in the file 'amg.c'
    long ifirst(10), iswtch(4), ierr;
    double eps(TOLERANCE);

    long nda  = static_cast< long >(m_matElemFactor*m_nonZeroElems) + m_matrixSize*static_cast< long >(m_matSizeFactor*MAT_MEM_SCALE);
    long ndja = nda;
    long ndia = 3 * m_matrixSize;
    long ndu  = ndia;
    long ndf  = ndia;
    long ndig = 4 * ndia;
       
    ierr = aux1r5_(m_matElems, m_rowPtr, m_colIndex, m_solVecBuffer, m_rhsVecBuffer, 
        m_someIdxBuffer, &nda, &ndia, &ndja, &ndu, &ndf, &ndig, &m_matrixSize, &SYMMETRIC_MAT, &eps, 
        &ifirst, &iswtch, &SLVR_OUTPUT, &VERBOSE_SLVR, &ierr );
    
    if(ierr != 0) 
    {
        string fluidType("water");
        if(dynamic_cast< const Oil * >(fluid) != NULL)
            fluidType = "oil";

        cout << endl
            << "=============================================================="     << endl
            << "Warning: The pressure solver failed to reach the requested"         << endl 
            << "tolerance. The solution might be incorrect."                        << endl
            << "Error code: " << ierr                                               << endl
            << "Achieved error norm was: " << eps                                   << endl
            << "Solving for fluid: " << fluidType                                   << endl
            << "=============================================================="     << endl
            << endl;
    }

    elapsed += (double)(clock() - startSolve) / CLOCKS_PER_SEC;
        
    for(int j = 0; j < m_matrixSize; ++j)                                           // Pass back the results from the
        m_poreHash[j]->solverResults(fluid, resistivitySolve, m_solVecBuffer[j]);   // solver. These values will 
                                                                                    // subsequently be used for calculating
    for(size_t inp = 0; inp < m_inletPores.size(); ++inp)                              // pressure profiles on the lattice
    {
        Pore *inletPore = dynamic_cast< Pore* >(m_inletPores[inp]);
        if(inletPore) 
        {
            double localInletPrs(inletPrs);
            if(!resistivitySolve && USE_GRAVITY)
            {
                localInletPrs += inletPore->gravityCorrection(fluid->density(), inletPore->node()->xPos(), 
                    inletPore->node()->yPos(), inletPore->node()->zPos());
            }

            inletPore->solverResults(fluid, resistivitySolve, localInletPrs);
        }
    }

    for(size_t op = 0; op < m_outletPores.size(); ++op)
    {
        Pore *outletPore = dynamic_cast< Pore* >(m_outletPores[op]);
        if(outletPore) 
        {
            double localOutletPrs(outletPrs);
            if(!resistivitySolve && USE_GRAVITY)
            {
                localOutletPrs += outletPore->gravityCorrection(fluid->density(), outletPore->node()->xPos(),
                    outletPore->node()->yPos(), outletPore->node()->zPos());
            }

            outletPore->solverResults(fluid, resistivitySolve, localOutletPrs);
        }
    }
  
    if(writeMatrix) 
    {
        ostringstream fileName;
        fileName.flags(ios::showpoint);
        fileName.flags(ios::fixed);
        fileName.precision(0);
        fileName << m_matrixFileName;

        if(resistivitySolve)
            fileName << "_resist";
        else if(dynamic_cast< const Water* >(fluid) != 0)
            fileName << "_water";
        else
            fileName << "_oil";

        fileName << "_sw_" << waterSat*100.0;

        if(m_matlabFormat)
            dumpMatrixMatlab(fileName.str());
        else
            dumpMatrixStd(fileName.str());
    }
        
	if(writeVelocity) {
        ostringstream fileName;
        fileName.flags(ios::showpoint);
        fileName.flags(ios::fixed);
        fileName.precision(0);
        fileName << m_matrixFileName;

        if(resistivitySolve)
            fileName << "_resist";
        else if(dynamic_cast< const Water* >(fluid) != 0)
            fileName << "_water";
        else
            fileName << "_oil";

        fileName << "_sw_" << waterSat*100.0;

        sort(m_throatConductances.begin(), m_throatConductances.end(), throatIndexCompare());
        pair<const RockElem*, double> dummy(0, 0.0);
        int idx(-99);
        for(size_t i = 0; i < m_throatConductances.size(); ++i)
        {
            int tmp(m_throatConductances[i].first->orenIndex());
            if(tmp == idx) m_throatConductances[i] = dummy;
            idx = tmp;
        }
        m_throatConductances.erase(remove(m_throatConductances.begin(), m_throatConductances.end(), dummy),
            m_throatConductances.end());

        if(m_matlabFormat)
            dumpVelocityMatlab(fileName.str(), fluid, resistivitySolve);
        else
            dumpVelocityStd(fileName.str(), fluid, resistivitySolve);
		
		m_throatConductances.clear();	// Clean up after yourself
    }
    
    return getFlowRate(fluid, flowError, resistivitySolve);
}

///////////////////////////////////////////////////////////////////////////////////////////
// Only the pressures in the pores connected to the outlet are actually needed to compute
// the flowrate leaving the network. The conductances and index pointers to the solution
// vector are stored in a vector that is filled during matrix filling.
//////////////////////////////////////////////////////////////////////////////////////////
double Solver::getFlowRate(const Fluid* fluid, double& flowError, bool resistSolve) const
{
    double flowOut(0.0), flowIn(0.0);

    for(size_t i = 0; i < m_networkOutlets.size(); ++i)
    {
        double porePrs = m_solVecBuffer[m_networkOutlets[i].first()];
        double outletPrs = m_networkOutlets[i].third();
        double conductance = m_networkOutlets[i].second() / SCALE_FACTOR;
        double gravCorr = m_networkOutlets[i].fourth();

        flowOut += conductance * (porePrs - outletPrs - gravCorr);
    }

    for(size_t j = 0; j < m_networkInlets.size(); ++j)
    {
        double porePrs = m_solVecBuffer[m_networkInlets[j].first()];
        double inletPrs = m_networkInlets[j].third();
        double conductance = m_networkInlets[j].second() / SCALE_FACTOR;
        double gravCorr = m_networkInlets[j].fourth();

        flowIn += conductance * (inletPrs - porePrs + gravCorr);
    }

    flowError = fabs(flowOut - flowIn) / flowOut;
    return (flowOut + flowIn) / 2.0;
}

//////////////////////////////////////////////////////////////////////////////////////////////
// Both the matrix and rhs vectors are filled. The matrix is filled taking advantage of it's 
// sparse property. Hence it is represented in terms a three arrays; value, row pointer and
// column index. The matrix size and number of non-zero elements are also determined. The rhs
// vector is stored in a C type array rather than the MV format since the required size is
// not known during filling. It'll have to be converted before solving the system.
//////////////////////////////////////////////////////////////////////////////////////////////
void Solver::fillMatrix(double inletPrs, double outletPrs, const Fluid *fluid, bool writeVelocity, bool resistivitySolve)
{
    int row(0), nonZeroIdx(0), poreIdx, conn;
    m_networkOutlets.clear();                               // Ensure we don't have left over soliutions from before
    m_networkInlets.clear();
    for(poreIdx = 1; poreIdx < m_outletIdx; ++poreIdx)
    {
        Pore *currPore = dynamic_cast< Pore* >(m_network[poreIdx]);
        if(currPore == NULL)
        {
            cerr << "==============================================" << endl
                << "Error: Did not retrieve correct pore in solver" << endl
                << "==============================================" << endl;
            exit(-1);
        }                                                   // We need some way of recording which column in the matrix
                                                            // a given index refers to. Searching the node vector would
        m_colHash[poreIdx - 1] = row;                       // be to inefficient => O(n). The node vector is used the 
                                                            // way, ie given a value in the solution vector we need to 
                                                            // quickly figure out which pore index it refers to 
        
        if(currPore->canBePassedToSolver() && currPore->isInsideSolverBox())         
        {
            m_rowPtr[row] = nonZeroIdx+1;                   // The row pointer contains a index pointer to the first element 
            m_poreHash[row] = currPore;                     // in every row in the value array. Used to create the sparse mat

            vector< pair<int, double> > nonZeros;
            double conductanceSum(0.0);
            m_rhsVecBuffer[row] = 0.0;                      
            
            double tmp(0.0);
            currPore->prevSolvrRes(fluid, resistivitySolve, tmp, m_solVecBuffer[row], tmp);
            
            for(conn = 0; conn < currPore->connectionNum(); ++conn)
            {
                double conductance(0.0);
				double deltaGrav(0.0);

				const RockElem* throat = currPore->connection(conn);                
                RockElem *nextPore = currPore->nextPore(conn, conductance, deltaGrav, fluid, resistivitySolve);

                assert(conductance >= 0.0);
                if(!USE_GRAVITY) deltaGrav = 0.0;
                conductance *= SCALE_FACTOR;                // Some of the preconditioners will assume a value to be zero if very
                                                            // small. Let's just scale the matrix a bit...
                                                            // Check that there is fluid conductance between pores
                if(conductance > 0.0)                       
                {
                    int nextPoreIndex(nextPore->node()->index());
					
					if(writeVelocity) 
                    {
						pair<const RockElem*, double> veloEntry(throat, conductance / SCALE_FACTOR);
						m_throatConductances.push_back(veloEntry);
					}
                   
                    conductanceSum += conductance;
                    m_rhsVecBuffer[row] += conductance*deltaGrav;
                    
                    if(nextPore->isOnInletSlvrBdr())                             // Inlet
                    {
                        double localInletPrs(inletPrs);
                        if(!resistivitySolve && USE_GRAVITY && nextPore->isExitOrEntryRes())
                        {
                            localInletPrs += currPore->gravityCorrection(fluid->density(), currPore->node()->xPos(),
                                currPore->node()->yPos(), currPore->node()->zPos());
                        }
                        else if(!resistivitySolve && USE_GRAVITY && !nextPore->isExitOrEntryRes())
                        {
                            localInletPrs += nextPore->gravityCorrection(fluid->density(), nextPore->node()->xPos(),
                                nextPore->node()->yPos(), nextPore->node()->zPos());
                        }
                                     
                        FourSome<int, double, double, double> inletPoint(row, conductance, localInletPrs, deltaGrav);
                        m_networkInlets.push_back(inletPoint);
                        m_rhsVecBuffer[row] += conductance * localInletPrs;
                    }
                    else if(nextPore->isOnOutletSlvrBdr())                       // Outlet
                    {
                        double localOutletPrs(outletPrs);
                        if(!resistivitySolve && USE_GRAVITY && nextPore->isExitOrEntryRes())
                        {
                            localOutletPrs += currPore->gravityCorrection(fluid->density(), currPore->node()->xPos(),
                                currPore->node()->yPos(), currPore->node()->zPos());
                        }
                        else if(!resistivitySolve && USE_GRAVITY && !nextPore->isExitOrEntryRes())
                        {
                            localOutletPrs += nextPore->gravityCorrection(fluid->density(), nextPore->node()->xPos(),
                                nextPore->node()->yPos(), nextPore->node()->zPos());
                        }

                        FourSome<int, double, double, double> outletPoint(row, conductance, localOutletPrs, deltaGrav);    
                        m_networkOutlets.push_back(outletPoint);                
                        m_rhsVecBuffer[row] += conductance * localOutletPrs;
                    }
                    else                                                        // Interior point
                    {
                        pair< int, double > entry;
                        entry.first = nextPoreIndex;
                        entry.second = - conductance;
                        nonZeros.push_back(entry);
                   }
                }
            }
            pair< int, double > entry;
            entry.first = poreIdx;
            entry.second = conductanceSum;
            nonZeros.push_back(entry);
            
            sort(nonZeros.begin(), nonZeros.end(), poreDiagonalFirst());
            for(size_t elem = 0; elem < nonZeros.size(); ++elem)
            {
                m_colIndex[nonZeroIdx] = nonZeros[elem].first;
                m_matElems[nonZeroIdx] = nonZeros[elem].second;                  
                ++nonZeroIdx;
            }            
            
            if(conductanceSum == 0.0)
            {
                string fluidType("water");
                if(typeid(*fluid) == typeid(const Oil)) fluidType = "oil";
                
                
                cout << endl 
                    << "=========================================================="         << endl
                    << "Whoa.. Shit went down in pore: " << poreIdx                         << endl
                    << "Number of connections: " <<  currPore->connectionNum()              << endl                    
                    << "Fluid we're solving for is: " << fluidType                          << endl
                    << "Contains oil: " << currPore->shape()->containsOil()                 << endl
                    << "Contains water: " << currPore->shape()->containsWat()               << endl
                    << "Bulk oil: " << currPore->shape()->bulkOil()                         << endl
                    << "Connection list: "                                                  << endl;

                for(int i = 0; i < currPore->connectionNum(); ++i)
                {
                    int throatIdx(currPore->connection(i)->latticeIndex() - m_outletIdx);
                    bool throatHasFluid(currPore->connection(i)->shape()->containsFluid(fluid));
                    
                    double cond(0.0), deltaG(0.0);
                    RockElem *connPore = currPore->nextPore(i, cond, deltaG, fluid, resistivitySolve);
                    
                    int nextPoreIdx(connPore->latticeIndex());
                    bool poreHasFluid(connPore->shape()->containsFluid(fluid));

                    cout << throatIdx << "(" << throatHasFluid << ") - "
                        << nextPoreIdx << "(" << poreHasFluid << "): g = " << cond << endl;
                }
                exit(-1);
            }                                               
            ++row;
        }
    }
    
    m_nonZeroElems = nonZeroIdx;                        // The size of the sparse matrix as well as 
    m_matrixSize = row;                                 // the number of non-zero elements are now known
    m_rowPtr[row] = nonZeroIdx+1;                       // Last element in row pointer should point to index 
    m_finalRowPointer = nonZeroIdx+1;                   // that is 1 past matrix element size.
    for(int i = 0; i < m_nonZeroElems; ++i)             // We retrieve the correct column
    {                                                   // index from the array containing
        int origPoreIdx = m_colIndex[i];                // the row indicies. Also note that 
        m_colIndex[i] = m_colHash[origPoreIdx-1]+1;     // pore indicies have a offset of 1
    }
}

