/****************************************************************/
/*      Copyright (c) 1998 Dept. of Materials, ICSTM            */
/*      All Rights Reserved                                     */
/*      THIS IS UNPUBLISHED PROPRIETARY SOURCE CODE OF ICSTM    */
/*      The copyright notice above does not evidence any        */
/*      actual or intended publication of such source code,     */
/*      and is an unpublished work by Dept. of Materials, ICSTM.*/
/*      This material contains CONFIDENTIAL INFORMATION that    */
/*      is the property of Imperial College. Any use,           */
/*      duplication or disclosure not specifically authorized   */
/*      by Imperial College is strictly prohibited.             */
/****************************************************************/
/* This code is part of the umats routines developed at in the  */
/* Materials Processing Group, Dept. of Materials, ICSTM.       */
/*      email p.d.lee or r.atwood @ic.ac.uk for details         */
/****************************************************************/

/****************************************************************/
/*      initface.c:                                             */
/* This subroutine calculates a list of offsets to step         */
/* a cell counter through the cells on a face of a cube         */
/* padded by one thickness of cells , and the offset necessary  */
/* to copy the  value to a cell within the padding, either      */
/* adjacent (pad offset) or opposite (wrap offset)              */
/****************************************************************/
/* Written by Peter D. Lee & Robert C. Atwood, Imperial College */
/* Aug 25, 1998                                                 */
/****************************************************************/
/*      MODIFIED by:                                            */
/*  PDL: Aug 16, 1998                                           */
/*  RCA: Nov 9, 1998  -- fixed bugs in lookup table             */
/****************************************************************/
/****** To Do List **********************************************/
/****************************************************************/
/* 1) add stuff for copyback, and general size...               */
/****************************************************************/
/*RCS Id:$Id: initface.c 887 2006-03-01 18:21:01Z rcatwood $*/
/*
RCS Log:$Log$
RCS Log:Revision 11.1  2006/03/01 18:20:39  rcatwood
RCS Log:Merging polycomponent and gas with meltback
RCS Log:
RCS Log:Revision 10.3  2005/12/01 14:38:01  rcatwood
RCS Log:Merged xly_05 changes into the main trunk
RCS Log:Primarily involving melt-back
RCS Log:
RCS Log:Revision 10.1.2.2  2005/11/23 18:18:53  rcatwood
RCS Log:Result of merging mould_source and xly meltback+curvature 2d versions
RCS Log:
RCS Log:Revision 10.1  2005/11/03 11:56:46  rcatwood
RCS Log:New version number -- using mould_src as base
RCS Log:
RCS Log:Revision 8.1.14.2  2005/11/02 11:55:05  rcatwood
RCS Log:Fixing up the revision nubmer after loss of repository
RCS Log:
RCS Log:Revision 9.1  2003/08/14 14:38:35  rcatwood
RCS Log:Working merge with decentered/porosity/procast, also including
RCS Log:Ali Chirazi's multicomponent (not tested in this version)
RCS Log:
RCS Log:Revision 8.1.8.2  2003/01/15 19:02:00  rcatwood
RCS Log:*** empty log message ***
RCS Log:
RCS Log:Revision 8.1.6.1  2002/11/06 17:27:46  rcatwood
RCS Log:NOT WORKING check-in of first stage merge with ca_procast
RCS Log:
RCS Log:Revision 7.0.8.1  2002/08/27 14:18:17  chirazi
RCS Log:adding files for multi-component-Procast version of CA
RCS Log:
RCS Log:Revision 7.0  2000/11/07 15:53:28  rcatwood
RCS Log:Multi Cell Pores added
RCS Log:
RCS Log:Revision 6.0  2000/09/25 18:03:35  rcatwood
RCS Log:After PORE_00 and NLM
RCS Log:
RCS Log:Revision 2.0  2000/08/02 10:21:56  rcatwood
RCS Log:Version used for pore paper runs
RCS Log:
RCS Log:Revision 1.1  2000/05/22 12:29:24  rcatwood
RCS Log:Fixed fs finish. Casolid to C from  W file. Global option
RCS Log:
RCS Log:Revision 5.1  2000/03/02 16:11:10  rcatwood
RCS Log:Merged xxu and rca versions
RCS Log:
RCS Log:Revision 5.0.2.1  2000/03/01 15:54:30  rcatwood
RCS Log:merged VAR and Multiblock updates. Not tested
RCS Log:
RCS Log:Revision 5.0.1.1  2000/02/22 19:04:27  rcatwood
RCS Log:Not yet tested
RCS Log:
RCS Log:Revision 4.2  1999/12/23 18:12:24  rcatwood
RCS Log:Version used for Mcwasp runs
RCS Log:
RCS Log:Revision 4.1  1999/12/16 13:33:44  rcatwood
RCS Log:Finalised improved use of RCS in all files.
RCS Log:
RCS Log:Revision 4.0.2.2  1999/12/16 12:31:32  rcatwood
RCS Log:Improving rcs id for all files, this may require several checkins to test.
RCS Log:
*/
#include <stdio.h>
#include "machine.h"
#include "cube.h"

/******************************************************/
/*  quick subroutine to dump out the lookup table     */
/******************************************************/
int dump_facecode (Frame * cubeptr)
{
  int i, j;
  FILE *facecodefile;
  char facename[MAX_STRING_LEN];

  sprintf (facename, "facecode.txt");
  facecodefile = fopen (facename, "w");

  fprintf (facecodefile, "\n");

  for (j = 0; j < 9; ++j) {
    fprintf (facecodefile, "\t%5i ", j);
  }
  fprintf (facecodefile, "\n");

  for (i = 0; i < 6; ++i) {
    fprintf (facecodefile, "%3i: ", i);
    for (j = 0; j < 10; ++j) {
      fprintf (facecodefile, "\t%5i ", cubeptr->flist[i][j]);
    }
    fprintf (facecodefile, "\n");
  }
  fclose (facecodefile);
  return (0);
}                               /* end of dump_facecode */

/***************************************************************/
/* This subroutine calculates a list of offsets to step        */
/* a cell counter through the cells on a face of a cube        */
/* padded by one thickness of cells , and the offset necessary */
/* to copy the  value to a cell within the padding, either     */
/* adjacent (pad offset) or opposite (wrap offset)             */
/***************************************************************/
int init_facecode (Frame * cubeptr, int *ins, int dim)
{
/******************************************************/
/* local variables                                    */
/******************************************************/

                           /**********************************/
  int fcode;                    /* the face code 0=negx etc       */
  int i, j, ii, ndim, loop1;    /* loop counters                  */
  int nums[5], inums[5];        /* temporarily storecommon values */
  int row, level, irow, ilevel; /* precalculate internal row,level */
  int facelist[6][10];          /* temporarily store the facelist */
  int outs[3] = { 0, 0, 0 };    /* calculate dim of inside cube   */
                           /**********************************/

  fprintf (stderr, "Entering init_facecode...\n");
  fprintf (stderr, "ins: %d, %d, %d; dim: %d\n", ins[0], ins[1], ins[2], dim);
/******************************************************/
/* Set stuff to zero                                  */
/******************************************************/
  for (i = 0; i < 6; ++i) {
    for (j = 0; j < 9; ++j) {
      facelist[i][j] = 0;
    }
  }

/******************************************************/
/* Precalculate the dimensions of inside cube         */
/******************************************************/
  ndim = dim;

  for (i = 0; i < ndim; ++i) {
    if (ins[i] < 1) {
      printf ("ERROR: block < one cell thick!! - %i %i\n", ins[i], i);
      return (1);
    }
    outs[i] = ins[i] + 2;
  }

/******************************************************/
/* Precalculate values that get used over and over    */
/******************************************************/
  row = outs[XAXIS];
  irow = ins[XAXIS];

  switch (ndim) {
  case THREE_D:
    ilevel = ins[XAXIS] * ins[YAXIS];
    level = outs[XAXIS] * outs[YAXIS];
    break;
  case TWO_D_CA:
    level = 0;
    break;
  default:
    fprintf (stderr, "ERROR: Hyperspatial solid'n zone... %iD!\n", ndim);
    break;
  }

  nums[0] = 1;
  nums[1] = row;
  nums[2] = level;
  nums[3] = level + row + 1;    /* lowest inside cell index */
  nums[4] = ins[ZAXIS] * level + ins[YAXIS] * row + ins[XAXIS]; /* highest index inner block */

  inums[0] = 1;
  inums[1] = irow;
  inums[2] = ilevel;
  inums[3] = 0;
  inums[4] = ins[ZAXIS] * ins[YAXIS] * ins[XAXIS] - 1;

/*************************************************************/
/* set up the start values for all faces, even=neg odd = pos */
/*************************************************************/
  for (i = 0; i < 2 * ndim; i += 2) {
    facelist[i][START] = nums[3];       /* all neg faces start at the same place */
    facelist[i][ISTART] = inums[3];     /* all neg faces start at the same place */
    facelist[i + 1][START] = nums[4];
    facelist[i + 1][ISTART] = inums[4];
  }

/******************************************************/
/* set up the offsets for looping and copying         */
/* 1=skip,2=jump,3=pad offset, 4=wrap offset          */
/*                                                    */
/* The axis code is given by   (i/2)%3  where i is the*/
/* facecode.                                          */
/******************************************************/
  loop1 = 2 * ndim;
  for (i = 0; i < loop1; i += 2) {
    for (j = 1; j < 4; ++j) {
      ii = (i / 2 + j) % 3;
      facelist[i][j] = nums[ii];
      facelist[i + 1][j] = nums[ii] * (-1);

      facelist[i][j + 5] = inums[ii];
      facelist[i + 1][j + 5] = inums[ii] * (-1);
    }
    facelist[i][4] = facelist[i][3] * ins[(i / 2) % 3] * -1;
    facelist[i + 1][4] = facelist[i][3] * ins[(i / 2) % 3];
  }
  facelist[0][NBOFF] = irow - 1;
  facelist[1][NBOFF] = -(irow - 1);
  facelist[2][NBOFF] = (ins[XAXIS] * (ins[YAXIS] - 1));
  facelist[3][NBOFF] = -(ins[XAXIS] * (ins[YAXIS] - 1));
  facelist[4][NBOFF] = (ins[XAXIS] * ins[YAXIS] * (ins[ZAXIS] - 1));
  facelist[5][NBOFF] = -(ins[XAXIS] * ins[YAXIS] * (ins[ZAXIS] - 1));
/******************************************************/
/* Fix up a little problem in 2D case ;-)             */
/******************************************************/
  if (ndim == TWO_D_CA) {
    facelist[2][1] = facelist[2][2];
    facelist[2][2] = 0;
    facelist[3][1] = facelist[3][2];
    facelist[3][2] = 0;
  }

/******************************************************/
/*Now store the facecodes  in the cube structure      */
/******************************************************/
  for (i = 0; i < 6; ++i) {
    for (j = 0; j < 10; ++j) {
      cubeptr->flist[i][j] = facelist[i][j];
    }
  }

/**********************************************************/
/* and equate the cube structure members with the arrays. */
/**********************************************************/
  for (i = 0; i < 3; ++i) {
    cubeptr->outs[i] = outs[i];
    cubeptr->ins[i] = ins[i];
  }
  cubeptr->ndim = ndim;
  cubeptr->ivalue = 0.0;
  fprintf (stderr, "Exiting init_facecode...\n");
  dump_facecode (cubeptr);
  return (0);
}                               /* end of init_facelist */

/* Little subroutine to get rcs id into the object code */
/* so you can use ident on the compiled program  */
/* also you can call this to print out or include the rcs id in a file*/
char const *rcs_id_initface_c ()
{
  static char const rcsid[] = "$Id: initface.c 887 2006-03-01 18:21:01Z rcatwood $";

  return (rcsid);
}

/* end of rcs_id_initface_c subroutine */
