/* #define DEBUG */
/****************************************************************/
/*      Copyright (c) 1998 Dept. of Materials, ICSTM            */
/*      All Rights Reserved                                     */
/*      THIS IS UNPUBLISHED PROPRIETARY SOURCE CODE OF ICSTM    */
/*      The copyright notice above does not evidence any        */
/*      actual or intended publication of such source code,     */
/*      and is an unpublished work by Dept. of Materials, ICSTM.*/
/*      This material contains CONFIDENTIAL INFORMATION that    */
/*      is the property of Imperial College. Any use,           */
/*      duplication or disclosure not specifically authorized   */
/*      by Imperial College is strictly prohibited.             */
/****************************************************************/
/* This code is part of the umats routines developed at in the  */
/* Materials Processing Group, Dept. of Materials, ICSTM.       */
/*	email p.d.lee or r.atwood @ic.ac.uk for details         */
/****************************************************************/

/****************************************************************/
/* READGEOPLUS.C:   (Part of CA code)                           */
/* Subroutine to read the geometry and initial conditions.      */
/* The file is formated, using the style:                       */
/*    # as first character:     Comment                         */
/* and values are input in the format:                          */
/*    command value  #comments                                  */
/****************************************************************/
/****************************************************************/
/* Written by Peter D. Lee & Robert C. Atwood, Imperial College */
/* Jul 1, 1998                                                  */
/* Modified:                                                    */
/* dec 7, 1998 RCA -- added thermocopule trace option           */
/****************************************************************/
/*RCS Id:$Id: readgeoplus.c 1490 2010-07-29 22:56:53Z  $*/

#include <stdio.h>
#include <string.h>
#include <math.h>
#include "machine.h"
#include "readgeoplus.h"
#include "blocks.h"

void set_geo_defaults (BB_struct * bp)
{
   /*********************************************************/
  /* Set all the default values...                         */
   /*********************************************************/
  /* from blocks.h */
  int i;

  if (bp->ctrl->procast == FALSE) {
    for (i = 0; i < 3; i++) {
      bp->nsb[i] = D_NSB;
      bp->nc[i] = D_NC_SB;
      bp->size_bb[i] = D_SIZE_BB;
    }
    bp->finish_time = D_FINISH_TIME;
  }
  bp->delt = D_DELT;
  bp->fsgrow = D_FS_GROW;
  bp->rev_fsgrow = 1/ D_FS_GROW ;  /* control the capture ... lyuan*/
  bp->window_velo = D_WINDOW_VELO;      /*by Wei WANG 11-07-02 */
  bp->window_disp = D_WINDOW_DISP;      /*by Wei WANG 11-07-02 */
  bp->iso_coef1 = D_ISO_COEF1;  /*curved isotherm*dong */
  bp->iso_coef2 = D_ISO_COEF2;  /*curved isotherm dong */
  bp->velo_coef = D_VELO_COEF;  /* Varying V */
  bp->grad_coef = D_GRAD_COEF;  /*varying G */
  bp->slope_coef = 0.0;
  bp->window_steps = 0 ;
  bp->coef_iso2 = 0;
  /* below are things for solo and testing modes...    */
  bp->Tinit = D_TINIT;
  bp->dim = THREE_D;
  /* details for FIDAP mode */
  bp->Cbdy_alloy = 0;
  bp->Cbdy_gas = 0;
  bp->tonecyc = 0;

#ifdef JUNK
  bp->fg->tring_on = FALSE;
  bp->fg->ntring = 0;

#endif
  bp->auto_fin = FALSE;
  /* below are things for bc's ...    */
  for (i = 0; i < 6; i++)
    bp->cubeptr.facectrl[i] = 1;        /* default is wrap */
  for (i = 0; i < 3; i++)
    bp->nzones[i] = D_NZONES;
}

int read_geoplus_vals (Ctrl_str * cp, BB_struct * bp)
{
  char line[MAX_STRING_LEN];
  char *token;
  int i, error;
  CA_FLOAT cellsize;
  double a, b, c, d;
  float e;
  int rflag = 0;
  extern CA_FLOAT global_pressure;

   /*********************************************************/
  /* Open the file                                         */
   /*********************************************************/
  if ((cp->fd_geo = fopen (cp->fn_geo, "r")) == NULL) {
    fprintf (stderr, "Error: can't open input file [%s]\n", cp->fn_geo);
    exit (0);
  }

  while (fgets (line, MAX_STRING_LEN, cp->fd_geo) != NULL) {

    /* ignore comment and blank lines */
    if (line[0] == '%' || line[0] == '#' || (token = strtok (line, " ,;\t")) == NULL) {
      continue;

      /*********************************************************/
      /* All values in the bigblock structure                  */
      /*********************************************************/
      /* NSubblocks int int int */
    } else if (strcasecmp (token, "NSubblocks") == 0) {
      for (i = 0; i < 3; i++) {
        if ((token = strtok (NULL, " ,;\t")) != NULL)
          bp->nsb[i] = atoi (token);
      }
      /* NCellsPerSB int int int */
    } else if (strcasecmp (token, "NCellsPerSB") == 0) {
      for (i = 0; i < 3; i++) {
        if ((token = strtok (NULL, " ,;\t")) != NULL)
          bp->nc[i] = atoi (token);
          if(cp->extinput == 1) bp->nc[i] = bp->ext_nc[i];
      }


      if(bp->nc[2]>1 && bp->ctrl->physica && bp->phy_tran ==0) bp->phy223d = 1;
          else bp->phy223d = 0;
		  
#ifdef ORIGIN
      /* BigBlockOrigin CA_FLOAT CA_FLOAT CA_FLOAT */
      
    } else if (strcasecmp (token, "BigBlockOrigin") == 0) {
      origin_flag = 1;
      for (i = 0; i < 3; i++) {
        if ((token = strtok (NULL, " ,;\t")) != NULL)
          bp->orig_bb[i] = atof (token);
      }
#endif /*ORIGIN*/
#ifdef BB_SIZE
        /* BigBlockSize CA_FLOAT CA_FLOAT CA_FLOAT */
    } else if (strcasecmp (token, "BigBlockSize") == 0) {
      for (i = 0; i < 3; i++) {
        if ((token = strtok (NULL, " ,;\t")) != NULL)
          bp->size_bb[i] = atof (token);
      }
#else /*input cell size */
    } else if (strcasecmp (token, "CellSize") == 0) {
      if ((token = strtok (NULL, " ,;\t")) != NULL)
        cellsize = atof (token);
        if(cp->extinput == 1) cellsize = bp->ext_cellsize;

      for (i = 0; i < 3; i++) {
        bp->size_bb[i] = cellsize * bp->nc[i] * bp->nsb[i];
      }
#endif /*BB or Cell size */
      /* AutoFinishTime */
    } else if (strcasecmp (token, "Nzones") == 0) {
      for (i = 0; i < 3; i++) {
        if ((token = strtok (NULL, " ,;\t")) != NULL)
          bp->nzones[i] = atof (token);
      }
    } else if (strcasecmp (token, "AutoFinishTime") == 0) {
      bp->auto_fin = TRUE;
      /* FinishTime CA_FLOAT */
    } else if (strcasecmp (token, "FinishTime") == 0) {
      if ((token = strtok (NULL, " ,;\t")) != NULL)
        bp->finish_time = atof (token);
      /* TimeStep CA_FLOAT */
    } else if (strcasecmp (token, "TimeStep") == 0) {
      if ((token = strtok (NULL, " ,;\t")) != NULL)
        bp->delt = atof (token);
      /* InitialTemperature CA_FLOAT */
      /**  \todo  Check initial temperature is below liquidus  -- general -- easy */
    } else if (strcasecmp (token, "InitialTemperature") == 0) {
      if ((token = strtok (NULL, " ,;\t")) != NULL)
        bp->Tinit = atof (token);
      /* InitialPressure CA_FLOAT */
    } else if (strcasecmp (token, "InitialPressure") == 0) {
      if ((token = strtok (NULL, " ,;\t")) != NULL)
        global_pressure = atof (token);
      /* FSGrow CA_FLOAT */
    } else if (strcasecmp (token, "FSGrow") == 0) {
      if ((token = strtok (NULL, " ,;\t")) != NULL)
        bp->fsgrow = atof (token);
        bp->rev_fsgrow = 1. / bp->fsgrow ; 
      /* NumDimensions */
    } else if (strcasecmp (token, "NumDimensions") == 0) {
      if ((token = strtok (NULL, " ,;\t")) != NULL)
        bp->dim = atoi (token);

      /* GradSlope */
    } else if (strcasecmp (token, "GradSlope") == 0) {
      if ((token = strtok (NULL, " ,;\t")) != NULL)
        bp->grad_slope = atof (token);
      /* slope variation*/
    } else if (strcasecmp (token, "Slope_Coef") == 0) {
      if ((token = strtok (NULL, " ,;\t")) != NULL)
	bp->slope_coef = atof (token);
      /* Gradient */
    } else if (strcasecmp (token, "Gradient") == 0) {
      if ((token = strtok (NULL, " ,;\t")) != NULL)
        bp->gradient = atof (token);

      /*  Curved isothermal coef1   */
    } else if (strcasecmp (token, "Iso_Coef_One") == 0) {
      if ((token = strtok (NULL, " ,;\t")) != NULL)
        bp->iso_coef1 = atof (token);

      /* Curved isothermal coef2  */
    } else if (strcasecmp (token, "Iso_Coef_Two") == 0) {
      if ((token = strtok (NULL, " ,;\t")) != NULL)
        bp->iso_coef2 = atof (token);

    } else if (strcasecmp (token, "Coef_Iso_Two") == 0) {
      if ((token = strtok (NULL, " ,;\t")) != NULL)
        bp->coef_iso2 = atof (token);
		  
      /* Velocity for directional (non-fidap) */

    } else if (strcasecmp (token, "Velocity") == 0) {
      if ((token = strtok (NULL, " ,;\t")) != NULL)
        bp->velocity = atof (token);

      /* variables for melt back xly 20041018 
       *     recalled by LY 10/2008*/
      /* time held for constant initial velocity */

    } else if (strcasecmp (token, "Time_Velo1") == 0) {
      if ((token = strtok (NULL, " ,;\t")) != NULL)
        bp->time_velo1 = atof (token);
      /* time stop decreasing velocity */
    } else if (strcasecmp (token, "Time_Velo2") == 0) {
      if ((token = strtok (NULL, " ,;\t")) != NULL)
        bp->time_velo2 = atof (token);
      /* time stop increasing velocity */
    } else if (strcasecmp (token, "Time_Velo3") == 0) {
      if ((token = strtok (NULL, " ,;\t")) != NULL)
        bp->time_velo3 = atof (token);
      /* time stop increasing velocity */
    } else if (strcasecmp (token, "Time_Velo4") == 0) {
      if ((token = strtok (NULL, " ,;\t")) != NULL)
        bp->time_velo4 = atof (token);
      /* time held of zero velocity */
    } else if (strcasecmp (token, "Time_Hold") == 0) {
      if ((token = strtok (NULL, " ,;\t")) != NULL)
        bp->time_hold = atof (token);
      /* velocity decrease per second, negative value */
    } else if (strcasecmp (token, "Velo_Coef1") == 0) {
      if ((token = strtok (NULL, " ,;\t")) != NULL)
        bp->velo_coef1 = atof (token);
      /* velocity increase per second, positive value */
    } else if (strcasecmp (token, "Velo_Coef2") == 0) {
      if ((token = strtok (NULL, " ,;\t")) != NULL)
        bp->velo_coef2 = atof (token);
    } else if (strcasecmp (token, "Velo_Coef3") == 0) {
      if ((token = strtok (NULL, " ,;\t")) != NULL)
        bp->velo_coef3 = atof (token);
    } else if (strcasecmp (token, "Velo_Coef4") == 0) {
      if ((token = strtok (NULL, " ,;\t")) != NULL)
        bp->velo_coef4 = atof (token);
  
      /* Velocity variation (direct Sol) */
    } else if (strcasecmp (token, "Velo_Coef") == 0) {
      if ((token = strtok (NULL, " ,;\t")) != NULL)
        bp->velo_coef = atof (token);
      /*Gradient Coefficient */
    } else if (strcasecmp (token, "Grad_Coef") == 0) {
      if ((token = strtok (NULL, " ,;\t")) != NULL)
        bp->grad_coef = atof (token);
      /*temperature curvature */
    } else if (strcasecmp (token, "Time_Curv") == 0) {
      if ((token = strtok (NULL, " ,;\t")) != NULL)
        bp->time_curv = atof (token);
      /*temperature transient time */
    } else if (strcasecmp (token, "Time_Tran") == 0) {
      if ((token = strtok (NULL, " ,;\t")) != NULL)
        bp->time_tran = atof (token);
      /*cell number for curved isotherm */
    } else if (strcasecmp (token, "Cell_No") == 0) {
      if ((token = strtok (NULL, " ,;\t")) != NULL)
        bp->cell_no = atof (token);
      /*trsnsient coef. for curved isotherm */
    } else if (strcasecmp (token, "Therm_Coef") == 0) {
      if ((token = strtok (NULL, " ,;\t")) != NULL)
        bp->therm_coef = atof (token);
      /* OctFactor */
    } else if (strcasecmp (token, "OctFactor") == 0) {
      if ((token = strtok (NULL, " ,;\t")) != NULL)
        bp->oct_factor = atof (token);

      /* Window_Velo CA_FLOAT *//* by Wei WANG 11-07-02 */
    } else if (strcasecmp (token, "Window_Velo") == 0) {
      if ((token = strtok (NULL, " ,;\t")) != NULL)
        bp->window_velo = atof (token);

      /* FaceCtrl bot(int) bot(int) bot(int) bot(int) bot(int) bot(int)  */
    } else if (strcasecmp (token, "FaceCtrl") == 0) {
      for (i = 0; i < 6; i++) {
        if ((token = strtok (NULL, " ,;\t")) != NULL)
          bp->cubeptr.facectrl[i] = atof (token);
      }
      /* boundary concentraiont for FIX_BDY condition  */
    } else if (strcasecmp (token, "Cbdy_alloy") == 0) {
      if ((token = strtok (NULL, " ,;\t")) != NULL)
        bp->Cbdy_alloy = atof (token);
      /* boundary concentraiont for FIX_BDY condition  */
    } else if (strcasecmp (token, "Cbdy_gas") == 0) {
      if ((token = strtok (NULL, " ,;\t")) != NULL)
        bp->Cbdy_gas = atof (token);

      /* by LY for velocity BCs*/
    } else if (strcasecmp (token, "Cycle_time") == 0) {
      if ((token = strtok (NULL, " ,;\t")) != NULL)
        bp->tonecyc = atof (token);
		  
      /* velocity bc settings */
    } else {
      fprintf (stderr, "Warning: Unknown command: %s.\n", token);
      rflag++;
    }
  }                             /* end while */

   /*********************************************************/
  /* Calculate subsiduary values...                        */
   /*********************************************************/
  if (bp->tc.on)
    bp->ctrl->coolrate = FALSE;

  if (bp->ctrl->physica || bp->ctrl->ffheattran) bp->ctrl->coolrate = FALSE;
  
  bp->ntsb = bp->nsb[0] * bp->nsb[1] * bp->nsb[2];
  if (bp->ntsb <= 0) {
    fprintf (stderr, "ERROR:read_geoplus_vals: bad values for subbblock numbers %i %i %i \n", bp->nsb[0], bp->nsb[1], bp->nsb[2]);
    fprintf (stderr, "exiting... \n");
    exit (0);
  }
  bp->ncsb = bp->nc[0] * bp->nc[1] * bp->nc[2];
  if (bp->ncsb <= 0) {
    fprintf (stderr, "ERROR:read_geoplus_vals: bad values for subblock size %i %i %i \n", bp->nc[0], bp->nc[1], bp->nc[2]);
    fprintf (stderr, "exiting... \n");
    exit (0);
  }
#ifdef ORIGIN
  for (i = 0; i < 3; i++) {
    if (!origin_flag) {
      bp->orig_bb[i] = 0.0;     /* set origin to 0,0,0 as default */
    }
    bp->tnc[i] = bp->nc[i] * bp->nsb[i];
    bp->size_c[i] = bp->size_bb[i] / ((CA_FLOAT) bp->tnc[i]);
  }
#else /*not ORIGIN */
  for (i = 0; i < 3; i++) {
    bp->orig_bb[i] = 0.0;       /* set origin to 0,0,0 as it does not matter */
    bp->tnc[i] = bp->nc[i] * bp->nsb[i];
    bp->size_c[i] = bp->size_bb[i] / ((CA_FLOAT) bp->tnc[i]);
  }
#endif /*ORIGIN*/
    bp->total_cell_number = bp->tnc[0] * bp->tnc[1] * bp->tnc[2];
  a = (double) bp->size_c[0];
  b = (double) bp->size_c[1];
  c = (double) bp->size_c[2];
  d = a * b * c;
  e = (float) d;
  fprintf (stderr, "x,y,z,volume,single-precision-volume, %.10g,%.10g,%.10g,%.10g,%.10g\n", a, b, c, d, e);
  d = (double) ((double) (bp->size_c[0]) * (double) (bp->size_c[1]) * (double) (bp->size_c[2]));
/*   bp->vol_c = (double)((double)(bp->size_c[0]) *  (double)(bp->size_c[1]) * (double)(bp->size_c[2]));*/
  bp->vol_c = bp->size_c[0] * bp->size_c[1] * bp->size_c[2];
  bp->yinv = 1 / ((CA_FLOAT) (bp->nc[1]));

  fprintf (stderr, "total #sb: %d, # cells/sb: %d\n", bp->ntsb, bp->ncsb);
  fprintf (stderr, "#cells in x,y,z: %d, %d, %d\n", bp->nc[0], bp->nc[1], bp->nc[2]);
  fprintf (stderr, "bigblock size: %f, %f, %f,", bp->size_bb[0], bp->size_bb[1], bp->size_bb[2]);
  fprintf (stderr, "cell vol:, %.10g\n", bp->vol_c);

#ifdef DEBUG_READGEO
  /* from blocks.h */
  fprintf (stderr, "NSubblocks %d %d %d\n", bp->nsb[0], bp->nsb[1], bp->nsb[2]);
  fprintf (stderr, "NCellsPerSB %d %d %d\n", bp->nc[0], bp->nc[1], bp->nc[2]);
  fprintf (stderr, "BigBlockSize %f %f %f\n", bp->size_bb[0], bp->size_bb[1], bp->size_bb[2]);
  fprintf (stderr, "FinishTime %f\n", bp->finish_time);
  fprintf (stderr, "TimeStep %f\n", bp->delt);
  fprintf (stderr, "InitialTemperature %f\n", bp->Tinit);
#endif /* DEBUG_READGEO */

  fclose (cp->fd_geo);
  fprintf (stderr, "Exiting read_geoplus().\n");
  return rflag;                 /* return the number of errors whilst reading file */
}                               /* end of subroutine read_geoplus */

int read_geoplus (Ctrl_str * cp, BB_struct * bp)
{
  int retval = 0;

  set_geo_defaults (bp);
  retval = read_geoplus_vals (cp, bp);
  return (retval);
}

/* Little subroutine to get rcs id into the object code */
/* so you can use ident on the compiled program  */
/* also you can call this to print out or include the rcs id in a file*/
char const *rcs_id_readgeoplus_c ()
{
  static char const rcsid[] = "$Id: readgeoplus.c 1490 2010-07-29 22:56:53Z  $";

  return (rcsid);
}

/* end of rcs_id_readgeoplus_c subroutine */
/*
RCS Log:$Log$
RCS Log:Revision 11.2  2006/06/07 19:37:11  rcatwood
RCS Log:Fixed bug where pore is written when not selected
RCS Log:
RCS Log:Revision 11.1  2006/03/01 18:20:40  rcatwood
RCS Log:Merging polycomponent and gas with meltback
RCS Log:
RCS Log:Revision 10.5  2005/12/06 13:09:54  rcatwood
RCS Log:Changed todo lists to Doxygen syntax
RCS Log:
RCS Log:Revision 10.4  2005/12/06 12:58:01  rcatwood
RCS Log:Improved the to-do list information
RCS Log:
RCS Log:Revision 10.3  2005/12/01 14:38:01  rcatwood
RCS Log:Merged xly_05 changes into the main trunk
RCS Log:Primarily involving melt-back
RCS Log:
RCS Log:Revision 10.1.2.2  2005/11/23 18:18:53  rcatwood
RCS Log:Result of merging mould_source and xly meltback+curvature 2d versions
RCS Log:
RCS Log:Revision 10.1.2.1  2005/11/07 17:47:56  rcatwood
RCS Log:Branch uisng Xiao Li Yang final version
RCS Log:
RCS Log:Revision 9.3.12.4  2005/01/18 11:09:09  xly
RCS Log:*** empty log message ***
RCS Log:Revision 10.1  2005/11/03 11:56:47  rcatwood
RCS Log:New version number -- using mould_src as base
RCS Log:
RCS Log:Revision 9.3.12.3  2004/08/06 18:56:01  xly
RCS Log:add coef for transient thermal curvature
RCS Log:
RCS Log:Revision 9.3.12.2  2004/08/05 09:51:18  xly
RCS Log:*** empty log message ***
RCS Log:
RCS Log:Revision 9.3.12.1  2004/08/02 13:09:58  xly
RCS Log:*** empty log message ***
RCS Log:Revision 8.5.8.2  2005/11/02 11:55:06  rcatwood
RCS Log:Fixing up the revision nubmer after loss of repository
RCS Log:
RCS Log:Revision 9.3  2003/11/17 18:33:14  hdong
RCS Log:
RCS Log:for varying the pulling velocity and thermal gradient
RCS Log:
RCS Log:Revision 9.2  2003/11/03 19:14:26  hdong
RCS Log:
RCS Log:log message
RCS Log:
RCS Log:Revision 9.1  2003/08/14 14:38:39  rcatwood
RCS Log:Working merge with decentered/porosity/procast, also including
RCS Log:Ali Chirazi's multicomponent (not tested in this version)
RCS Log:
RCS Log:Revision 8.5.2.13  2003/08/14 14:18:04  rcatwood
RCS Log:Working ca_procast new version, on linux
RCS Log:Added surface nucleation
RCS Log:Added mould source term
RCS Log:Changed printout headers
RCS Log:Temperature output image
RCS Log:
RCS Log:Revision 8.5.2.12  2003/05/19 18:55:17  rcatwood
RCS Log:Addded option to allow horizontal or vertical directional growth
RCS Log:and flux boundary condition
RCS Log:
RCS Log:Revision 8.5.2.11  2003/05/14 11:49:54  rcatwood
RCS Log:Fixed temperature input to conform wiht new temperature calculation methods
RCS Log:and control input file
RCS Log:
RCS Log:Revision 8.5.2.10  2003/04/08 11:44:35  rcatwood
RCS Log:after bug fix for directional gradient
RCS Log:
RCS Log:Revision 8.5.2.9  2003/04/08 09:51:29  rcatwood
RCS Log:fixed velocity read bug
RCS Log:
RCS Log:Revision 8.5.2.8  2003/03/14 16:41:47  hdong
RCS Log:Fixed temperature option selection for re-start.
RCS Log:Fixed output_excel for restart case
RCS Log:
RCS Log:Revision 8.5.2.7  2003/03/13 15:32:10  rcatwood
RCS Log:Added histogram zone input in geoplus file
RCS Log:
RCS Log:Revision 8.5.2.6  2003/03/13 14:15:57  rcatwood
RCS Log:Added multiple zones for grain size measurement (hard coded)
RCS Log:Added grain size counting in decentered (without diffusion)
RCS Log:
RCS Log:Revision 8.5.2.5  2003/02/05 13:15:21  rcatwood
RCS Log:Fixed some memory errors when list-file NOT used
RCS Log:Added ifdefs to get rid of sgi-specific image output routines
RCS Log:
RCS Log:Revision 8.5.2.4  2003/01/22 16:53:46  rcatwood
RCS Log:Almost working read_fg version
RCS Log:
RCS Log:Revision 8.5.2.3  2003/01/17 16:09:30  rcatwood
RCS Log:Before changing all CA_FLOAT to CA_CA_FLOAT
RCS Log:
RCS Log:Revision 8.5.2.2  2003/01/15 19:02:01  rcatwood
RCS Log:*** empty log message ***
RCS Log:
RCS Log:Revision 8.3.2.1  2002/11/06 17:27:48  rcatwood
RCS Log:NOT WORKING check-in of first stage merge with ca_procast
RCS Log:
RCS Log:Revision 8.3  2002/11/04 13:35:42  rcatwood
RCS Log:Finished restart re-read reorgainization
RCS Log:
RCS Log:Revision 8.2  2002/10/24 18:10:55  rcatwood
RCS Log:Incorporated NOT_CASTING mode with decentered octahedron
RCS Log:
RCS Log:Revision 8.1  2002/10/17 17:01:03  rcatwood
RCS Log:New version number! for decentered/porosity merge! Alpha Version!
RCS Log:
RCS Log:Revision 7.2  2002/10/17 16:52:38  rcatwood
RCS Log:Merge from branch: combined Robert (porosity) and Wei (decentered octahedron) versions
RCS Log:
RCS Log:Revision 7.1.14.2  2002/09/12 10:48:08  rcatwood
RCS Log:Changed pore routine to avoid getxyz routine
RCS Log:
RCS Log:Revision 7.1.14.1  2002/08/22 15:13:43  ww1
RCS Log:Merge of Wei Wang's decentered octahedron/square method
RCS Log:
RCS Log:Revision 7.1  2001/02/19 19:28:47  rcatwood
RCS Log:fixed histo
RCS Log:for grains
RCS Log:
RCS Log:and also make TcTrace mode override const. cooling rate
RCS Log:
RCS Log:Revision 7.0  2000/11/07 15:53:28  rcatwood
RCS Log:Multi Cell Pores added
RCS Log:
RCS Log:Revision 6.0  2000/09/25 18:03:34  rcatwood
RCS Log:After PORE_00 and NLM
RCS Log:
RCS Log:Revision 2.0  2000/08/02 10:21:56  rcatwood
RCS Log:Version used for pore paper runs
RCS Log:
RCS Log:Revision 1.2  2000/06/06 18:43:04  rcatwood
RCS Log:Changed r start and growth of pores -- pore radius equilibrium
RCS Log:
RCS Log:Revision 1.1  2000/05/22 12:29:24  rcatwood
RCS Log:Fixed fs finish. Casolid to C from  W file. Global option
RCS Log:
RCS Log:Revision 5.4  2000/05/04 18:36:45  rcatwood
RCS Log:Fixed fs finish. Casolid to C from  W file. Global option
RCS Log:
RCS Log:Revision 5.3  2000/03/06 16:51:01  rcatwood
RCS Log:added % comment
RCS Log:
RCS Log:Revision 5.2  2000/03/06 14:11:59  rcatwood
RCS Log:removed Cinit from readgeoplus, in matprops now!
RCS Log:
RCS Log:Revision 5.1  2000/03/02 16:11:10  rcatwood
RCS Log:Merged xxu and rca versions
RCS Log:
RCS Log:Revision 5.0.2.1  2000/03/01 15:54:30  rcatwood
RCS Log:merged VAR and Multiblock updates. Not tested
RCS Log:
RCS Log:Revision 1.2  2000/02/23 15:54:07  xxu
RCS Log:read one more flag for temperature interpolation
RCS Log:Revision 5.0.1.2  2000/02/29 18:00:25  rcatwood
RCS Log:Bug fixed when growing into new block
RCS Log:
RCS Log:Revision 5.0.1.1  2000/02/22 19:04:27  rcatwood
RCS Log:Not yet tested
RCS Log:
RCS Log:Revision 4.4  2000/01/13 16:59:03  rcatwood
RCS Log:Binary dump of grain and temperature,Quick temp. image.
RCS Log:
RCS Log:Revision 4.3  2000/01/10 14:57:00  rcatwood
RCS Log:Quick SIGTERM handler in ca_wrapper
RCS Log:
RCS Log:Revision 4.2  1999/12/23 18:12:24  rcatwood
RCS Log:Version used for Mcwasp runs
RCS Log:
RCS Log:Revision 4.1  1999/12/16 13:33:44  rcatwood
RCS Log:Finalised improved use of RCS in all files.
RCS Log:
RCS Log:Revision 4.0.2.2  1999/12/16 12:31:32  rcatwood
RCS Log:Improving rcs id for all files, this may require several checkins to test.
RCS Log:
*/
