/****************************************************************/
/*   Copyright (c) 1998 Dept. of Materials, ICSTM               */
/*   All Rights Reserved                                        */
/*   THIS IS UNPUBLISHED PROPRIETARY SOURCE CODE OF ICSTM       */
/*   The copyright notice above does not evidence any           */
/*   actual or intended publication of such source code,        */
/*   and is an unpublished work by Dept. of Materials, ICSTM.   */
/*   This material contains CONFIDENTIAL INFORMATION that       */
/*   is the property of Imperial College. Any use,              */
/*   duplication or disclosure not specifically authorized      */
/*   by Imperial College is strictly prohibited.                */
/****************************************************************/
/* This code is part of the umats routines developed at in the  */
/* Materials Processing Group, Dept. of Materials, ICSTM.       */
/*      email p.d.lee or r.atwood @ic.ac.uk for details         */
/****************************************************************/

/*RCS Id:$Id: sb_diffuse_gas_poly.c 932 2006-08-30 12:39:33Z rcatwood $*/
#include <stdio.h>
#include <math.h>
#include <string.h>
#include "blocks.h"
#include "machine.h"
#include "ca_matrix.h"
#include "sb_diffuse.h"
#include "props.h"
#include "mould_sources.h"

#define MAX_DIFF_WARN 10

#ifdef CHECK_GAS
extern CA_FLOAT checkgas (BB_struct * bp, int callflag);
#endif

int sb_diffuse_gas_poly (BB_struct * bp, int sbnum)
{
  int dumb = 0;
  int index_ca;
  int printflag, errflg = 0, fileflag = 0, errors = 0;
  int nx, ny, nz, tsteps;
  int *oni, *onip, *onend;
  SB_struct *sp;
  Ctrl_str *cp;
  int t, i, j, k, l;
  static int warn_msg = 0;
  CA_FLOAT new_sol, OldS, NewS, partcoef,r_eff, km;
  CA_FLOAT *c_temp_p;
  CA_FLOAT **sol_gas_values;
  CA_FLOAT *oldfs, *oldfs_start, *newfs, *newfs_start;
  CA_FLOAT old_alpha, new_alpha, alpha_ratio, conc;     /*readability variables */
  CA_FLOAT *np, *op, flux, nbconc, *nsol, *osol, dxsq, dt, nbsum;
  CA_FLOAT rmax, rmin, tmax, tmin;
  CA_FLOAT r, rs, rl, ds, rsrl, dl, cellds, celldl, dtx, dmax, dmin, d; /*solid, liquid */
  int iphs, iphs_tot;
  CA_FLOAT *newfs_poly[NPHAMAX], *oldfs_poly[NPHAMAX];
  CA_FLOAT nbfs[NPHAMAX],fs_av[NPHAMAX];

#ifdef CHECK_GAS
  static CA_FLOAT diffgas0 = 0, diffgas1 = 0, change = 0;

  diffgas0 = checkgas (bp, 3);
  change = diffgas0 - diffgas1;
  if (ABS (change) > 1e-5) {
    fprintf (stderr, "WARNING:%s gas change limit exceeded! %.5g\n",__func__, change);
  }
#endif

  cp = bp->ctrl;
  iphs_tot = cp->NUM_PHS;



/* set up local neighbourhood */
/* use 6cell only for now     */
  rmin = dmin = 1;
  rmax = dmax = 0;
  tmax = 0;
  tmin = 10000;
  oni = bp->nbhd.onq;           /*padded */
  onip = oni;
  onend = oni + 6;
/* set up local values and pointers */
  printflag = ((bp->step % bp->ctrl->scr_dmp_freq));

  sp = bp->sb[sbnum];
  bp->cubeptr.curr = sbnum;
  nx = bp->nc[0];
  ny = bp->nc[1];
  nz = bp->nc[2];
  op = osol = bp->ftmp_two;     /* gas solute */
  np = nsol = sp->c_sol;        /* gas solute */
  c_temp_p = sp->c_temp;        /* temperature of cell */
  sol_gas_values = bp->c_sol_values->block_array;

  if (bp->ctrl->scheil == TRUE) {
    newfs = newfs_start = bp->ftmp_three;
    oldfs = oldfs_start = sp->sch_fs;
  } else {
    newfs = newfs_start = bp->ftmp_one;
    oldfs = oldfs_start = sp->c_fs;
  }                             /*end of schiel test */


  for (iphs = 0; iphs < iphs_tot; iphs++) {
    newfs_poly[iphs]= bp->ftmp_one_poly[iphs];
    newfs_poly[iphs] += bp->cubeptr.flist[0][START];
    oldfs_poly[iphs] = sp->c_fs_poly[iphs];
  }

  tsteps = bp->ctrl->diffuse_step;
  dxsq = bp->size_c[0] * bp->size_c[0];
  r = 0;
  /*delta t over delta x squared */
  dtx = (bp->delt / (dxsq * tsteps));

/* get the padded frac solid array to check */

/* make a copy of the solute concentration array */

  bp->cubeptr.ivalue = bp->Cbdy_gas;
  errflg += fcopy_matrix (PAD, osol, nsol, bp, sol_gas_values, sbnum);  /* (flag, to, from, bp) */
  bp->cubeptr.ivalue = 0.0;

  /* add the source at the mould surface if selected */
  if (bp->mprops.gasprops.mould_src != 0) {
    sb_mould_src (bp, &(bp->mprops.gasprops), sbnum, osol);
  }
  /* rewind array pointers */
  np = nsol;
  op = osol + bp->cubeptr.flist[0][START];      /*rewind */
  newfs = newfs_start + bp->cubeptr.flist[0][START];
  oldfs = oldfs_start;
  
  /* get the initial temperature for comparison */
  tmin = *c_temp_p;

  /************************************************/
  /* now calculate the finite difference          */
  /************************************************/
  /* DIFFUSION LOOP                               */
  /************************************************/
  /* Run through all cells updating as needed.    */
  /************************************************/

  for (k = 0, index_ca = 0; (k < nz && index_ca < bp->tnc[0] * bp->tnc[1] * bp->tnc[2]); k++) { /* loop cells in z direction */
    for (j = 0; j < ny; j++) {  /* loop cells in y direction */
      for (i = 0; i < nx; i++) {        /* loop cells in x direction */

        /* update concentration for diffusion, using part. coeff */

        /* skip cells that are not in the casting */
        if (*oldfs != NOT_CASTING) {

          old_alpha = 1.0;
          new_alpha = 1.0;

          for (iphs = 0; iphs < iphs_tot; iphs++) {
            old_alpha -= (1-bp->mprops.gasprops.part_coef[iphs])**oldfs_poly[iphs];
            new_alpha -= (1-bp->mprops.gasprops.part_coef[iphs])**newfs_poly[iphs];
          }
          alpha_ratio = (old_alpha / new_alpha);

          nbsum = 0;
          conc = *op;
#ifdef CELL_DIFF_ARR
          celldl = get_dl (*c_temp_p, &(bp->mprops.gasprops));
          cellds = get_ds (*c_temp_p, &(bp->mprops.gasprops));
#else
          celldl = get_dl (*c_temp_p);
          cellds = get_ds (*c_temp_p);
#endif
          rs = cellds * dtx;
          rl = celldl * dtx;

          /* check for error/warning conditions */
          if ((rs > COURANT_LIMIT || rl > COURANT_LIMIT)) {
            if (warn_msg < MAX_DIFF_WARN) {
              fprintf (stderr, "SB_DIFFUSE_GAS: WARNING: courant values too high!, %1.2e,%1.2e\n", rs, rl);
            }
            if (warn_msg > WARN_EXIT_LIMIT) {
              fprintf (stderr, "ERROR: sb_diffuse_gas: Warning limit exceeded. Exiting. %i\n", warn_msg);
              exit (warn_msg);
            }

            warn_msg++;
          }
          /* end of warnign check */

          for (onip = oni; onip < onend; onip++) {      /* neighbour sum loop */

            for (iphs = 0; iphs < iphs_tot; iphs++) {
              nbfs[iphs] = *(newfs_poly[iphs] + *onip);

              /* skip nb cells that are not in the casting */
              if (nbfs[iphs] == NOT_CASTING)
                continue;

              /* averaged frac solid */
              fs_av[iphs] = 0.5 * (*newfs_poly[iphs] + nbfs[iphs]);
            }/*End loop on the solid phases */

            d = getav_d (celldl, cellds, fs_av[0]);
            /*diffusion in the second stoechiometric phase is not taken into account*/
            
            r = dtx * d;
            nbconc = *(op + *onip);
            nbsum += r * (nbconc - conc);
            /* get values to print for monitoring */
            if (printflag == 0) {
              if (d > dmax)
                dmax = d;
              if (d < dmin)
                dmin = d;
              if (r > rmax)
                rmax = r;
              if (r < rmin)
                rmin = r;
              if (*c_temp_p > tmax)
                tmax = *c_temp_p;
              if (*c_temp_p < tmin)
                tmin = *c_temp_p;
            }
          }                     /* end of neighbour sum loop */
          *np = alpha_ratio * *op + nbsum / new_alpha;
          #ifdef TRAP_BIG_GAS
          { 
              int dumb;
              if (*np > TRAP_BIG_GAS || -(*np) > TRAP_BIG_GAS){
                dumb = 1;
              }
          }
          #endif
        }
        /* end of NOT_CASTING test */
        c_temp_p++;
        np++;
        index_ca++;
        oldfs++;
        newfs++;


        for (iphs = 0; iphs < iphs_tot; iphs++) {
          oldfs_poly[iphs]++;
          newfs_poly[iphs]++;
        }

        op++;
      }                         /*x */
      op += 2;
      newfs+=2;

      for (iphs = 0; iphs < iphs_tot; iphs++) {
        newfs_poly[iphs] += 2;
      }
    }                           /*y */
    op += 2 * (nx + 2);
    newfs +=2 *( nx + 2 );

    for (iphs = 0; iphs < iphs_tot; iphs++) {
      newfs_poly[iphs] += 2 * (nx + 2);
    }
  }                             /*z */

  if (printflag == 0) {
    fprintf (stderr, "sb_diffuse_gas: dmax %.5g  dmin %.5g %i\n", dmax, dmin, bp->step);
    fprintf (stderr, "sb_diffuse_gas: rmax %.5g  rmin %.5g\n", rmax, rmin);
    fprintf (stderr, "sb_diffuse_gas: tmax %.5g  tmin %.5g\n", tmax, tmin);
  }
#ifdef CHECK_GAS
  diffgas1 = checkgas (bp, 4);
  change = diffgas1 - diffgas0;
  if (ABS (change) > 1e-5) {
    fprintf (stderr, "WARNING:%s gas change limit exceeded! %.5g\n",__func__, change);
  }
#endif

  return (errflg);
}                               /* end of sb_diffuse */

/* Little subroutine to get rcs id into the object code */
/* so you can use ident on the compiled program  */
/* also you can call this to print out or include the rcs id in a file*/
char const *rcs_id_sb_diffuse_gas_poly_c ()
{
  static char const rcsid[] = "$Id: sb_diffuse_gas_poly.c 932 2006-08-30 12:39:33Z rcatwood $";

  return (rcsid);
}

/* end of rcs_id_sb_diffuse_gas_c subroutine */

/*
RCS Log:$Log$
RCS Log:Revision 11.2  2006/08/30 12:39:27  rcatwood
RCS Log:Added a constant name step.blz block restart file output
RCS Log:
RCS Log:Revision 11.1  2006/03/01 18:20:40  rcatwood
RCS Log:Merging polycomponent and gas with meltback
RCS Log:
RCS Log:Revision 1.1.2.2  2006/02/20 14:51:23  rcatwood
RCS Log:Correced evil log message
RCS Log:
RCS Log:Revision 1.1.2.1  2006/02/20 14:10:25  lthuinet
RCS Log:sb_diffuse_gas_poly.c is an extension of sb_diffuse_gas.c to take into account diffusion of gas into secondary eutectic phase
RCS Log:
RCS Log:Revision 10.5.2.1  2006/02/01 14:36:48  rcatwood
RCS Log:solved all implicit function declarations
RCS Log:
RCS Log:Revision 10.5  2005/12/06 13:09:54  rcatwood
RCS Log:Changed todo lists to Doxygen syntax
RCS Log:
RCS Log:Revision 10.4  2005/12/06 12:58:01  rcatwood
RCS Log:Improved the to-do list information
RCS Log:
RCS Log:Revision 10.3  2005/12/01 14:38:02  rcatwood
RCS Log:Merged xly_05 changes into the main trunk
RCS Log:Primarily involving melt-back
RCS Log:
RCS Log:Revision 10.1.2.2  2005/11/23 18:18:53  rcatwood
RCS Log:Result of merging mould_source and xly meltback+curvature 2d versions
RCS Log:
RCS Log:Revision 10.1  2005/11/03 11:56:47  rcatwood
RCS Log:New version number -- using mould_src as base
RCS Log:
RCS Log:Revision 8.1.14.2  2005/11/02 11:55:06  rcatwood
RCS Log:Fixing up the revision nubmer after loss of repository
RCS Log:
RCS Log:Revision 9.1.4.2  2004/05/27 11:37:49  rcatwood
RCS Log:Bug in sb_diffuse_alloy: referring to old mould_src flag
RCS Log:
RCS Log:Revision 9.1.4.1  2004/03/04 11:29:25  rcatwood
RCS Log:*** empty log message ***
RCS Log:
RCS Log:Revision 9.1  2003/08/14 14:38:40  rcatwood
RCS Log:Working merge with decentered/porosity/procast, also including
RCS Log:Ali Chirazi's multicomponent (not tested in this version)
RCS Log:
RCS Log:Revision 8.1.8.5  2003/05/19 18:55:17  rcatwood
RCS Log:Addded option to allow horizontal or vertical directional growth
RCS Log:and flux boundary condition
RCS Log:
RCS Log:Revision 8.1.8.4  2003/02/27 23:04:39  rcatwood
RCS Log:Removed use of old temperature routines , all temperatures shoudl
RCS Log:be determined by checking the array c_temp in teh subblock, if the
RCS Log:subblock is open
RCS Log:
RCS Log:Revision 8.1.8.3  2003/01/22 16:53:46  rcatwood
RCS Log:Almost working read_fg version
RCS Log:
RCS Log:Revision 8.1.8.2  2003/01/15 19:02:02  rcatwood
RCS Log:*** empty log message ***
RCS Log:
RCS Log:Revision 8.1.6.1  2002/11/06 17:27:48  rcatwood
RCS Log:NOT WORKING check-in of first stage merge with ca_procast
RCS Log:
RCS Log:Revision 8.1  2002/10/17 17:01:03  rcatwood
RCS Log:New version number! for decentered/porosity merge! Alpha Version!
RCS Log:
RCS Log:Revision 7.4  2002/10/17 16:52:38  rcatwood
RCS Log:Merge from branch: combined Robert (porosity) and Wei (decentered octahedron) versions
RCS Log:
RCS Log:Revision 7.3.10.3  2002/10/17 16:03:44  rcatwood
RCS Log:Fixed a subtle problem concerning the amount of gas in the pores
RCS Log:This required adding a lot of debugging #ifdef CHECK_GAS lines
RCS Log:
RCS Log:Revision 7.3.10.2  2002/09/27 14:38:46  rcatwood
RCS Log:Added write block structure values routines
RCS Log:Modified matprops to allow T and C for eutectic to b read in
RCS Log:Modified props to use these values
RCS Log:Modified Makefile to make the obj files in a subdirectory
RCS Log:
RCS Log:Revision 7.3.10.1  2002/08/27 12:36:36  rcatwood
RCS Log:Removed some old debug ifdef's
RCS Log:Organised header files for calc_sb
RCS Log:Improved comments
RCS Log:Fixed format warnings
RCS Log:
RCS Log:Revision 7.3  2002/01/28 14:00:01  rcatwood
RCS Log:After thesis
RCS Log:
RCS Log:Revision 7.2  2001/03/23 18:56:08  rcatwood
RCS Log:added choice list and minimum H to pore routines.
RCS Log:
RCS Log:Revision 7.1  2001/03/05 11:40:37  rcatwood
RCS Log:Fixed not-casting problems -- hopefully
RCS Log:Gas diffusion was exiting without updating the array pointer,
RCS Log:and pores could multiply in the mould area!
RCS Log:
RCS Log:Revision 7.0  2000/11/07 15:53:28  rcatwood
RCS Log:Multi Cell Pores added
RCS Log:
RCS Log:Revision 6.1  2000/10/24 14:53:57  rcatwood
RCS Log:Changed grain nuc to include block_nuc method
RCS Log:
RCS Log:Revision 6.0.1.1  2000/10/16 17:29:36  rcatwood
RCS Log:started makeing non-casting cells
RCS Log:
RCS Log:Revision 6.0  2000/09/25 18:03:36  rcatwood
RCS Log:After PORE_00 and NLM
RCS Log:
RCS Log:Revision 2.0  2000/08/02 10:21:56  rcatwood
RCS Log:Version used for pore paper runs
RCS Log:
RCS Log:Revision 1.1  2000/05/22 12:29:24  rcatwood
RCS Log:Fixed fs finish. Casolid to C from  W file. Global option
RCS Log:
RCS Log:Revision 5.6  2000/05/04 18:36:45  rcatwood
RCS Log:Fixed fs finish. Casolid to C from  W file. Global option
RCS Log:
RCS Log:Revision 5.5  2000/04/11 14:44:05  rcatwood
RCS Log:Seperated castats routines. Fixed sreenprint bug and error overruns
RCS Log:
RCS Log:Revision 5.4  2000/03/15 16:25:32  rcatwood
RCS Log:backup checkin
RCS Log:
RCS Log:Revision 5.2  2000/03/08 18:38:23  rcatwood
RCS Log:speed up by only testing min,max during print step.
RCS Log:
RCS Log:Revision 5.1  2000/03/02 16:11:10  rcatwood
RCS Log:Merged xxu and rca versions
RCS Log:
RCS Log:Revision 5.0.2.1  2000/03/01 15:54:30  rcatwood
RCS Log:merged VAR and Multiblock updates. Not tested
RCS Log:
RCS Log:Revision 5.0.1.1  2000/02/22 19:04:27  rcatwood
RCS Log:Not yet tested
RCS Log:
RCS Log:Revision 4.6  2000/01/20 17:43:54  rcatwood
RCS Log:Pore bug fix. Bindump of fs, ss, diff coeff
RCS Log:
RCS Log:Revision 4.5  2000/01/20 17:41:17  rcatwood
RCS Log:Pore bug fix. Bindump of fs, ss, diff coeff
RCS Log:
RCS Log:Revision 4.4  2000/01/06 10:48:11  rcatwood
RCS Log:Fixed bug -- prototype in sb_diffuse_gas
RCS Log:
RCS Log:Revision 4.3  1999/12/23 18:12:24  rcatwood
RCS Log:Version used for Mcwasp runs
RCS Log:
RCS Log:Revision 4.2  1999/12/21 10:26:15  rcatwood
RCS Log:Solute arrays migrated to structure.
RCS Log:
RCS Log:Revision 4.1  1999/12/16 13:33:44  rcatwood
RCS Log:Finalised improved use of RCS in all files.
RCS Log:
RCS Log:Revision 4.0.2.2  1999/12/16 12:31:32  rcatwood
RCS Log:Improving rcs id for all files, this may require several checkins to test.
RCS Log:
*/
