/* #define DEBUG */
/****************************************************************/
/*   Copyright (c) 1998 Dept. of Materials, ICSTM               */
/*   All Rights Reserved                                        */
/*   THIS IS UNPUBLISHED PROPRIETARY SOURCE CODE OF ICSTM       */
/*   The copyright notice above does not evidence any           */
/*   actual or intended publication of such source code,        */
/*   and is an unpublished work by Dept. of Materials, ICSTM.   */
/*   This material contains CONFIDENTIAL INFORMATION that       */
/*   is the property of Imperial College. Any use,              */
/*   duplication or disclosure not specifically authorized      */
/*   by Imperial College is strictly prohibited.                */
/****************************************************************/
/* This code is part of the umats routines developed at in the  */
/* Materials Processing Group, Dept. of Materials, ICSTM.       */
/*      email p.d.lee or r.atwood @ic.ac.uk for details         */
/****************************************************************/

/****************************************************************/
/* sb_temp_calc.c:						*/
/* Calculate the new temperature for a sub-block. This routine  */
/* is only to be used in solo mode, as the Temperatrue is       */
/* supplied when running with a FEM model, or in postprocessing */
/* mode.                                                        */
/****************************************************************/
/****************************************************************/
/* Written by Peter D. Lee & Robert C. Atwood, Imperial College */
/* Aug 22, 1998                                                  */
/****************************************************************/
/* 	MODIFIED by:						*/
/*  PDL: Aug 22, 1998						*/
/****************************************************************/
/****** To Do List **********************************************/
/*General:							*/
/* ONLY DOES CONT. Drop in Temp!!!                              */
/* 1) Convert to true enthalpy calc...                          */
/****************************************************************/
/*RCS Id:$Id: sb_temp_calc.c 1491 2010-07-30 00:32:44Z  $*/
#include <stdio.h>
#include <signal.h>
#include "machine.h"
#include "blocks.h"
#include "interp.h"
#include "datatrans.h"
#include "ff_mpflow.h"
#include "math.h"

/* functions used from fidap_interp_calc.c */
extern CA_FLOAT fidap_interp_calc (FGrid_str * fg, CA_FLOAT x, CA_FLOAT y, CA_FLOAT time);

/* functions used from thermo_trace_calc.c */
extern CA_FLOAT thermo_trace_calc (TC_str * tc, CA_FLOAT tt);

/* functiosn used from bigblock.c */
extern void set_global_temperature (BB_struct * bp);

extern int vectorin(CA_FLOAT ***data3din, CA_FLOAT *data1dout, int io, int ie, int jo, int je, int ko, int ke);
extern int vectorout(CA_FLOAT ***data3dout, CA_FLOAT *data1din, int io, int ie, int jo, int je, int ko, int ke);

void sb_update_cell_temp (BB_struct * bp, int sbnum);

/***************************************************************/
/* Calculate the temperature if the finite grid method is used */
/***************************************************************/

CA_FLOAT fg_temp_calc (BB_struct * bp, int sbnum, int x, int y, int z)
{
  /* z is ignored but all temp calc functions must have the same args */
  int index_ca_2d;
  CA_FLOAT T_result, diff;

  index_ca_2d = x + bp->nc[0] * y;
  diff = bp->sb[sbnum]->c_fg_temp_next[index_ca_2d] - bp->sb[sbnum]->c_fg_temp[index_ca_2d];
  T_result = bp->sb[sbnum]->c_fg_temp[index_ca_2d] + bp->fg->wfact * diff;
  return (T_result);
}

CA_FLOAT const_temp_calc (BB_struct * bp, int sbnum, int x, int y, int z)
{
  return (bp->sb[sbnum]->Tvals.Tavg);
}

CA_FLOAT procast_temp_calc (BB_struct * bp, int sbnum, int x, int y, int z)
{
/**  \todo  improve procast temp array --  procast - coupled */
/* put the array at the subblock level or */
/* quick fix -- link the single subblock array to point to */
/* the same array as bp->current_cell_temp */

  int index_ca;
  CA_FLOAT this_temp;

  index_ca = x + y * bp->nc[0] + z * bp->nc[0] * bp->nc[1];
  return (bp->current_cell_temp[index_ca]);

}                               /* end procast_temp_calc */

CA_FLOAT dir_temp_calc (BB_struct * bp, int sbnum, int x, int y, int z)
{
  CA_FLOAT T, rad, rn, axis;    /* tmp CA_FLOAT var. */
  CA_FLOAT v, vt, tgrad, slope, iso_coef1, iso_coef2;   /* tmp CA_FLOAT var. */
  CA_FLOAT velo_coef, grad_coef, slope_coef;        /*varying G. V */
  
  if (bp->ctrl->swap_xy == 1) {
/* horizontal growth */
    axis = bp->sb[sbnum]->orig_sb[0] + x * bp->size_c[0];
    rad = bp->sb[sbnum]->orig_sb[1] + y * bp->size_c[1];
  } else {
    /* vertical growth */
    /* Swap x and y. by Wei WANG 07-08-02 */
    /* swap the direact in 3D, growth at z direction by LYUAN*/
    rad = bp->sb[sbnum]->orig_sb[0] + x * bp->size_c[0];
    axis = bp->sb[sbnum]->orig_sb[1] + y * bp->size_c[1];

    if(bp->nc[2]>1) {
    axis = bp->sb[sbnum]->orig_sb[2] + z * bp->size_c[2]; 
    }
  }

  rn = (CA_FLOAT) y / (CA_FLOAT) (bp->nc[1]);
/*****************************************************/
/* Calculate the temperature using analytic approx.  */
/*  uses a set gradient and velocity                 */
/* which is probably most useful for comparing with  */
/* directional solidification experiments.           */
/*****************************************************/
  velo_coef = bp->velo_coef;
  grad_coef = bp->grad_coef;
  slope_coef = bp->slope_coef;
  tgrad = bp->gradient + grad_coef * bp->sim_time;
  v = bp->velocity + velo_coef * bp->sim_time;
  
  vt = 0.5 * (v + bp->velocity)* bp->sim_time - bp->window_steps * bp->size_c[0];
  
  iso_coef1 = bp->iso_coef1;
 // iso_coef2 = bp->iso_coef2 / bp->size_c[0] / bp->nc[0] + bp->coef_iso2 *  bp->sim_time;
  iso_coef2 = bp->iso_coef2  + bp->coef_iso2 *  bp->sim_time;
  slope = bp->grad_slope + slope_coef * bp->sim_time;     /* thermal gradient deviation slope */
 //   T = bp->Tinit + (tgrad) * (axis - vt) - ((rn < 0.5) ? 0 : (slope * (rn - 0.5)));
    T = bp->Tinit + (tgrad) * (axis - vt)* cos(slope*PI/180) + rad* sin(slope*PI/180)*tgrad ;  /*for inclined T gradient ...by ly*/
//    T = bp->Tinit + (tgrad) * (axis - vt);
    T -= tgrad * (iso_coef1 * rad + iso_coef2 * rad * rad);
                                                      /**dong*/
 //  T = MIN (T, bp->Tinit + CAP_TEMP_OFFSET);
  return (T);
}                               /* end of dir_temp_calc */

/*------------------------------------------------------------*/
/*--------For pulling velocity variation, by LYUAN------------*/
/*------------------------------------------------------------*/
CA_FLOAT dir_temp_remelt(BB_struct * bp, int sbnum, int x, int y, int z){
  
  CA_FLOAT T, rad, axis;
  CA_FLOAT v, vt, tgrad, slope, sim_time;
  CA_FLOAT vtemp1, vtemp2, vtemp3;
  CA_FLOAT disttemp1, disttemp2, disttemp3;
  CA_FLOAT velo_coef, grad_coef;
  CA_FLOAT velo_coef1, velo_coef2, velo_coef3, velo_coef4;
  CA_FLOAT time_velo1, time_velo2, time_velo3, time_velo4;
  CA_FLOAT time_hold, velocity;
 
/*  FILE *inf;
  inf = fopen("v_t.cvs", "a+");
*/ 
  // output the V-t curve

  
  sim_time = bp->sim_time;
  slope = bp->grad_slope;
  
  if (bp->ctrl->swap_xy == 1) {
   /* horizontal growth */
    axis = bp->sb[sbnum]->orig_sb[0] + x * bp->size_c[0];
    rad = bp->sb[sbnum]->orig_sb[1] + y * bp->size_c[1];
  } else {
   /* vertical growth */
    rad = bp->sb[sbnum]->orig_sb[0] + x * bp->size_c[0];
    axis = bp->sb[sbnum]->orig_sb[1] + y * bp->size_c[1];

    if(bp->nc[2]>1) {
    axis = bp->sb[sbnum]->orig_sb[2] + z * bp->size_c[2];
    }
  }
    
  velo_coef = bp->velo_coef;
  grad_coef = bp->grad_coef;
  velo_coef1 = bp->velo_coef1;
  velo_coef2 = bp->velo_coef2;
  velo_coef3 = bp->velo_coef3;
  velo_coef4 = bp->velo_coef4;
  time_hold = bp->time_hold;
  time_velo1 = bp->time_velo1;
  time_velo2 = bp->time_velo2;
  time_velo3 = bp->time_velo3;
  time_velo4 = bp->time_velo4;
  velocity = bp->velocity;
  tgrad = bp->gradient + grad_coef * bp->sim_time;


/*  if (time_hold < time_velo2) {
    printf("time_hold < time_velo2!!!\nEXIT... \n"), exit(1) ;
    printf("time_hold: %.3f time_velo2 : %.3f \n\ni", time_hold, time_velo2);
  }
  */  

  vtemp1 = bp->velocity + velo_coef1 * time_velo1;
  vtemp2 = vtemp1 + velo_coef2 * (time_velo2-time_velo1);
  vtemp3 = velo_coef3 * (time_velo3-time_hold);
//  vtemp3 = vtemp2 + velo_coef3 * (time_velo3-time_velo2);
  disttemp1 = 0.5* (vtemp1 + velocity) * time_velo1;
  disttemp2 = disttemp1 + 0.5* (vtemp1 + vtemp2) * (time_velo2-time_velo1);
  disttemp3 = disttemp2 + 0.5* vtemp3 * (time_velo3-time_hold);
//  disttemp3 = disttemp2 +0.5* (vtemp2 + vtemp3) * (time_velo3-time_hold);

  if (time_hold ==  time_velo2) {
    vtemp3 = vtemp2 + velo_coef3 * (time_velo3-time_velo2);
    disttemp3 = disttemp2 +0.5* (vtemp2 + vtemp3) * (time_velo3-time_hold);
  }

  if (time_velo1 >= sim_time){
    v = velocity + velo_coef1 * sim_time;   /* initial stage */
    vt = 0.5 * (v + velocity) * sim_time;
  } else if (time_velo1 < sim_time && time_velo2 >= sim_time) {
    v = vtemp1 + velo_coef2 * (sim_time - time_velo1);
    vt = disttemp1 + 0.5 * (v + vtemp1) * (sim_time - time_velo1);
  } else if (time_velo2 < sim_time && time_hold >= sim_time) {
    v = 0.0;
    vt = disttemp2;
  } else if (time_hold < sim_time && time_velo3 >= sim_time) {
      if(time_hold - time_velo2 > 0) {
        v = velo_coef3 * (sim_time - time_hold);  //time_hold happens
	vt = disttemp2 + 0.5 * v * (sim_time - time_hold);
      } else if(time_hold == time_velo2) {
	v = vtemp2 + velo_coef3 * (sim_time - time_hold);  //ignore time_hold
   	vt = disttemp2 + 0.5 * (v + vtemp2) * (sim_time - time_velo2);
      }
  } else if (time_velo3 < sim_time) {
      v = vtemp3 + velo_coef4 * (sim_time - time_velo3);
      vt = disttemp3 + 0.5 * (v + vtemp3) * (sim_time - time_velo3);
  }

  T = bp->Tinit + tgrad * (axis - vt) * cos(slope*PI/180) + rad* sin(slope * PI/180)*tgrad;
/*
  if(x==2 && y==2 && ((int)(sim_time/bp->delt) % 100 == 0))  fprintf(inf, "%f\t%.4f\n", v, sim_time);
  fclose(inf);
*/ 
  //output for V-t curve
  
  return (T);  
}

/****************************************************************/
/****************************************************************/
/* Update the cell temperature for the whole array              */
/****************************************************************/
/****************************************************************/
void sb_update_cell_temp (BB_struct * bp, int sbnum)
{
  /* Update the entire cell temperature array */
  int i, j, k;
  CA_FLOAT x, y, z;
  CA_FLOAT *c_temp_p;           /* temperature pointer */
  int *gr_p;
  int n_casting = 0;
  int indexnum = 0;
  CA_FLOAT t_sum = 0, tmin = 10000.0, tmax = 0.0;
  int IMAX, JMAX, KMAX;

  IMAX = bp->tnc[0];
  JMAX = bp->tnc[1];
  KMAX = bp->tnc[2];
				  
//  Ctrl_str *cp;
//  bp->ctrl=cp;

  if (bp->sb[sbnum]->open != TRUE)
    return;
  /**  \todo  Figure out a way to test not yet open subblocks  -- multiblock */
  /* without needing the T array defined already */

  c_temp_p = bp->sb[sbnum]->c_temp;
  gr_p = bp->sb[sbnum]->gr;

 if(!(bp->ctrl->physica) && !(bp->ctrl->ffheattran) && !(bp->ctrl->extinput)){
  for (k = 0; k < bp->nc[2]; k++) {
    for (j = 0; j < bp->nc[1]; j++) {
      for (i = 0; i < bp->nc[0]; i++) {
        if (*gr_p != NOT_CASTING) {
          *c_temp_p = (bp->cell_temp_func) (bp, sbnum, i, j, k);
          t_sum += *c_temp_p;
          tmin = MIN (tmin, *c_temp_p);
          tmax = MAX (tmax, *c_temp_p);
          n_casting++;
        }

        gr_p++;
        c_temp_p++;
      }                        
    }                          
  }                            
  } 
 
 else if(bp->ctrl->physica){
//  bp->sb[sbnum]->c_temp = bp->ca_cell_Tar;
//  c_temp_p=bp->ca_cell_Tar;
//  gr_p = bp->sb[sbnum]->gr;
    
   for (k = 0; k < bp->nc[2]; k++) {
     for (j = 0; j < bp->nc[1]; j++) {
       for (i = 0; i < bp->nc[0]; i++) {
         if (*gr_p != NOT_CASTING) {
	  *c_temp_p = bp->ca_cell_Tar[indexnum];
          t_sum += bp->ca_cell_Tar[indexnum];
          tmin = MIN (tmin, bp->ca_cell_Tar[indexnum]);
          tmax = MAX (tmax, bp->ca_cell_Tar[indexnum]);
          n_casting++;
         }
       indexnum ++;
       gr_p++;
       c_temp_p++;
       }                         /* end of i loop */
     }                           /* end of j loop */
   }         
 } 
 
 else if(bp->ctrl->ffheattran){
   
   /* 3D array transfers into 1D array */	 
   vectorout(bp->ca_cell_t, c_temp_p, 1, IMAX, 1, JMAX, 1, KMAX); 

   for (k = 0; k < bp->nc[2]; k++) {
     for (j = 0; j < bp->nc[1]; j++) {
       for (i = 0; i < bp->nc[0]; i++) {
         if (*gr_p != NOT_CASTING) {
	   t_sum += c_temp_p[indexnum];
	   tmin = MIN (tmin, c_temp_p[indexnum]);
	   tmax = MAX (tmax, c_temp_p[indexnum]);
	   n_casting++;
	 }
       indexnum ++;
       gr_p++;
       }                         /* end of i loop */
     }                           /* end of j loop */
   }
  
 } else if(bp->ctrl->extinput == 1){
 
   vectorout(bp->ext_temp, c_temp_p, 0, bp->nc[0]-1, 0, bp->nc[1]-1,0, bp->nc[2]-1);

   for (k = 0; k < bp->nc[2]; k++) {
     for (j = 0; j < bp->nc[1]; j++) {
       for (i = 0; i < bp->nc[0]; i++) {
	 if (*gr_p != NOT_CASTING) {
	   t_sum += c_temp_p[indexnum];
	   tmin = MIN (tmin, c_temp_p[indexnum]);
	   tmax = MAX (tmax, c_temp_p[indexnum]);
	   n_casting++;
	 }
         indexnum ++;
	 gr_p++;
       }                         /* end of i loop */
     }                           /* end of j loop */
   }
   
 }
  
 
  /* copy the stat vals into the subblock */
  bp->sb[sbnum]->Tvals.Tavg = t_sum / (n_casting);
  bp->sb[sbnum]->Tvals.Tmin = tmin;
  bp->sb[sbnum]->Tvals.Tmax = tmax;
  bp->sb[sbnum]->Tvals.TminReached = MIN (tmin, bp->sb[sbnum]->Tvals.TminReached);
}                               /* end of update cell temp routine */

/****************************************************************/
/****************************************************************/
/* sb_temp_calc: Calculate the new temperature for a sub-block  */
/****************************************************************/
/****************************************************************/
int sb_temp_calc (BB_struct * bp, int sbnum)
{
  CA_FLOAT delT;                /* tmp CA_FLOAT var. */
  T_struct *Tp;
  Mat_str *mp;

  Tp = &(bp->sb[sbnum]->Tvals);
  mp = &(bp->mprops);

/* check for temperature lookup table */
/* used by Ali's version -- not sure what it is for!*/
/** \todo  ask Ali about this temperature lookup control - maybe obsolete - procast */
  if (bp->ctrl->temp_lookup == FALSE) {
    if (!(bp->ctrl->use_cell_temp)) {
      /* Constant Cooling only */
      Tp->TminReached = MIN (Tp->TminReached, Tp->Tavg);        /* update minimum T reached first... */
      Tp->Tavg_old = Tp->Tavg;
      if (bp->ctrl->t_input) {
        /* Using thermocouple trace input */
        Tp->Tavg = thermo_trace_calc (&(bp->tc), bp->sim_time);

      } else if (bp->ctrl->coolrate == TRUE) {
        /* constant cooling rate */
        Tp->Tavg += (bp->ctrl->delT) * (bp->delt);
        Tp->del_fs = 0;

      } else {

        /* Using heatflow calculation */
        /* calc the change in temperature */
        /* delT = (QSV*delt + latentH*del_fs)/rhocp; */
        /*delT = (mp->QSV * bp->delt + (mp->latentH * Tp->del_fs*bp->vol_sb))/mp->rhocp; */
        delT = (mp->QSV * bp->delt + (mp->latentH * Tp->del_fs)) / mp->rhocp;
        Tp->Tavg += delT;
        Tp->del_fs = 0;
      }

    }

    /* update the cell temperature array, for both isothermal and cell temperature */
    sb_update_cell_temp (bp, sbnum);
  } else {
    /* Ali Chirazi global temperature option */
    set_global_temperature (bp);
    Tp->Tavg = bp->this_temp;
  }

  /*Failsafe finish if temperature goes below zero or FROZEN */
  if (Tp->Tavg < FROZEN) {
    fprintf (stderr, "ERROR: sb_temp_calc: Temperature has frozen! %.5g  Finishing ...\n", Tp->Tavg);
    return (raise (SIGTERM));
  }

  return (0);
}                               /* end of sb_temp_calc subroutine */

/*PROCAST routines here!!! */
/*PROCAST routines here!!! */
/*PROCAST routines here!!! */
/*PROCAST routines here!!! */
/*PROCAST routines here!!! */
/******************************************************************/
/* special case of Procast mode on calculation for temperature*****/
/*****************************************************************/
void cell_temp_calc_procast (BB_struct * bp, int index_ca)
{
  float t_elapsed, temp_change;

  if (bp->cell_element_array[index_ca] >= 0) {

    if (bp->first_micro == 0) {
      bp->current_cell_temp[index_ca] = bp->cell_temp_procast[index_ca];
    } else {
      t_elapsed = bp->delt * bp->micro_step;
      temp_change = bp->cell_temp_change_procast[index_ca] * t_elapsed;
      bp->current_cell_temp[index_ca] = bp->cell_temp_procast[index_ca] + temp_change;
      /*
         bp->current_cell_temp[index_ca]+=(bp->cell_temp_change_procast[index_ca]*bp->delt);
       */
    }
  } else {
    bp->current_cell_temp[index_ca] = 0.0;
  }

  return;
}

#ifdef AAA
void cell_temp_calc_procast (BB_struct * bp, int index_ca)
{
  double t_elapsed, temp_change;

  if (bp->cell_element_array[index_ca] >= 0) {
    if (bp->first_micro == 0) {
      bp->current_cell_temp[index_ca] = bp->cell_temp_procast[index_ca];
    } else {
      t_elapsed = (double) ((double) bp->delt * (double) bp->micro_step);
      temp_change = ((double) (bp->cell_temp_change_procast[index_ca]) * t_elapsed);
      bp->current_cell_temp[index_ca] = (float) ((double) (bp->cell_temp_procast[index_ca])
                                                 + temp_change);

      /*
         bp->current_cell_temp[index_ca]+=(bp->cell_temp_change_procast[index_ca]*bp->delt);
       */
    }
  } else {
    bp->current_cell_temp[index_ca] = 0.0;
  }

  return;
}
#endif
/******************************************************************/
/* special case of Procast mode on for min temp calculation */
/******************************************************************/
float min_temp_calc_procast (BB_struct * bp)
{
/*********from user_rtns.c**************/
#ifdef PROCAST_CA
  extern void shell (int n1, float *a);
#endif

 /****local variables*********/
  int k;
  float min_temp;
  float *local_temp;
  int local_counter;

    /*******allocate local temp array**************/
  local_temp = (float *) calloc (bp->total_cell_number + 1, sizeof (float));

    /***copy the temperature array into a local array for sort purposes***/
  for (k = 0, local_counter = 0; k < bp->total_cell_number; k++) {
    if (bp->cell_element_array[k] >= 0) {
      local_counter++;
      local_temp[local_counter] = bp->current_cell_temp[k];
    }
  }
    /*************************************************/

    /***********call shell function to sort the array*********/
#ifdef PROCAST_CA
  shell (local_counter, local_temp);
#endif

  min_temp = local_temp[1];
    /**********free the local array once the min temp is assigned******/
  free (local_temp);

  return (min_temp);
}

/****************************************************************/
/******************************************************************/
/* special case of Procast mode on calculation for pressure*****/
/*****************************************************************/
void find_cell_pressure (BB_struct * bp, int index_ca)
{
  if (bp->cell_element_array[index_ca] >= 0) {
    if (bp->first_micro == 0) {
      bp->current_cell_pres[index_ca] = bp->cell_pres_procast[index_ca];
    } else {
      bp->current_cell_pres[index_ca] += (bp->cell_pres_grad_procast[index_ca] * bp->delt);
    }
  } else {
    bp->current_cell_pres[index_ca] = 0.0;
  }

  return;
}

/*finite element (fgrid) routines here ! */
/*finite element (fgrid) routines here ! */
/*finite element (fgrid) routines here ! */
/*finite element (fgrid) routines here ! */
/*finite element (fgrid) routines here ! */

/*******************************************/
/* Set up the first temperature interp.    */
/*******************************************/
/* using the first two files */
/* TWO DIMENSIONAL temperature distribution only */
void sb_temp_setup (BB_struct * bp, int sbnum)
{
  int i, j, k, index_ca_2d;

  for (k = 0, index_ca_2d = 0; (k < bp->nc[2] && index_ca_2d < bp->nc[0] * bp->nc[1] * bp->nc[2]); k++) {
    for (j = 0; j < bp->nc[1]; j++) {
      for (i = 0; i < bp->nc[0]; i++) {
        /* get the temperature at this (x,y) location and store */
        bp->sb[sbnum]->c_fg_temp[index_ca_2d] = trans_interp_calc (bp->fg, bp->sb[sbnum]->nnd, bp, sbnum, i, j);
        bp->sb[sbnum]->c_fg_temp_next[index_ca_2d] = trans_interp_calc (bp->fg_next, bp->sb[sbnum]->nnd_next, bp, sbnum, i, j);

        index_ca_2d++;
      }
    }
  }

}                               /* end of sb_temp_setup */

/* update the start and finish temperature arrays */
/* after a new finite element (fgrid) file is read in */
void fg_temp_upd (BB_struct * bp, int sbnum)
{
  int i, j, k, index_ca;
  CA_FLOAT *holder;
  NODENB_str *n_holder;

  /* exchange the pointers for the temperatrue arrays */
  holder = bp->sb[sbnum]->c_fg_temp;
  bp->sb[sbnum]->c_fg_temp = bp->sb[sbnum]->c_fg_temp_next;
  bp->sb[sbnum]->c_fg_temp_next = holder;

  /* exchange the pointers for the interploation arrays */
  n_holder = bp->sb[sbnum]->nnd;
  bp->sb[sbnum]->nnd = bp->sb[sbnum]->nnd_next;
  bp->sb[sbnum]->nnd_next = n_holder;

   /*******************************************/
  /* Calculate the weighting factors in z    */
  /* only when a new fg has been read in      */
   /*******************************************/
  wfact_z_calc (bp->fg_next, bp->sb[sbnum]->nnd_next, bp, sbnum);
  /* update the temperature for the next step */
  for (k = 0, index_ca = 0; (k < bp->nc[2] && index_ca < bp->nc[0] * bp->nc[1] * bp->nc[2]); k++) {
    for (j = 0; j < bp->nc[1]; j++) {
      for (i = 0; i < bp->nc[0]; i++) {
        bp->sb[sbnum]->c_fg_temp_next[index_ca] = trans_interp_calc (bp->fg_next, bp->sb[sbnum]->nnd_next, bp, sbnum, i, j);

        index_ca++;
      }                         /*i */
    }                           /*j */
  }                             /*k */
}                               /* end of fg_temp_upd */

/********************************************************/
/* Little subroutine to get rcs id into the object code */
/* so you can use ident on the compiled program  */
/* also you can call this to print out or include the rcs id in a file*/
/********************************************************/
char const *rcs_id_sb_temp_calc_c ()
{
  static char const rcsid[] = "$Id: sb_temp_calc.c 1491 2010-07-30 00:32:44Z  $";

  return (rcsid);
}

/* end of rcs_id_sb_temp_calc_c subroutine */
/*
RCS Log:$Log$
RCS Log:Revision 11.1  2006/03/01 18:20:40  rcatwood
RCS Log:Merging polycomponent and gas with meltback
RCS Log:
RCS Log:Revision 10.5  2005/12/06 13:09:54  rcatwood
RCS Log:Changed todo lists to Doxygen syntax
RCS Log:
RCS Log:Revision 10.4  2005/12/06 12:58:01  rcatwood
RCS Log:Improved the to-do list information
RCS Log:
RCS Log:Revision 10.3  2005/12/01 14:38:02  rcatwood
RCS Log:Merged xly_05 changes into the main trunk
RCS Log:Primarily involving melt-back
RCS Log:
RCS Log:Revision 10.1.2.2  2005/11/23 18:18:53  rcatwood
RCS Log:Result of merging mould_source and xly meltback+curvature 2d versions
RCS Log:
RCS Log:Revision 10.1  2005/11/03 11:56:47  rcatwood
RCS Log:New version number -- using mould_src as base
RCS Log:
RCS Log:Revision 8.4.8.2  2005/11/02 11:55:06  rcatwood
RCS Log:Fixing up the revision nubmer after loss of repository
RCS Log:
RCS Log:Revision 9.6  2003/11/18 11:50:16  rcatwood
RCS Log:*** empty log message ***
RCS Log:
RCS Log:Revision 9.5  2003/11/17 18:53:51  rcatwood
RCS Log:Replaced swap-xy option
RCS Log:
RCS Log:Revision 9.4  2003/11/17 18:33:14  hdong
RCS Log:
RCS Log:for varying the pulling velocity and thermal gradient
RCS Log:
RCS Log:Revision 9.3  2003/11/03 19:14:26  hdong
RCS Log:
RCS Log:log message
RCS Log:
RCS Log:Revision 9.2  2003/09/16 11:59:15  rcatwood
RCS Log:Improved micro/macro interpolation
RCS Log:
RCS Log:Revision 9.1  2003/08/14 14:38:40  rcatwood
RCS Log:Working merge with decentered/porosity/procast, also including
RCS Log:Ali Chirazi's multicomponent (not tested in this version)
RCS Log:
RCS Log:Revision 8.4.2.11  2003/08/14 14:18:04  rcatwood
RCS Log:Working ca_procast new version, on linux
RCS Log:Added surface nucleation
RCS Log:Added mould source term
RCS Log:Changed printout headers
RCS Log:Temperature output image
RCS Log:
RCS Log:Revision 8.4.2.10  2003/05/19 18:55:17  rcatwood
RCS Log:Addded option to allow horizontal or vertical directional growth
RCS Log:and flux boundary condition
RCS Log:
RCS Log:Revision 8.4.2.9  2003/05/14 11:49:54  rcatwood
RCS Log:Fixed temperature input to conform wiht new temperature calculation methods
RCS Log:and control input file
RCS Log:
RCS Log:Revision 8.4.2.8  2003/04/08 09:41:11  rcatwood
RCS Log:rubbish in dir_temp_calc
RCS Log:
RCS Log:Revision 8.4.2.7  2003/03/14 16:41:47  hdong
RCS Log:Fixed temperature option selection for re-start.
RCS Log:Fixed output_excel for restart case
RCS Log:
RCS Log:Revision 8.4.2.6  2003/02/27 23:04:40  rcatwood
RCS Log:Removed use of old temperature routines , all temperatures shoudl
RCS Log:be determined by checking the array c_temp in teh subblock, if the
RCS Log:subblock is open
RCS Log:
RCS Log:Revision 8.4.2.5  2003/02/26 18:44:49  rcatwood
RCS Log:Modified (non-decentered) so that the temperature uses an array
RCS Log:calcuated before each step. Exception: Procast mode still overrides
RCS Log:and uses Ali's routine.
RCS Log:
RCS Log:Revision 8.4.2.4  2003/01/23 17:47:28  rcatwood
RCS Log:finite grid applied to decentered square,
RCS Log:works, but not checked for correct results.
RCS Log:
RCS Log:Revision 8.4.2.2  2003/01/15 19:02:02  rcatwood
RCS Log:*** empty log message ***
RCS Log:
RCS Log:Revision 8.2.6.1  2002/11/06 17:27:48  rcatwood
RCS Log:NOT WORKING check-in of first stage merge with ca_procast
RCS Log:
RCS Log:Revision 7.3.8.1  2002/08/27 14:18:19  chirazi
RCS Log:adding files for multi-component-Procast version of CA
RCS Log:
RCS Log:Revision 7.3  2001/02/19 21:35:25  rcatwood
RCS Log:fixed more histo,
RCS Log:also found memory allocation bug, not sur eif it is fixed.
RCS Log:
RCS Log:Revision 7.2  2001/02/19 19:28:47  rcatwood
RCS Log:fixed histo
RCS Log:for grains
RCS Log:
RCS Log:and also make TcTrace mode override const. cooling rate
RCS Log:
RCS Log:Revision 7.1  2000/12/06 21:10:40  rcatwood
RCS Log:fixed up heatfolw, tctrace
RCS Log:
RCS Log:Revision 7.0  2000/11/07 15:53:28  rcatwood
RCS Log:Multi Cell Pores added
RCS Log:
RCS Log:Revision 6.0  2000/09/25 18:03:35  rcatwood
RCS Log:After PORE_00 and NLM
RCS Log:
RCS Log:Revision 2.0  2000/08/02 10:21:56  rcatwood
RCS Log:Version used for pore paper runs
RCS Log:
RCS Log:Revision 1.1  2000/05/22 12:29:24  rcatwood
RCS Log:Fixed fs finish. Casolid to C from  W file. Global option
RCS Log:
RCS Log:Revision 5.1  2000/03/02 16:11:10  rcatwood
RCS Log:Merged xxu and rca versions
RCS Log:
RCS Log:Revision 5.0.2.1  2000/03/01 15:54:30  rcatwood
RCS Log:merged VAR and Multiblock updates. Not tested
RCS Log:
RCS Log:Revision 1.3  2000/02/23 20:21:31  rcatwood
RCS Log:Improved interpolation version
RCS Log:
RCS Log:Revision 1.2  2000/02/23 15:54:07  xxu
RCS Log:read one more flag for temperature interpolation
RCS Log:Revision 5.0.1.1  2000/02/22 19:04:27  rcatwood
RCS Log:Not yet tested
RCS Log:
RCS Log:Revision 4.5  2000/02/15 15:29:11  rcatwood
RCS Log:Version after McWasp submitted
RCS Log:
RCS Log:Revision 4.4  2000/01/06 10:48:11  rcatwood
RCS Log:Fixed bug -- prototype in sb_diffuse_gas
RCS Log:
RCS Log:Revision 4.3  1999/12/23 18:12:24  rcatwood
RCS Log:Version used for Mcwasp runs
RCS Log:
RCS Log:Revision 4.2  1999/12/23 18:09:21  rcatwood
RCS Log:Solute arrays migrated to structure.
RCS Log:
RCS Log:Revision 4.1  1999/12/16 13:33:44  rcatwood
RCS Log:Finalised improved use of RCS in all files.
RCS Log:
RCS Log:Revision 4.0.2.2  1999/12/16 12:31:32  rcatwood
RCS Log:Improving rcs id for all files, this may require several checkins to test.
RCS Log:
*/
