/* #define DEBUG_FIDAP */
/****************************************************************/
/*   Copyright (c) 1998 Dept. of Materials, ICSTM               */
/*   All Rights Reserved                                        */
/*   THIS IS UNPUBLISHED PROPRIETARY SOURCE CODE OF ICSTM       */
/*   The copyright notice above does not evidence any           */
/*   actual or intended publication of such source code,        */
/*   and is an unpublished work by Dept. of Materials, ICSTM.   */
/*   This material contains CONFIDENTIAL INFORMATION that       */
/*   is the property of Imperial College. Any use,              */
/*   duplication or disclosure not specifically authorized      */
/*   by Imperial College is strictly prohibited.                */
/****************************************************************/
/* This code is part of the umats routines developed at in the  */
/* Materials Processing Group, Dept. of Materials, ICSTM.       */
/*   email p.d.lee or r.atwood or x.xu @ic.ac.uk for details    */
/****************************************************************/

/****************************************************************/
/* trans_interp_calc.c:                                         */
/* Subroutine to interpolate the temperature at a point         */
/* in the CA code using values calculated in a                  */
/* transient state solution of the heat, mass and               */
/* momentum transfer in VAR ingots.                             */
/****************************************************************/
/****************************************************************/
/* Written by X. Xu, P.D. Lee & R.C. Atwood, Imperial College   */
/* Feb. 18, 2000                                                */
/****************************************************************/
/*      MODIFIED by:                                            */
/****************************************************************/
/****** To Do List **********************************************/
/*General:                                                      */
/* 1)                                                           */
/****************************************************************/
#include <stdio.h>
#include <math.h>
/* include header files requred by subroutines */
#include "machine.h"
#include "blocks.h"
#include "fidap.h"              /* included for def. of FGrid_struct */
#include "nearnode.h"

/****************************************************************/
/****************************************************************/
/* trans_interp_calc.c                                          */
/* Subroutine to interpolate the temperature of cell            */
/****************************************************************/
/* Input Variables:                                             */
/*   *fg:		ptr to the FGrid_str structure that     */
/*       		holds all FIDAP geometry and T's data.  */
/*   r:	       	        radial coordinate of a cell centre      */
/*   z:                 Height coordination of a cell centre    */
/*      		                                        */
/* Output Variables:    NONE                                    */
/*      		                                        */
/* Returned Value:                                              */
/*   temperature in the centre of cellz                         */
/****************************************************************/

CA_FLOAT trans_interp_calc (FGrid_str * fg, NODENB_str * node_ptr, BB_struct * bp, int sbnum, int x, int y)
{
  int nr1, nr2, nz1, nz2, nnr;
  CA_FLOAT T, *fg_T_pt;

  /* pointer of val. for local usage. */
  fg_T_pt = fg->Fidap_T;
  nnr = fg->nr;

  /* In Z direction, check the node down the cell is over/below the CFD domain */

  nz1 = node_ptr->nd[x];

  if (nz1 == -1) {
    T = fg->Tmax;
  } else if (nz1 == fg->nz - 1) {
    T = fg->Tmin;
  } else {

    nr1 = node_ptr->nl[y];      /* near left node. */
    nr2 = nr1 + 1;
    nz2 = nz1 + 1;

    /* using pre-calculated interpolation weight functions (linear) */
    /*  T = wd*wl*tp1 + wd*wr*tp2 + wu*wl*tp3 + wu*wr*tp4; */
    T = node_ptr->wd[x] * (node_ptr->wl[y] * fg_T_pt[nnr * nz1 + nr1] +
                           node_ptr->wr[y] * fg_T_pt[nnr * nz1 + nr2]) +
      node_ptr->wu[x] * (node_ptr->wl[y] * fg_T_pt[nnr * nz2 + nr1] + node_ptr->wr[y] * fg_T_pt[nnr * nz2 + nr2]);
  }
  return (T);
}

/* end of trans_interp_calc(FGrid_str *fg, BB_struct *bp, int sbnum, int x, int y). */

/* Little subroutine to get rcs id into the object code */
/* so you can use ident on the compiled program  */
/* also you can call this to print out or include the rcs id in a file*/
char const *rcs_id_trans_interp_calc_c ()
{
  static char const rcsid[] = "$Id: trans_interp_calc.c 887 2006-03-01 18:21:01Z rcatwood $";

  return (rcsid);
}

/* end of rcs_id_fidap_interp_calc_c subroutine */

/*RCS Id:$Id: trans_interp_calc.c 887 2006-03-01 18:21:01Z rcatwood $*/
/*
RCS Log:$Log$
RCS Log:Revision 11.1  2006/03/01 18:20:40  rcatwood
RCS Log:Merging polycomponent and gas with meltback
RCS Log:
RCS Log:Revision 10.3  2005/12/01 14:38:02  rcatwood
RCS Log:Merged xly_05 changes into the main trunk
RCS Log:Primarily involving melt-back
RCS Log:
RCS Log:Revision 10.1.2.2  2005/11/23 18:18:53  rcatwood
RCS Log:Result of merging mould_source and xly meltback+curvature 2d versions
RCS Log:
RCS Log:Revision 10.1  2005/11/03 11:56:48  rcatwood
RCS Log:New version number -- using mould_src as base
RCS Log:
RCS Log:Revision 8.1.14.2  2005/11/02 11:55:06  rcatwood
RCS Log:Fixing up the revision nubmer after loss of repository
RCS Log:
RCS Log:Revision 9.1  2003/08/14 14:38:40  rcatwood
RCS Log:Working merge with decentered/porosity/procast, also including
RCS Log:Ali Chirazi's multicomponent (not tested in this version)
RCS Log:
RCS Log:Revision 8.1.8.3  2003/01/22 16:53:47  rcatwood
RCS Log:Almost working read_fg version
RCS Log:
RCS Log:Revision 8.1.8.2  2003/01/15 19:02:02  rcatwood
RCS Log:*** empty log message ***
RCS Log:
RCS Log:Revision 8.1.6.1  2002/11/06 17:27:48  rcatwood
RCS Log:NOT WORKING check-in of first stage merge with ca_procast
RCS Log:
RCS Log:Revision 8.1  2002/10/17 17:01:04  rcatwood
RCS Log:New version number! for decentered/porosity merge! Alpha Version!
RCS Log:
RCS Log:Revision 7.1  2002/10/17 16:52:38  rcatwood
RCS Log:Merge from branch: combined Robert (porosity) and Wei (decentered octahedron) versions
RCS Log:
RCS Log:Revision 7.0.14.1  2002/09/12 10:48:08  rcatwood
RCS Log:Changed pore routine to avoid getxyz routine
RCS Log:
RCS Log:Revision 7.0  2000/11/07 15:53:28  rcatwood
RCS Log:Multi Cell Pores added
RCS Log:
RCS Log:Revision 6.0  2000/09/25 18:03:36  rcatwood
RCS Log:After PORE_00 and NLM
RCS Log:
RCS Log:Revision 2.0  2000/08/02 10:21:56  rcatwood
RCS Log:Version used for pore paper runs
RCS Log:
RCS Log:Revision 1.1  2000/05/22 12:29:24  rcatwood
RCS Log:Fixed fs finish. Casolid to C from  W file. Global option
RCS Log:
RCS Log:Revision 5.1  2000/03/02 16:10:16  rcatwood
RCS Log:Merged xxu and rca versions
RCS Log:
RCS Log:Revision 5.0.2.1  2000/03/01 16:06:53  rcatwood
RCS Log:merge VAR and Multiblock
RCS Log:
RCS Log:Revision 5.0  2000/03/01 16:00:54  rcatwood
RCS Log:Merge robert and xxu versions
RCS Log:
RCS Log:Revision 1.9  2000/02/28 15:59:36  xxu
RCS Log:improved version for temp. interpolation.
RCS Log:
RCS Log:Revision 1.8  2000/02/28 15:56:38  xxu
RCS Log:improved version for efficient inperpolation.
RCS Log:
RCS Log:Revision 1.7  2000/02/28 10:56:58  xxu
RCS Log:ci fidap.h
RCS Log:set an offset for transient calculation
RCS Log:
RCS Log:Revision 1.6  2000/02/24 19:00:38  xxu
RCS Log:solve Z_cell_max > Z_node_max problem
RCS Log:
RCS Log:Revision 1.5  2000/02/24 18:57:43  xxu
RCS Log:sort out problem caused by Z_cell_max  > Z_node_max
RCS Log:
RCS Log:Revision 1.4  2000/02/24 17:54:43  xxu
RCS Log:change in steady state temp interpolation
RCS Log:
RCS Log:Revision 1.3  2000/02/23 16:37:12  rcatwood
RCS Log:Improved interpolation version
RCS Log:
RCS Log:Revision 1.2  2000/02/23 15:54:07  xxu
RCS Log:read one more flag for temperature interpolation
RCS Log:
RCS Log:Revision 1.1  2000/02/22 17:14:11  xxu
RCS Log:Initial revision
RCS Log:
RCS Log:Revision 1.1  2000/02/22 14:20:26  xxu
RCS Log:Initial revision
RCS Log:

*/
