/* #define DEBUG_FIDAP */
/****************************************************************/
/*   Copyright (c) 1998 Dept. of Materials, ICSTM               */
/*   All Rights Reserved                                        */
/*   THIS IS UNPUBLISHED PROPRIETARY SOURCE CODE OF ICSTM       */
/*   The copyright notice above does not evidence any           */
/*   actual or intended publication of such source code,        */
/*   and is an unpublished work by Dept. of Materials, ICSTM.   */
/*   This material contains CONFIDENTIAL INFORMATION that       */
/*   is the property of Imperial College. Any use,              */
/*   duplication or disclosure not specifically authorized      */
/*   by Imperial College is strictly prohibited.                */
/****************************************************************/
/* This code is part of the umats routines developed at in the  */
/* Materials Processing Group, Dept. of Materials, ICSTM.       */
/*   email p.d.lee or r.atwood or x.xu @ic.ac.uk for details    */
/****************************************************************/

/****************************************************************/
/* wfact_z_calc.c:                                              */
/* Subroutine to calculation the weighting factors for cells    */
/* position against the nodes position                          */
/****************************************************************/
/* Written by X. Xu, P.D. Lee & R.C. Atwood, Imperial College   */
/* Feb. 18, 2000                                                */
/****************************************************************/
/*      MODIFIED by:                                            */
/****************************************************************/
/****** To Do List **********************************************/
/*General:                                                      */
/* 1)                                                           */
/****************************************************************/
#include <stdio.h>
#include <math.h>
/* include header files requred by subroutines */
#include "machine.h"
#include "blocks.h"
#include "fidap.h"              /* included for def. of FGrid_struct */
#include "nearnode.h"
/*subroutines later in this file*/
int search_z (FGrid_str * fg, CA_FLOAT z);
int compare_z (FGrid_str * fg, int pn, CA_FLOAT z);

/****************************************************************/
/****************************************************************/
/* wfact_z_calc.c:                                             */
/* Subroutine to calculate weighting factor of cells in height  */
/* direction                                                    */
/****************************************************************/
/* Input Variables:                                             */
/*   *fg:		ptr to the FGrid_str structure that     */
/*       		holds all FIDAP geometry and T's data.  */
/*   *bp:		hold location of cellsin the CA model   */
/*   time:		time in the CA model             [s]    */
/*      		                                        */
/* Output Variables:    weight factor in height direction.      */
/*      		                                        */
/* Returned Value:      NONE                                    */
/****************************************************************/
void wfact_z_calc (FGrid_str * fg, NODENB_str * node_ptr, BB_struct * bp, int sbnum)
{
  CA_FLOAT zc, zm;
  int i, ii, ic, imax, pnd, nnz;
  CA_FLOAT sub_off, trans_off, cell_offset, size_cell, *fg_z_pt;
  CA_FLOAT time;

  time = bp->sim_time;
  /* set pointers and variables for local usage */
  fg_z_pt = fg->z;
  nnz = fg->nz;
  trans_off = fg->z_off;        /* offset for start CA calculation within melt pool */
  sub_off = bp->sb[sbnum]->orig_sb[0];  /*     lower left corner of the subblock     */
  size_cell = bp->size_c[0];    /* cell size in x direction, it equals to that in y & Z dir. */
  cell_offset = sub_off - 0.5 * size_cell;      /* center of the cell in front of it. */

  /* calculate z arrays of cells in Z direction */
  zc = cell_offset;             /* initial value */
  imax = bp->nc[0];

  for (i = 0; i < imax; i++) {
    zc += size_cell;

    /* calculate Z position of cell in a CFD domain */

    if (fg->state == TRANSIENT) {
      zm = fg->h_ingot - trans_off - zc;
    } else if (fg->state == STEADY) {
      zm = -zc + fg->v * time;
    }

         /*********************************************************************/
    /*  check if the CA cell is out of the CFD domain, if it is over the */
    /*  top of  CFD domain, set node No.= -1, therefore use  it in       */
    /*  temperature interpolation subroutine for Tcell = Tmax, otherwise */
    /*  if below the bottom of CFD domain , set node No.the Maxium and   */
    /*  T = Tmin                                                         */
         /*********************************************************************/

    if (zm <= 0.0) {
      node_ptr->nd[i] = -1;
    } else if (zm > fg->h_ingot - trans_off) {
      node_ptr->nd[i] = fg->nz - 1;
    } else {

          /******************************************************************/
      /* Make a bisection search for the nearest node in the first step, */
      /*  compare zm of cell with the nearest node in other steps.      */
          /******************************************************************/

#ifdef JUNK
      if (bp->step < 1) {
#endif
        ic = search_z (fg, zm);
#ifdef JUNK
      } else {
        pnd = node_ptr->nd[i];
        ic = compare_z (fg, pnd, zm);
      }
#endif
      node_ptr->nd[i] = ic;     /* nearest node down the cell.  */
           /****************************************************/
      /* factor that the cell is away from the down node, */
      /* when it lies on the node, wd[i] = 1.             */
           /****************************************************/
      node_ptr->wd[i] = (fg_z_pt[ic + 1] - zm) / (fg_z_pt[ic + 1] - fg_z_pt[ic]);
      node_ptr->wu[i] = 1 - node_ptr->wd[i];    /* factor from upper nodes. */
    }
  }
/* Debug only */
  /*       if ((node_ptr->wd[i] > 1) || (node_ptr->wd[i] < 0 )) {
     fprintf(stderr,"ERROR wd, i [%f %d]", node_ptr->wd[i], i);
     }
   */
}                               /*  end of wfact_z_calc    */

/* subroutine for search the node before the CA cell  */

int search_z (FGrid_str * fg, CA_FLOAT z)
{
  int i, im, iu, il, JP;
  CA_FLOAT *fg_z_pt;

  fg_z_pt = fg->z;

  iu = fg->nz - 1;
  il = 0;
  while (iu - il > 1) {
    im = (iu + il) / 2;
    if ((z > fg_z_pt[il]) && (z > fg_z_pt[im])) {
      il = im;
    } else {
      iu = im;
    }
  }
  JP = il;
  return (JP);
}

/* Subroutine for compare the coordicate of the cell with the node coor. in current step */
int compare_z (FGrid_str * fg, int pn, CA_FLOAT z)
{
  int i, imax, JP;
  CA_FLOAT *fg_z_pt;

  fg_z_pt = fg->z;
  JP = pn;
  imax = 2;

  for (i = imax; i >= 1; i--) {
    if (z > fg_z_pt[pn + i]) {
      JP = JP + i;
      break;
    }
  }
  return (JP);
}

/* Little subroutine to get rcs id into the object code */
/* so you can use ident on the compiled program  */
/* also you can call this to print out or include the rcs id in a file*/
char const *rcs_id_wfact_z_calc_c ()
{
  static char const rcsid[] = "$Id: wfact_z_calc.c 887 2006-03-01 18:21:01Z rcatwood $";

  return (rcsid);
}

/* end of rcs_id_fidap_interp_calc_c subroutine */

/*RCS Id:$Id: wfact_z_calc.c 887 2006-03-01 18:21:01Z rcatwood $*/
/*
RCS Log:$Log$
RCS Log:Revision 11.1  2006/03/01 18:20:40  rcatwood
RCS Log:Merging polycomponent and gas with meltback
RCS Log:
RCS Log:Revision 10.3  2005/12/01 14:38:02  rcatwood
RCS Log:Merged xly_05 changes into the main trunk
RCS Log:Primarily involving melt-back
RCS Log:
RCS Log:Revision 10.1.2.2  2005/11/23 18:18:53  rcatwood
RCS Log:Result of merging mould_source and xly meltback+curvature 2d versions
RCS Log:
RCS Log:Revision 10.1  2005/11/03 11:56:48  rcatwood
RCS Log:New version number -- using mould_src as base
RCS Log:
RCS Log:Revision 8.1.12.2  2005/11/02 11:55:06  rcatwood
RCS Log:Fixing up the revision nubmer after loss of repository
RCS Log:
RCS Log:Revision 9.1  2003/08/14 14:38:40  rcatwood
RCS Log:Working merge with decentered/porosity/procast, also including
RCS Log:Ali Chirazi's multicomponent (not tested in this version)
RCS Log:
RCS Log:Revision 8.1.6.1  2003/01/22 16:53:47  rcatwood
RCS Log:Almost working read_fg version
RCS Log:
RCS Log:Revision 8.1  2002/10/17 17:01:04  rcatwood
RCS Log:New version number! for decentered/porosity merge! Alpha Version!
RCS Log:
RCS Log:Revision 7.0  2000/11/07 15:53:28  rcatwood
RCS Log:Multi Cell Pores added
RCS Log:
RCS Log:Revision 6.0  2000/09/25 18:03:36  rcatwood
RCS Log:After PORE_00 and NLM
RCS Log:
RCS Log:Revision 2.0  2000/08/02 10:21:56  rcatwood
RCS Log:Version used for pore paper runs
RCS Log:
RCS Log:Revision 1.1  2000/05/22 12:29:24  rcatwood
RCS Log:Fixed fs finish. Casolid to C from  W file. Global option
RCS Log:
RCS Log:Revision 5.1  2000/03/02 16:10:16  rcatwood
RCS Log:Merged xxu and rca versions
RCS Log:
RCS Log:Revision 5.0.2.2  2000/03/02 13:09:59  rcatwood
RCS Log:Fixed copy-mat bug.
RCS Log:
RCS Log:Revision 5.0.2.1  2000/03/01 16:06:53  rcatwood
RCS Log:merge VAR and Multiblock
RCS Log:
RCS Log:Revision 5.0  2000/03/01 16:00:54  rcatwood
RCS Log:Merge robert and xxu versions
RCS Log:
RCS Log:Revision 1.13  2000/02/28 15:10:35  xxu
RCS Log:add local pointer for efficienct interpolation.
RCS Log:
RCS Log:Revision 1.12  2000/02/28 12:16:44  xxu
RCS Log:set local pointers for more efficient calculation
RCS Log:
RCS Log:Revision 1.11  2000/02/25 21:02:37  rcatwood
RCS Log:Fixed greater than
RCS Log:
RCS Log:Revision 1.10  2000/02/25 11:44:10  xxu
RCS Log:set Tmax on the top of the pool
RCS Log:
RCS Log:Revision 1.9  2000/02/25 11:12:00  xxu
RCS Log:h_ingot replace the hs_ingot for setting Tmin
RCS Log:
RCS Log:Revision 1.8  2000/02/24 18:56:30  xxu
RCS Log:sort out problem caused by Z_cell > Z_node_max
RCS Log:
RCS Log:Revision 1.7  2000/02/24 17:52:50  xxu
RCS Log:change on steady state
RCS Log:
RCS Log:Revision 1.6  2000/02/23 20:21:31  rcatwood
RCS Log:Improved interpolation version
RCS Log:
RCS Log:Revision 1.5  2000/02/23 18:43:33  xxu
RCS Log:added prototyeps
RCS Log:
RCS Log:Revision 1.4  2000/02/23 16:37:12  rcatwood
RCS Log:Improved interpolation version
RCS Log:
RCS Log:Revision 1.3  2000/02/23 15:54:07  xxu
RCS Log:weighting fact of z direction
RCS Log:
RCS Log:Revision 1.2  2000/02/22 16:11:46  xxu
RCS Log:incremental improvement
RCS Log:
RCS Log:Revision 1.1  2000/02/22 14:20:26  xxu
RCS Log:Initial revision
RCS Log:
*/
